import numpy as np
from typing import List, Tuple, Dict, Sequence

def bbox_iou(box: np.ndarray, boxes: np.ndarray) -> np.ndarray:

    tl = np.maximum(box[:2], boxes[:, :2])            # top-left
    br = np.minimum(box[2:], boxes[:, 2:])            # bottom-right
    inter_wh = np.clip(br - tl, a_min=0, a_max=None)
    inter = inter_wh[:, 0] * inter_wh[:, 1]

    area1 = (box[2] - box[0]) * (box[3] - box[1])
    area2 = (boxes[:, 2] - boxes[:, 0]) * (boxes[:, 3] - boxes[:, 1])

    union = area1 + area2 - inter + 1e-9
    return inter / union


def near_image_edge(box: Sequence[float],
                    img_wh: Tuple[int, int],
                    margin_ratio: float = 0.05) -> bool:
    w, h = img_wh
    x1, y1, x2, y2 = box
    margin_x = w * margin_ratio
    margin_y = h * margin_ratio
    return (
        x1 <= margin_x or
        # y1 <= margin_y or
        (w - x2) <= margin_x or
        (h - y2) <= margin_y
    )


from .glm import glm_reasoning, glm_reasoning_2

def disappeared_suddenly(
    track_boxes: List[Tuple[int, Sequence[float]]],
    other_boxes_by_frame: Dict[int, List[Sequence[float]]],
    gen_video_path,
    glm_dir,
    img_size: Tuple[int, int],
    *,
    edge_margin: float = 0.05,
    iou_threshold: float = 0.2,
    min_track_len: int = 5
) -> bool:
    if len(track_boxes) < min_track_len:
        print(f'len track_bbox is {len(track_boxes)}: too short, skip')
        return False

    last_fid, last_box = track_boxes[-1]
    last_box = np.asarray(last_box, dtype=float)

    if last_fid == 100:
        print(f'no missing: 100 frames')
        return False

    if near_image_edge(last_box, img_size, edge_margin):
        print(f'no missing: in corner')
        return False

    others = other_boxes_by_frame.get(last_fid, [])
    if len(others):
        ious = bbox_iou(last_box, np.asarray(others, dtype=float))
        if (ious >= iou_threshold).any():
            print(f'no missing: occ with objects from the first frame')
            return False

    # 4) normal ones
    first_fid, first_box = track_boxes[0]
    first_fid_, first_box_ = track_boxes[1]
    first_fids = [first_fid, first_fid_]
    first_boxs = [first_box, first_box_]

    last_fids = []
    last_boxs = []
    idx_last = max(0, len(track_boxes)-5)
    for id in [idx_last, -1]:
        last_fid, last_box = track_boxes[id]
        last_fids.append(last_fid)
        last_boxs.append(last_box)

    next_fids = [last_fids[-1]+1]
    if last_fids[-1]+2 <=100:
        next_fids.append(last_fids[-1]+2)
    else:
        next_fids.append(last_fids[-1]+1)

    is_normal_missing = glm_reasoning(
        gen_video_path,
        first_fids, first_boxs,
        last_fids, last_boxs,
        next_fids,
        glm_dir)
    if is_normal_missing:
        return False
    
    return True


def get_missing_per_scene(missings):
    scene_score = []
    count = 0
    for id, missing in missings:
        if missing:
            # score = 1.0 - ( (id+1) / 101)
            score = 1
            count += 1
            scene_score.append(score)
    if len(scene_score) > 0:
        # return np.mean(scene_score) * count / len(missings)
        return np.mean(scene_score)
    else:
        return 0.0
    

def get_missing_per_agent(missings):
    scene_score = []
    count = 0
    for id, missing in missings:
        if missing:
            # score = 1.0 - ( (id+1) / 101)
            score = 1
            count += 1
    scene_score = count
    return scene_score
    if len(scene_score) > 0:
        # return np.mean(scene_score) * count / len(missings)
        return np.mean(scene_score)
    else:
        return 0.0

