#!/usr/bin/env python3
"""Interactive script to optimize ground truth annotations based on user instructions."""

import json
import sys
import tempfile
import subprocess
import textwrap
from pathlib import Path
from typing import Optional

# Add project root to path
root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.llms import get_llm
from src.utils.settings import settings
from src.utils.log import logger


def create_optimization_prompt(current_annotation: str, user_instructions: str, video_path: str) -> str:
    """Create a prompt for optimizing ground truth annotation based on user instructions.
    
    Args:
        current_annotation (str): Current ground truth annotation.
        user_instructions (str): User's revision instructions.
        video_path (str): Path to the video file.
        
    Returns:
        str: Prompt for annotation optimization.
    """
    return f"""You are an expert traffic safety analyst helping to refine ground truth annotations for dashcam video evaluation.

Video: {Path(video_path).name}

Current Ground Truth Annotation:
{current_annotation}

User Revision Instructions:
{user_instructions}

Your task is to revise the annotation according to the user's specific instructions while maintaining:
1. **Factual accuracy**: Only modify what the user specifically requests
2. **Narrative flow**: Keep the chronological, story-like format
3. **Technical precision**: Use accurate traffic safety terminology  
4. **Objective tone**: Maintain professional, factual language
5. **Completeness**: Ensure no important safety details are lost

Instructions for revision:
- Follow the user's instructions precisely
- If instructions are unclear, make the most reasonable interpretation
- Preserve all information not explicitly mentioned for change
- Maintain the same narrative style and structure
- Keep the annotation focused on observable facts

Provide the revised annotation:"""


def optimize_annotation(current_annotation: str, user_instructions: str, video_path: str, model_id: Optional[str] = None) -> str:
    """Optimize annotation using LLM based on user instructions.
    
    Args:
        current_annotation (str): Current ground truth annotation.
        user_instructions (str): User's revision instructions.
        video_path (str): Path to the video file.
        model_id (str, optional): LLM model to use.
        
    Returns:
        str: Optimized annotation.
    """
    # Get LLM
    if model_id:
        llm = get_llm(model_id)
    else:
        llm = get_llm(settings.app.llm['main'])
    
    # Generate prompt
    prompt = create_optimization_prompt(current_annotation, user_instructions, video_path)
    
    # Generate optimized annotation
    logger.debug("Optimizing annotation using LLM...")
    response = llm.invoke(prompt)
    
    return response.content.strip()


def format_annotation_for_editor(annotation: str, line_width: int = 75) -> str:
    """Format annotation text for display in editor with proper line wrapping.
    
    Args:
        annotation (str): The annotation text to format.
        line_width (int): Maximum width for each line.
        
    Returns:
        str: Formatted annotation with comment prefixes and line wrapping.
    """
    if not annotation:
        return ""
    
    # Split annotation into sentences to avoid breaking mid-sentence
    sentences = annotation.replace('. ', '.\n').split('\n')
    formatted_lines = []
    
    for sentence in sentences:
        sentence = sentence.strip()
        if not sentence:
            continue
            
        # Wrap long sentences
        wrapped_lines = textwrap.wrap(sentence, width=line_width)
        
        for line in wrapped_lines:
            formatted_lines.append(f"# {line}")
    
    return '\n'.join(formatted_lines)


def find_ground_truth_file(file_number: str, ground_truth_dir: Path) -> Optional[Path]:
    """Find ground truth file by number prefix.
    
    Args:
        file_number (str): File number (e.g., "000", "001").
        ground_truth_dir (Path): Directory containing ground truth files.
        
    Returns:
        Optional[Path]: Path to the found file, or None if not found.
    """
    # Try different patterns
    patterns = [
        f"{file_number}_*.json",
        f"{file_number}.json", 
        f"*{file_number}*.json"
    ]
    
    for pattern in patterns:
        matches = list(ground_truth_dir.glob(pattern))
        if matches:
            return matches[0]  # Return first match
    
    return None


def get_multiline_input_with_editing(current_annotation: str, video_path: str) -> str:
    """Get multi-line user input with editing capabilities using system editor.
    
    Args:
        current_annotation (str): Current annotation to display for reference.
        video_path (str): Path to the video file.
    
    Returns:
        str: User's revision instructions.
    """
    # Format annotation with proper line wrapping
    formatted_annotation = format_annotation_for_editor(current_annotation)
    
    # Create template content with current annotation for reference
    template_content = f"""# REVISION INSTRUCTIONS FOR: {Path(video_path).name}
# 
# CURRENT ANNOTATION (for reference):
{formatted_annotation}
#
# ========================================
# Enter your revision instructions below:
# Lines starting with # are comments and will be ignored
# Write your specific instructions here:

"""
    
    print("💡 Opening editor for revision instructions...")
    print("   - Write your multi-line instructions in the editor")
    print("   - Lines starting with # are comments")
    print("   - Save and close the editor when done")
    print("   - Use Ctrl+C to cancel")
    print("-" * 40)
    
    # Determine editor
    editor = get_system_editor()
    
    try:
        # Create temporary file with template
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as tmp_file:
            tmp_file.write(template_content)
            tmp_file_path = tmp_file.name
        
        # Open editor
        try:
            subprocess.run([editor, tmp_file_path], check=True)
        except subprocess.CalledProcessError:
            print(f"❌ Failed to open editor: {editor}")
            return get_simple_multiline_input()
        except FileNotFoundError:
            print(f"❌ Editor not found: {editor}")
            return get_simple_multiline_input()
        
        # Read the edited content
        with open(tmp_file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        # Clean up temporary file
        Path(tmp_file_path).unlink(missing_ok=True)
        
        # Process content (remove comments and empty lines)
        lines = []
        for line in content.split('\n'):
            line = line.strip()
            if line and not line.startswith('#'):
                lines.append(line)
        
        result = '\n'.join(lines).strip()
        
        if result:
            print(f"\n📝 Your instructions ({len(lines)} lines):")
            print("-" * 40)
            print(result)
            print("-" * 40)
        
        return result
        
    except KeyboardInterrupt:
        print("\n❌ Cancelled by user.")
        return ""
    except Exception as e:
        print(f"❌ Error with editor: {e}")
        return get_simple_multiline_input()


def get_system_editor() -> str:
    """Get the system's default text editor.
    
    Returns:
        str: Editor command.
    """
    import os
    
    # Try environment variables first
    for var in ['EDITOR', 'VISUAL']:
        editor = os.getenv(var)
        if editor:
            return editor
    
    # Try common editors
    common_editors = ['nano', 'vim', 'vi', 'code', 'notepad']
    
    for editor in common_editors:
        try:
            subprocess.run(['which', editor], capture_output=True, check=True)
            return editor
        except (subprocess.CalledProcessError, FileNotFoundError):
            continue
    
    # Fallback
    return 'nano'


def get_simple_multiline_input() -> str:
    """Fallback simple multi-line input method.
    
    Returns:
        str: User's revision instructions.
    """
    print("\n💡 Enter your revision instructions (press Enter twice when done):")
    print("   Example: 'Change the speed from steady to slow'")
    print("   Example: 'Add more details about the lane change'")
    print("   Example: 'Fix the vehicle color from silver to white'")
    print("-" * 40)
    
    instructions = []
    empty_line_count = 0
    
    while True:
        try:
            line = input()
            if line.strip() == "":
                empty_line_count += 1
                if empty_line_count >= 2 and instructions:
                    break
            else:
                empty_line_count = 0
                instructions.append(line)
        except KeyboardInterrupt:
            print("\n❌ Cancelled by user.")
            return ""
    
    return '\n'.join(instructions)


def display_current_annotation(gt_data: dict) -> None:
    """Display current annotation for review.
    
    Args:
        gt_data (dict): Ground truth data.
    """
    annotation = gt_data.get('ground_truth', {}).get('annotation', '')
    video_path = gt_data.get('video_path', 'Unknown')
    
    print("=" * 80)
    print(f"📹 Video: {Path(video_path).name}")
    print("=" * 80)
    print("📝 Current Annotation:")
    print("-" * 40)
    print(annotation)
    print("=" * 80)


def interactive_optimization(ground_truth_file: Path, model_id: Optional[str] = None) -> None:
    """Interactive annotation optimization session.
    
    Args:
        ground_truth_file (Path): Path to the ground truth file.
        model_id (str, optional): LLM model to use.
    """
    try:
        # Load ground truth file
        with open(ground_truth_file, 'r', encoding='utf-8') as f:
            gt_data = json.load(f)
        
        current_annotation = gt_data.get('ground_truth', {}).get('annotation', '')
        video_path = gt_data.get('video_path', '')
        
        if not current_annotation or current_annotation == "MANUAL_ANNOTATION_REQUIRED":
            print("❌ No annotation found or annotation is still placeholder.")
            print("Please run 2_1_generate_ground_truth_annotations.py first.")
            return
        
        # Display current annotation
        display_current_annotation(gt_data)
        
        # Get user instructions with editing capability
        user_instructions = get_multiline_input_with_editing(current_annotation, video_path)
        
        if not user_instructions.strip():
            print("❌ No instructions provided. Exiting.")
            return
        
        print(f"\n🔄 Processing instructions with model: {model_id or settings.app.llm['main']}")
        print("⏳ Optimizing annotation...")
        
        # Optimize annotation
        try:
            optimized_annotation = optimize_annotation(
                current_annotation, user_instructions, video_path, model_id
            )
            
            if not optimized_annotation:
                print("❌ Failed to generate optimized annotation.")
                return
            
            # Display optimized annotation
            print("\n" + "=" * 80)
            print("✨ OPTIMIZED ANNOTATION:")
            print("-" * 40)
            print(optimized_annotation)
            print("=" * 80)
            
            # Ask for confirmation
            while True:
                choice = input("\n📋 Options:\n  [s] Save optimized annotation\n  [r] Retry with new instructions\n  [c] Cancel (keep original)\n\nYour choice (s/r/c): ").strip().lower()
                
                if choice == 's':
                    # Save optimized annotation
                    gt_data['ground_truth']['annotation'] = optimized_annotation
                    
                    with open(ground_truth_file, 'w', encoding='utf-8') as f:
                        json.dump(gt_data, f, indent=2, ensure_ascii=False)
                    
                    print(f"✅ Annotation saved to {ground_truth_file.name}")
                    return
                
                elif choice == 'r':
                    # Retry with new instructions
                    interactive_optimization(ground_truth_file, model_id)
                    return
                
                elif choice == 'c':
                    print("❌ Cancelled. Original annotation preserved.")
                    return
                
                else:
                    print("❌ Invalid choice. Please enter 's', 'r', or 'c'.")
        
        except Exception as e:
            logger.error(f"Failed to optimize annotation: {e}")
            print(f"❌ Optimization failed: {e}")
            
    except Exception as e:
        logger.error(f"Failed to process {ground_truth_file}: {e}")
        print(f"❌ Failed to process file: {e}")


def main():
    """Main function for interactive annotation optimization."""
    
    print("=" * 60)
    print("OPTIMIZE GROUND TRUTH ANNOTATIONS INTERACTIVELY")
    print("=" * 60)
    
    # Configuration
    ground_truth_dir = root / "data" / "evaluation" / "ground_truth"
    
    if not ground_truth_dir.exists():
        print(f"❌ Ground truth directory not found: {ground_truth_dir}")
        return
    
    # Parse command line arguments
    model_id = None
    file_number = None
    
    if len(sys.argv) < 2:
        print("❌ Usage: python 2_2_optimize_ground_truth_annotation.py <file_number> [--model=model_id]")
        print("   Example: python 2_2_optimize_ground_truth_annotation.py 000")
        print("   Example: python 2_2_optimize_ground_truth_annotation.py 001 --model=\"openai:gpt-4o\"")
        return
    
    file_number = sys.argv[1]
    
    if len(sys.argv) > 2 and sys.argv[2].startswith('--model='):
        model_id = sys.argv[2].split('=')[1]
    
    # Find ground truth file
    ground_truth_file = find_ground_truth_file(file_number, ground_truth_dir)
    
    if not ground_truth_file:
        print(f"❌ No ground truth file found with number: {file_number}")
        print(f"   Searched in: {ground_truth_dir}")
        
        # Show available files
        json_files = list(ground_truth_dir.glob("*.json"))
        if json_files:
            print("\n📁 Available files:")
            for f in json_files:
                print(f"   {f.name}")
        return
    
    print(f"📁 Found file: {ground_truth_file.name}")
    if model_id:
        print(f"🤖 Using model: {model_id}")
    print()
    
    # Start interactive optimization
    interactive_optimization(ground_truth_file, model_id)
    
    print("\n" + "=" * 60)
    print("OPTIMIZATION SESSION COMPLETE")
    print("=" * 60)


if __name__ == "__main__":
    main()