"""Configuration and model loading for component evaluation."""

import sys
from pathlib import Path
from typing import Dict, List

# Add project root to path
root = Path(__file__).parent.parent.parent
sys.path.append(str(root))


class ModelConfigLoader:
    """Dynamic model configuration loader."""
    
    @staticmethod
    def load_models_from_file(file_path: str) -> List[str]:
        """Load and validate models from text file.
        
        Args:
            file_path: Path to model configuration file
            
        Returns:
            List of model identifiers
            
        Raises:
            FileNotFoundError: If model file doesn't exist
            ValueError: If no valid models found
        """
        file_path = Path(file_path)
        if not file_path.exists():
            raise FileNotFoundError(f"Model configuration file not found: {file_path}")
            
        models = []
        with open(file_path, 'r') as f:
            for line_num, line in enumerate(f, 1):
                model = line.strip()
                if model and not model.startswith('#'):  # Skip empty lines and comments
                    models.append(model)
        
        if not models:
            raise ValueError(f"No models found in {file_path}")
            
        return models
    
    @staticmethod
    def get_component_models(component: str) -> List[str]:
        """Get model list for specific component.
        
        Args:
            component: Component name ('annotation', 'scene', 'violation', 'accident', 'assessment')
            
        Returns:
            List of model identifiers for the component
        """
        if component == 'annotation':
            file_path = root / 'evaluation' / 'models' / 'annotation.txt'
        else:
            file_path = root / 'evaluation' / 'models' / 'text.txt'
            
        models = ModelConfigLoader.load_models_from_file(file_path)
        print(f"📋 Loaded {len(models)} models for {component} from {file_path.name}")
        return models
    
    @staticmethod
    def get_all_models() -> Dict[str, List[str]]:
        """Get all models for all components.
        
        Returns:
            Dictionary mapping component names to model lists
        """
        return {
            'annotation': ModelConfigLoader.get_component_models('annotation'),
            'scene': ModelConfigLoader.get_component_models('scene'),
            'violation': ModelConfigLoader.get_component_models('violation'),
            'accident': ModelConfigLoader.get_component_models('accident'),
            'assessment': ModelConfigLoader.get_component_models('assessment')
        }
    
    @staticmethod
    def validate_model_files():
        """Validate model configuration files exist and contain models.
        
        Returns:
            Tuple of (annotation_models, text_models)
        """
        annotation_file = root / 'evaluation' / 'models' / 'annotation.txt'
        text_file = root / 'evaluation' / 'models' / 'text.txt'
        
        if not annotation_file.exists():
            raise FileNotFoundError(f"Annotation models file not found: {annotation_file}")
        if not text_file.exists():
            raise FileNotFoundError(f"Text models file not found: {text_file}")
            
        annotation_models = ModelConfigLoader.load_models_from_file(annotation_file)
        text_models = ModelConfigLoader.load_models_from_file(text_file)
        
        print(f"✅ Found {len(annotation_models)} annotation models")
        print(f"✅ Found {len(text_models)} text models")
        
        # Validate model format
        all_models = annotation_models + text_models
        for model in all_models:
            if ':' not in model:
                print(f"⚠️  Warning: Model '{model}' may not have correct platform:model format")
        
        return annotation_models, text_models


class EvaluationConfig:
    """Configuration settings for component evaluation."""
    
    # Data paths
    DATA_DIR = root / "data" / "evaluation"
    COMPONENT_EVAL_DIR = DATA_DIR / "component_eval"
    GROUND_TRUTH_DIR = DATA_DIR / "ground_truth"
    SYSTEM_OUTPUTS_DIR = DATA_DIR / "system_outputs"
    
    # Component evaluation paths
    CACHE_DIR = COMPONENT_EVAL_DIR / "cache"
    RESULTS_DIR = COMPONENT_EVAL_DIR / "results"
    REPORTS_DIR = COMPONENT_EVAL_DIR / "reports"
    EXPORTS_DIR = COMPONENT_EVAL_DIR / "exports"
    
    # Hierarchical cache directories (mirrors system_outputs structure)
    TRADITIONAL_CACHE_DIR = CACHE_DIR / "traditional"
    LLM_JUDGE_CACHE_DIR = CACHE_DIR / "llm_judge"
    
    # Evaluation settings
    COMPONENTS = ['annotation', 'scene', 'violation', 'accident', 'assessment']
    
    # LLM-as-judge settings
    LLM_JUDGE_MODEL = "openai:gpt-4.1"
    
    # State management
    STATE_FILE = CACHE_DIR / "evaluation_state.json"
    
    @classmethod
    def ensure_directories(cls):
        """Ensure all necessary directories exist."""
        for dir_path in [cls.CACHE_DIR, cls.RESULTS_DIR, cls.REPORTS_DIR, cls.EXPORTS_DIR]:
            dir_path.mkdir(parents=True, exist_ok=True)
        
        # Ensure component result directories exist
        for component in cls.COMPONENTS:
            component_dir = cls.RESULTS_DIR / component
            component_dir.mkdir(exist_ok=True)