"""FastAPI backend for video review and clip marking tool."""

import json
import logging
import re
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime
import uvicorn

from fastapi import FastAPI, HTTPException, Request
from fastapi.staticfiles import StaticFiles
from fastapi.responses import HTMLResponse, FileResponse, StreamingResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel

from evaluation.make_dataset.s2_video_reviewer.models import (
    ClipMarking, VideoReviewStatus, VideoQueue, ReviewSession, ClipExport,
    ClipStatistics, CategoryStats
)
from evaluation.make_dataset.s2_video_reviewer.video_manager import VideoManager

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Initialize FastAPI app
app = FastAPI(
    title="DriveGuard Video Review Tool",
    description="Tool for manually reviewing and marking clips from dashcam videos",
    version="1.0.0"
)

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Initialize video manager
video_manager = VideoManager()

# Global state for current session
current_session: Optional[ReviewSession] = None
current_queue: Optional[VideoQueue] = None
clip_markings: Dict[str, List[ClipMarking]] = {}  # video_id -> list of clips

# File paths
REVIEW_PROGRESS_DIR = video_manager.review_progress_dir
REVIEW_STATE_FILE = REVIEW_PROGRESS_DIR / "review_state.json"
VIDEO_QUEUE_FILE = REVIEW_PROGRESS_DIR / "video_queue.json"

# Serve static files
static_dir = Path(__file__).parent / "static"
app.mount("/static", StaticFiles(directory=str(static_dir)), name="static")


# === Data Persistence Functions ===

def load_individual_video_markings() -> Dict[str, List[ClipMarking]]:
    """Load clip markings from individual video files.
    
    Returns:
        Dictionary mapping video_id to list of ClipMarking objects
    """
    markings = {}
    
    # Find all individual marking files
    marking_files = list(REVIEW_PROGRESS_DIR.glob("*_review_markings.json"))
    
    for marking_file in marking_files:
        try:
            with open(marking_file, 'r', encoding='utf-8') as f:
                data = json.load(f)
            
            video_info = data.get('video_info', {})
            video_id = video_info.get('video_id')
            
            if not video_id:
                logger.warning(f"No video_id found in {marking_file.name}, skipping")
                continue
            
            # Convert clip markings to ClipMarking objects
            clips_data = data.get('clip_markings', [])
            clips = [ClipMarking(**clip_data) for clip_data in clips_data]
            markings[video_id] = clips
            
            logger.debug(f"Loaded {len(clips)} clips for video {video_id} from {marking_file.name}")
            
        except Exception as e:
            logger.warning(f"Failed to load markings from {marking_file.name}: {e}")
            continue
    
    return markings


def load_clip_markings():
    """Load clip markings from individual video files."""
    global clip_markings
    
    # Load from individual video files only
    clip_markings = load_individual_video_markings()
    
    if clip_markings:
        total_clips = sum(len(clips) for clips in clip_markings.values())
        logger.info(f"Loaded {total_clips} clips from {len(clip_markings)} individual video files")
    else:
        clip_markings = {}
        logger.info("No clip markings found")


def save_individual_video_markings(video_id: str, clips: List[ClipMarking]):
    """Save clip markings for a single video to its own JSON file.
    
    Args:
        video_id: The video ID to save markings for
        clips: List of clip markings for this video
    """
    try:
        # Clean video_id for filename
        safe_video_id = re.sub(r'[^a-zA-Z0-9_-]', '_', video_id)
        markings_filename = f"{safe_video_id}_review_markings.json"
        markings_path = REVIEW_PROGRESS_DIR / markings_filename
        
        # Get video metadata if available
        video = None
        if current_queue:
            video = next((v for v in current_queue.videos if v.video_id == video_id), None)
        
        # Create comprehensive markings structure
        video_markings = {
            'markings_info': {
                'created_with': 'DriveGuard s2_video_reviewer',
                'markings_version': '1.0',
                'created_at': datetime.now().isoformat(),
                'last_updated': datetime.now().isoformat()
            },
            'video_info': {
                'video_id': video_id,
                'video_path': video.video_path if video else '',
                'video_category': video.video_category if video else 'unknown',
                'video_title': video.video_title if video else '',
                'video_duration': video.video_duration if video else 0.0,
                'total_clips_marked': len(clips)
            },
            'clip_markings': [clip.model_dump() for clip in clips]
        }
        
        # Save markings file
        with open(markings_path, 'w', encoding='utf-8') as f:
            json.dump(video_markings, f, indent=2, default=str)
        
        logger.debug(f"Saved markings for video {video_id}: {len(clips)} clips")
        return True
        
    except Exception as e:
        logger.error(f"Failed to save markings for video {video_id}: {e}")
        return False


def save_clip_markings():
    """Save clip markings to individual video files."""
    try:
        # Save individual video marking files
        success_count = 0
        for video_id, clips in clip_markings.items():
            if save_individual_video_markings(video_id, clips):
                success_count += 1
        
        total_clips = sum(len(clips) for clips in clip_markings.values())
        logger.info(f"Saved {total_clips} clips to {success_count} individual video files")
        
    except Exception as e:
        logger.error(f"Failed to save clip markings: {e}")


def calculate_clip_statistics() -> ClipStatistics:
    """Calculate comprehensive statistics from all clip markings."""
    stats = ClipStatistics()
    
    if not clip_markings:
        return stats
    
    # Collect all clips and calculate basic stats
    all_clips = []
    for video_id, clips in clip_markings.items():
        all_clips.extend(clips)
    
    if not all_clips:
        return stats
    
    # Basic totals
    stats.total_clips = len(all_clips)
    stats.total_videos_with_clips = len(clip_markings)
    stats.total_clip_duration = sum(clip.duration for clip in all_clips)
    stats.average_clip_duration = stats.total_clip_duration / len(all_clips)
    
    # Count clips by video
    for video_id, clips in clip_markings.items():
        stats.clips_by_video[video_id] = len(clips)
    
    # Category analysis
    category_data = {}
    
    for clip in all_clips:
        # Category stats
        category = clip.video_category
        if category not in stats.clips_by_category:
            stats.clips_by_category[category] = 0
            category_data[category] = {
                'clips': [],
                'video_ids': set()
            }
        
        stats.clips_by_category[category] += 1
        category_data[category]['clips'].append(clip)
        category_data[category]['video_ids'].add(clip.video_id)
        
        # Event type stats
        event_type = clip.event_type or "unmarked"
        if event_type not in stats.clips_by_event_type:
            stats.clips_by_event_type[event_type] = 0
        stats.clips_by_event_type[event_type] += 1
        
        # Severity stats
        severity = clip.severity_level
        if severity not in stats.clips_by_severity:
            stats.clips_by_severity[severity] = 0
        stats.clips_by_severity[severity] += 1
    
    # Generate detailed category stats
    for category, data in category_data.items():
        clips = data['clips']
        category_stats = CategoryStats()
        
        category_stats.clip_count = len(clips)
        category_stats.video_count = len(data['video_ids'])
        category_stats.total_duration = sum(clip.duration for clip in clips)
        category_stats.average_duration = category_stats.total_duration / len(clips)
        
        # Severity breakdown for this category
        for clip in clips:
            severity = clip.severity_level
            if severity not in category_stats.severity_breakdown:
                category_stats.severity_breakdown[severity] = 0
            category_stats.severity_breakdown[severity] += 1
        
        # Event type breakdown for this category
        for clip in clips:
            event_type = clip.event_type or "unmarked"
            if event_type not in category_stats.event_type_breakdown:
                category_stats.event_type_breakdown[event_type] = 0
            category_stats.event_type_breakdown[event_type] += 1
        
        stats.category_details[category] = category_stats
    
    return stats


def load_session_state():
    """Load review session state."""
    global current_session, current_queue
    
    if REVIEW_STATE_FILE.exists():
        try:
            with open(REVIEW_STATE_FILE, 'r') as f:
                session_data = json.load(f)
            current_session = ReviewSession(**session_data)
            logger.info(f"Loaded session state: video {current_session.current_video_index}")
        except Exception as e:
            logger.error(f"Failed to load session state: {e}")
            current_session = None
    
    if VIDEO_QUEUE_FILE.exists():
        try:
            with open(VIDEO_QUEUE_FILE, 'r') as f:
                queue_data = json.load(f)
            
            # Convert video data back to VideoReviewStatus objects
            videos = [VideoReviewStatus(**video_data) for video_data in queue_data['videos']]
            
            # Fix review status for videos that have clips but show as not_reviewed
            for video in videos:
                if video.review_status == "not_reviewed" and video.clips_marked > 0:
                    video.review_status = "has_clips"
                    logger.info(f"Updated review status for video {video.video_id}: not_reviewed -> has_clips")
            
            current_queue = VideoQueue(
                queue_id=queue_data.get('queue_id', ''),
                current_index=queue_data.get('current_index', 0),
                videos=videos,
                first_unreviewed_index=0  # Will be calculated below
            )
            
            # Calculate first_unreviewed_index from existing queue
            for i, video in enumerate(videos):
                if video.review_status == "not_reviewed":
                    current_queue.first_unreviewed_index = i
                    break
            else:
                # No unreviewed videos found
                current_queue.first_unreviewed_index = len(videos)
            logger.info(f"Loaded video queue with {len(videos)} videos")
        except Exception as e:
            logger.error(f"Failed to load video queue: {e}")
            current_queue = None


def save_session_state():
    """Save review session state."""
    if current_session:
        try:
            with open(REVIEW_STATE_FILE, 'w') as f:
                json.dump(current_session.model_dump(), f, indent=2, default=str)
        except Exception as e:
            logger.error(f"Failed to save session state: {e}")
    
    if current_queue:
        try:
            # Create simplified video objects for queue persistence (no clip arrays)
            simplified_videos = []
            for video in current_queue.videos:
                simplified_video = {
                    'video_id': video.video_id,
                    'video_path': video.video_path,
                    'video_category': video.video_category,
                    'video_title': video.video_title,
                    'video_duration': video.video_duration,
                    'review_status': video.review_status,
                    'skip_reason': video.skip_reason,
                    'review_completed': video.review_completed.isoformat() if video.review_completed else None,
                    'relevance_score': video.relevance_score,
                    'clips_marked': video.clips_marked,
                    'total_duration_marked': video.total_duration_marked
                }
                simplified_videos.append(simplified_video)
            
            data = {
                'queue_id': current_queue.queue_id,
                'current_index': current_queue.current_index,
                'last_updated': current_queue.last_updated.isoformat(),
                'videos': simplified_videos
            }
            with open(VIDEO_QUEUE_FILE, 'w') as f:
                json.dump(data, f, indent=2, default=str)
        except Exception as e:
            logger.error(f"Failed to save video queue: {e}")


# === API Endpoints ===

@app.on_event("startup")
async def startup_event():
    """Initialize the application."""
    logger.info("Starting DriveGuard Video Review Tool")
    
    # Load existing data
    load_clip_markings()
    load_session_state()
    
    # Initialize session and queue if needed
    global current_session, current_queue
    if current_session is None:
        current_session = ReviewSession()
    
    # Check if existing queue has proper ordering (reviewed videos first, then unreviewed)
    queue_needs_rebuild = False
    if current_queue is not None:
        # Check for mixed ordering (reviewed videos appearing after unreviewed videos)
        found_unreviewed = False
        for video in current_queue.videos:
            if video.review_status == "not_reviewed":
                found_unreviewed = True
            elif found_unreviewed and video.review_status != "not_reviewed":
                # Found reviewed video after unreviewed video - queue has mixed ordering
                queue_needs_rebuild = True
                logger.info("Detected queue with mixed ordering - rebuilding queue for proper structure")
                break
    
    if current_queue is None or queue_needs_rebuild:
        # Build shuffled queue with smart filtering (include completed videos to access marked clips)
        current_queue = video_manager.build_shuffled_video_queue(include_completed=True)
        current_session.total_videos = len(current_queue.videos)
        
        # Auto-resume: set to first unreviewed video (after completed ones)
        if len(current_queue.videos) > 0:
            # Position at first unreviewed video, or start of queue if none
            first_unreviewed_index = getattr(current_queue, 'first_unreviewed_index', 0)
            current_queue.current_index = first_unreviewed_index
            current_session.current_video_index = first_unreviewed_index
            
            if first_unreviewed_index > 0:
                logger.info(f"Starting at first unreviewed video (index {first_unreviewed_index}) "
                           f"with {first_unreviewed_index} completed videos before it")
            else:
                logger.info(f"Starting from beginning (no completed videos found)")
        else:
            logger.info("No videos found in queue")
    else:
        # Existing session with proper ordering - position at first unreviewed video
        logger.info("Found existing queue with proper ordering - positioning at first unreviewed video")
        
        # Calculate first unreviewed index from existing queue
        first_unreviewed_index = 0
        for i, video in enumerate(current_queue.videos):
            if video.review_status == "not_reviewed":
                first_unreviewed_index = i
                break
        else:
            # No unreviewed videos found
            first_unreviewed_index = len(current_queue.videos)
        
        # Always position at first unreviewed video on restart
        current_queue.current_index = first_unreviewed_index
        current_session.current_video_index = first_unreviewed_index
        current_session.total_videos = len(current_queue.videos)
        
        if first_unreviewed_index < len(current_queue.videos):
            logger.info(f"Positioned at first unreviewed video (index {first_unreviewed_index}) "
                       f"with {first_unreviewed_index} reviewed videos before it")
        else:
            logger.info("All videos have been reviewed!")
        
        # Update first_unreviewed_index attribute for consistency
        if hasattr(current_queue, 'first_unreviewed_index'):
            current_queue.first_unreviewed_index = first_unreviewed_index
    
    # Save the updated session state
    save_session_state()
    
    queue_type = "shuffled" if getattr(current_queue, 'shuffle_categories', False) else "standard"
    total_videos_info = f"Loaded {queue_type} queue with {len(current_queue.videos)} videos"
    logger.info(total_videos_info)


@app.get("/", response_class=HTMLResponse)
async def root():
    """Serve the main application page."""
    static_file = Path(__file__).parent / "static" / "index.html"
    if static_file.exists():
        return FileResponse(static_file)
    else:
        return HTMLResponse("""
        <html>
            <head><title>Video Review Tool</title></head>
            <body>
                <h1>DriveGuard Video Review Tool</h1>
                <p>The static files are not yet created. Please create static/index.html</p>
            </body>
        </html>
        """)


@app.get("/api/status")
async def get_status():
    """Get current application status."""
    return {
        "status": "running",
        "session": current_session.model_dump() if current_session else None,
        "queue_stats": {
            "total_videos": len(current_queue.videos) if current_queue else 0,
            "current_index": current_queue.current_index if current_queue else 0,
            "videos_completed": sum(1 for v in current_queue.videos if v.review_status != "not_reviewed") if current_queue else 0
        },
        "total_clips": sum(len(clips) for clips in clip_markings.values())
    }


@app.get("/api/videos")
async def get_videos():
    """Get all videos in the queue."""
    if not current_queue:
        raise HTTPException(status_code=500, detail="Video queue not initialized")
    
    return {
        "videos": [video.model_dump() for video in current_queue.videos],
        "current_index": current_queue.current_index,
        "total": len(current_queue.videos)
    }


@app.get("/api/videos/current")
async def get_current_video():
    """Get the current video being reviewed."""
    if not current_queue:
        raise HTTPException(status_code=500, detail="Video queue not initialized")
    
    current_video = current_queue.get_current_video()
    if not current_video:
        raise HTTPException(status_code=404, detail="No current video")
    
    # Add clips for this video
    video_clips = clip_markings.get(current_video.video_id, [])
    current_video.clips_marked = len(video_clips)
    current_video.total_duration_marked = sum(clip.duration for clip in video_clips)
    
    # Include clips array in response for frontend timeline
    response = current_video.model_dump()
    response['clips'] = [clip.model_dump() for clip in video_clips]
    
    return response


@app.get("/api/videos/{video_id}")
async def get_video(video_id: str):
    """Get a specific video by ID."""
    if not current_queue:
        raise HTTPException(status_code=500, detail="Video queue not initialized")
    
    for video in current_queue.videos:
        if video.video_id == video_id:
            # Add clips for this video
            video_clips = clip_markings.get(video_id, [])
            video.clips_marked = len(video_clips)
            video.total_duration_marked = sum(clip.duration for clip in video_clips)
            
            # Include clips array in response for frontend timeline
            response = video.model_dump()
            response['clips'] = [clip.model_dump() for clip in video_clips]
            
            return response
    
    raise HTTPException(status_code=404, detail="Video not found")


@app.get("/api/videos/{video_id}/stream")
async def stream_video(video_id: str, request: Request):
    """Stream a video file."""
    if not current_queue:
        raise HTTPException(status_code=500, detail="Video queue not initialized")
    
    # Find the video
    video = None
    for v in current_queue.videos:
        if v.video_id == video_id:
            video = v
            break
    
    if not video:
        raise HTTPException(status_code=404, detail="Video not found")
    
    video_path = Path(video.video_path)
    if not video_path.exists():
        raise HTTPException(status_code=404, detail="Video file not found")
    
    # For now, just return the file directly
    # In a production system, you'd want proper streaming with range support
    return FileResponse(
        video_path,
        media_type="video/mp4",
        headers={"Accept-Ranges": "bytes"}
    )


def find_duplicate_clip(clip: ClipMarking, existing_clips: List[ClipMarking], tolerance: float = 0.5) -> Optional[ClipMarking]:
    """Check if a clip already exists with the same content (video_id + timing).
    
    Args:
        clip: The new clip to check for duplicates
        existing_clips: List of existing clips for this video
        tolerance: Time tolerance in seconds for considering clips as duplicates
        
    Returns:
        Existing clip if duplicate found, None otherwise
    """
    for existing_clip in existing_clips:
        # Check if timing overlaps within tolerance
        time_diff = abs(existing_clip.start_time - clip.start_time)
        duration_match = existing_clip.duration == clip.duration
        
        # Consider it a duplicate if start times are within tolerance and durations match
        if time_diff <= tolerance and duration_match:
            return existing_clip
    
    return None

@app.post("/api/clips")
async def create_or_update_clip(clip: ClipMarking):
    """Create a new clip marking or update an existing one."""
    global clip_markings
    
    # Ensure end_time is calculated
    clip.end_time = clip.start_time + clip.duration
    clip.last_modified = datetime.now()
    
    # Get existing clips for this video
    existing_clips = clip_markings.get(clip.video_id, [])
    
    # Check if this is an update (clip_id already exists)
    updated = False
    for i, existing_clip in enumerate(existing_clips):
        if existing_clip.clip_id == clip.clip_id:
            # Update existing clip, preserving original marked_at time
            clip.marked_at = existing_clip.marked_at
            clip_markings[clip.video_id][i] = clip
            updated = True
            break
    
    # If not an update, check for content-based duplicates
    if not updated:
        duplicate_clip = find_duplicate_clip(clip, existing_clips)
        if duplicate_clip:
            logger.info(f"Duplicate clip detected for video {clip.video_id}: "
                       f"new clip at {clip.start_time}s matches existing clip {duplicate_clip.clip_id} at {duplicate_clip.start_time}s")
            return {
                "success": True, 
                "clip_id": duplicate_clip.clip_id, 
                "updated": False,
                "duplicate": True,
                "message": f"Clip already exists at {duplicate_clip.start_time}s with duration {duplicate_clip.duration}s",
                "existing_clip": duplicate_clip.model_dump()
            }
        
        # Create new clip if no duplicate found
        if clip.video_id not in clip_markings:
            clip_markings[clip.video_id] = []
        
        # Set marked_at for new clips
        clip.marked_at = datetime.now()
        clip_markings[clip.video_id].append(clip)
    
    # Save to disk
    save_clip_markings()
    
    action = "Updated" if updated else "Created"
    logger.info(f"{action} clip {clip.clip_id} for video {clip.video_id}")
    return {"success": True, "clip_id": clip.clip_id, "updated": updated, "duplicate": False}


@app.get("/api/clips")
async def get_clips(video_id: Optional[str] = None):
    """Get clip markings, optionally filtered by video ID."""
    if video_id:
        clips = clip_markings.get(video_id, [])
        return {"clips": [clip.model_dump() for clip in clips]}
    else:
        all_clips = []
        for video_clips in clip_markings.values():
            all_clips.extend(clip.model_dump() for clip in video_clips)
        return {"clips": all_clips}


@app.delete("/api/clips/{clip_id}")
async def delete_clip(clip_id: str):
    """Delete a clip marking."""
    global clip_markings
    
    # Find and remove the clip
    for video_id, clips in clip_markings.items():
        for i, clip in enumerate(clips):
            if clip.clip_id == clip_id:
                del clips[i]
                save_clip_markings()
                logger.info(f"Deleted clip {clip_id}")
                return {"success": True}
    
    raise HTTPException(status_code=404, detail="Clip not found")


@app.get("/api/statistics/clips")
async def get_clip_statistics():
    """Get comprehensive statistics about all clip markings."""
    try:
        # Ensure clip markings are loaded
        if not clip_markings:
            load_clip_markings()
        
        stats = calculate_clip_statistics()
        logger.info(f"Generated statistics: {stats.total_clips} clips from {stats.total_videos_with_clips} videos")
        return stats.model_dump()
        
    except Exception as e:
        logger.error(f"Failed to calculate statistics: {e}")
        raise HTTPException(status_code=500, detail=f"Failed to calculate statistics: {str(e)}")


@app.post("/api/navigation/next")
async def next_video():
    """Move to the next video."""
    if not current_queue or not current_session:
        raise HTTPException(status_code=500, detail="Queue or session not initialized")
    
    if current_queue.advance_to_next():
        current_session.current_video_index = current_queue.current_index
        current_session.last_activity = datetime.now()
        save_session_state()
        
        current_video = current_queue.get_current_video()
        return {"success": True, "current_video": current_video.model_dump() if current_video else None}
    else:
        return {"success": False, "message": "Already at last video"}


@app.post("/api/navigation/previous")
async def previous_video():
    """Move to the previous video."""
    if not current_queue or not current_session:
        raise HTTPException(status_code=500, detail="Queue or session not initialized")
    
    if current_queue.go_to_previous():
        current_session.current_video_index = current_queue.current_index
        current_session.last_activity = datetime.now()
        save_session_state()
        
        current_video = current_queue.get_current_video()
        return {"success": True, "current_video": current_video.model_dump() if current_video else None}
    else:
        return {"success": False, "message": "Already at first video"}


class VideoStatusUpdate(BaseModel):
    status: str
    skip_reason: Optional[str] = None


@app.post("/api/videos/{video_id}/status")
async def update_video_status(video_id: str, update: VideoStatusUpdate):
    """Update the review status of a video."""
    if not current_queue or not current_session:
        raise HTTPException(status_code=500, detail="Queue or session not initialized")
    
    # Find and update the video
    for video in current_queue.videos:
        if video.video_id == video_id:
            video.review_status = update.status
            if update.skip_reason:
                video.skip_reason = update.skip_reason
            
            if video.review_completed is None:
                video.review_completed = datetime.now()
            
            # Update session stats
            if update.status == "skipped":
                current_session.videos_skipped += 1
            elif update.status in ["has_clips", "no_clips"]:
                current_session.videos_completed += 1
                if update.status == "has_clips":
                    current_session.videos_with_clips += 1
            
            save_session_state()
            return {"success": True}
    
    raise HTTPException(status_code=404, detail="Video not found")


@app.get("/api/export")
async def export_clips():
    """Export all clip markings."""
    all_clips = []
    for video_clips in clip_markings.values():
        all_clips.extend(video_clips)
    
    export = ClipExport(
        clips=all_clips,
        export_directory=str(REVIEW_PROGRESS_DIR)
    )
    export.calculate_statistics()
    
    export_file = REVIEW_PROGRESS_DIR / f"clip_export_{datetime.now().strftime('%Y%m%d_%H%M%S')}.json"
    with open(export_file, 'w') as f:
        json.dump(export.model_dump(), f, indent=2, default=str)
    
    logger.info(f"Exported {len(all_clips)} clips to {export_file}")
    return export.model_dump()

@app.get("/api/queue/status")
async def get_queue_status():
    """Get detailed queue status for debugging."""
    if not current_queue:
        return {"error": "Queue not initialized"}
    
    # Get completion statistics
    all_queue = video_manager.build_video_queue(include_completed=True)
    completed_videos = []
    pending_videos = []
    
    for video in all_queue.videos:
        if video_manager.is_video_completed(video.video_id):
            completed_videos.append({
                "video_id": video.video_id,
                "title": video.video_title[:50] + "..." if len(video.video_title) > 50 else video.video_title,
                "clips_marked": video.clips_marked,
                "category": video.video_category
            })
        else:
            pending_videos.append({
                "video_id": video.video_id,
                "title": video.video_title[:50] + "..." if len(video.video_title) > 50 else video.video_title,
                "category": video.video_category
            })
    
    return {
        "current_queue_size": len(current_queue.videos),
        "current_index": current_queue.current_index,
        "total_videos_discovered": len(all_queue.videos),
        "completed_videos_count": len(completed_videos),
        "pending_videos_count": len(pending_videos),
        "completed_videos": completed_videos,
        "pending_videos": pending_videos,
        "current_video": current_queue.get_current_video().model_dump() if current_queue.get_current_video() else None
    }


def main():
    """Main entry point for the application."""
    logger.info("Starting DriveGuard Video Review Tool")
    
    # Check if videos directory exists
    if not video_manager.videos_dir.exists():
        logger.error(f"Videos directory not found: {video_manager.videos_dir}")
        logger.error("Please ensure videos have been downloaded using the YouTube downloader first.")
        return
    
    # Start the server
    uvicorn.run(
        "evaluation.make_dataset.s2_video_reviewer.main:app",
        host="127.0.0.1",
        port=8000,
        reload=True,
        log_level="info"
    )


if __name__ == "__main__":
    main()