"""Data models for video review and clip marking system."""

from typing import Dict, List, Optional, Literal
from pydantic import BaseModel, Field
from datetime import datetime
from uuid import uuid4


class ClipMarking(BaseModel):
    """Model for a marked video clip."""
    
    clip_id: str = Field(default_factory=lambda: str(uuid4()))
    video_id: str = Field(..., description="YouTube video ID")
    video_path: str = Field(..., description="Full path to video file")
    video_category: str = Field(..., description="Category from download metadata")
    
    start_time: float = Field(..., ge=0, description="Clip start time in seconds")
    duration: float = Field(default=10.0, ge=3.0, le=60.0, description="Clip duration in seconds")
    end_time: float = Field(..., ge=0, description="Clip end time in seconds (calculated)")
    
    event_type: str = Field(default="", description="User-assigned event type")
    description: str = Field(default="", description="User description of the clip")
    severity_level: int = Field(default=3, ge=1, le=5, description="Safety severity (1=minor, 5=critical)")
    confidence: float = Field(default=1.0, ge=0.0, le=1.0, description="User confidence in clip quality")
    
    marked_at: datetime = Field(default_factory=datetime.now)
    last_modified: datetime = Field(default_factory=datetime.now)
    
    def model_post_init(self, __context):
        """Calculate end time after initialization."""
        if hasattr(self, 'start_time') and hasattr(self, 'duration'):
            self.end_time = self.start_time + self.duration


class VideoReviewStatus(BaseModel):
    """Model for tracking video review progress."""
    
    video_id: str = Field(..., description="YouTube video ID")
    video_path: str = Field(..., description="Full path to video file")
    video_title: str = Field(default="", description="Video title from metadata")
    video_category: str = Field(..., description="Category from download metadata")
    video_duration: float = Field(default=0.0, description="Total video duration in seconds")
    
    review_status: Literal["not_reviewed", "has_clips", "no_clips", "skipped"] = Field(
        default="not_reviewed",
        description="Current review status"
    )
    skip_reason: str = Field(default="", description="Reason for skipping (if skipped)")
    
    clips_marked: int = Field(default=0, description="Number of clips marked for this video")
    total_duration_marked: float = Field(default=0.0, description="Total duration of all clips")
    
    review_started: Optional[datetime] = Field(default=None)
    review_completed: Optional[datetime] = Field(default=None)
    review_duration_seconds: float = Field(default=0.0, description="Time spent reviewing this video")
    
    # Metadata from original download (read-only)
    relevance_score: int = Field(default=0)
    safety_keywords: List[str] = Field(default_factory=list)
    weather_mentioned: List[str] = Field(default_factory=list)
    time_of_day: List[str] = Field(default_factory=list)


class ReviewSession(BaseModel):
    """Model for tracking overall review progress."""
    
    session_id: str = Field(default_factory=lambda: str(uuid4()))
    started_at: datetime = Field(default_factory=datetime.now)
    last_activity: datetime = Field(default_factory=datetime.now)
    
    current_video_index: int = Field(default=0)
    current_video_id: Optional[str] = Field(default=None)
    
    total_videos: int = Field(default=0)
    videos_completed: int = Field(default=0)
    videos_skipped: int = Field(default=0)
    videos_with_clips: int = Field(default=0)
    
    total_clips_marked: int = Field(default=0)
    total_clip_duration: float = Field(default=0.0)
    
    # User preferences
    default_clip_duration: float = Field(default=10.0, ge=3.0, le=60.0)
    auto_advance_videos: bool = Field(default=True)
    show_metadata_hints: bool = Field(default=True)


class VideoMetadata(BaseModel):
    """Model for video metadata from download checkpoint (read-only)."""
    
    video_id: str
    title: str
    url: str
    category: str
    duration: Optional[float] = None
    upload_date: Optional[str] = None
    
    # Engagement metrics
    view_count: int = 0
    like_count: int = 0
    comment_count: int = 0
    
    # Channel info
    channel_name: Optional[str] = None
    channel_id: Optional[str] = None
    channel_url: Optional[str] = None
    
    # Content info
    description: str = ""
    tags: List[str] = Field(default_factory=list)
    thumbnail: Optional[str] = None
    
    # Technical info
    resolution: Optional[str] = None
    fps: Optional[float] = None
    video_codec: Optional[str] = None
    audio_codec: Optional[str] = None
    
    # DriveGuard-specific
    relevance_score: int = 0
    search_query_used: str = ""
    search_rank: int = 0
    safety_keywords: List[str] = Field(default_factory=list)
    weather_mentioned: List[str] = Field(default_factory=list)
    time_of_day: List[str] = Field(default_factory=list)
    
    # File info
    file_path: str
    file_size: int = 0
    download_time: str = ""


class VideoQueue(BaseModel):
    """Model for managing video review queue."""
    
    queue_id: str = Field(default_factory=lambda: str(uuid4()))
    created_at: datetime = Field(default_factory=datetime.now)
    last_updated: datetime = Field(default_factory=datetime.now)
    
    videos: List[VideoReviewStatus] = Field(default_factory=list)
    current_index: int = Field(default=0)
    
    # Queue settings
    sort_by_priority: bool = Field(default=True)
    filter_by_category: Optional[str] = Field(default=None)
    show_completed: bool = Field(default=False)
    shuffle_categories: bool = Field(default=True, description="Shuffle videos across categories for variety")
    first_unreviewed_index: int = Field(default=0, description="Index of the first unreviewed video in the queue")
    
    def get_current_video(self) -> Optional[VideoReviewStatus]:
        """Get the currently active video for review."""
        if 0 <= self.current_index < len(self.videos):
            return self.videos[self.current_index]
        return None
    
    def get_next_video(self) -> Optional[VideoReviewStatus]:
        """Get the next video to review."""
        if self.current_index + 1 < len(self.videos):
            return self.videos[self.current_index + 1]
        return None
    
    def get_previous_video(self) -> Optional[VideoReviewStatus]:
        """Get the previous video."""
        if self.current_index > 0:
            return self.videos[self.current_index - 1]
        return None
    
    def advance_to_next(self) -> bool:
        """Move to the next video. Returns True if successful."""
        if self.current_index + 1 < len(self.videos):
            self.current_index += 1
            self.last_updated = datetime.now()
            return True
        return False
    
    def go_to_previous(self) -> bool:
        """Move to the previous video. Returns True if successful."""
        if self.current_index > 0:
            self.current_index -= 1
            self.last_updated = datetime.now()
            return True
        return False


class ClipExport(BaseModel):
    """Model for exporting clip data for the next pipeline step."""
    
    export_id: str = Field(default_factory=lambda: str(uuid4()))
    export_time: datetime = Field(default_factory=datetime.now)
    
    total_clips: int = Field(default=0)
    total_videos_reviewed: int = Field(default=0)
    export_directory: str = Field(default="")
    
    clips: List[ClipMarking] = Field(default_factory=list)
    
    # Statistics
    clips_by_category: Dict[str, int] = Field(default_factory=dict)
    clips_by_severity: Dict[int, int] = Field(default_factory=dict)
    average_clip_duration: float = Field(default=0.0)
    
    def calculate_statistics(self):
        """Calculate export statistics from clips."""
        if not self.clips:
            return
        
        # Count by event type
        self.clips_by_category = {}
        for clip in self.clips:
            event_type = clip.event_type or clip.video_category
            self.clips_by_category[event_type] = self.clips_by_category.get(event_type, 0) + 1
        
        # Count by severity
        self.clips_by_severity = {}
        for clip in self.clips:
            severity = clip.severity_level
            self.clips_by_severity[severity] = self.clips_by_severity.get(severity, 0) + 1
        
        # Average duration
        total_duration = sum(clip.duration for clip in self.clips)
        self.average_clip_duration = total_duration / len(self.clips) if self.clips else 0.0
        
        # Update totals
        self.total_clips = len(self.clips)
        unique_videos = len(set(clip.video_id for clip in self.clips))
        self.total_videos_reviewed = unique_videos


class CategoryStats(BaseModel):
    """Model for detailed category statistics."""
    
    clip_count: int = 0
    video_count: int = 0
    total_duration: float = 0.0
    average_duration: float = 0.0
    severity_breakdown: Dict[int, int] = Field(default_factory=dict)
    event_type_breakdown: Dict[str, int] = Field(default_factory=dict)


class ClipStatistics(BaseModel):
    """Model for comprehensive clip statistics."""
    
    generated_at: datetime = Field(default_factory=datetime.now)
    
    # Overall stats
    total_clips: int = 0
    total_videos_with_clips: int = 0
    total_clip_duration: float = 0.0
    average_clip_duration: float = 0.0
    
    # Category breakdowns
    clips_by_category: Dict[str, int] = Field(default_factory=dict)
    clips_by_event_type: Dict[str, int] = Field(default_factory=dict)
    clips_by_severity: Dict[int, int] = Field(default_factory=dict)
    clips_by_video: Dict[str, int] = Field(default_factory=dict)
    
    # Detailed category stats
    category_details: Dict[str, CategoryStats] = Field(default_factory=dict)