/**
 * Main Application JavaScript for DriveGuard Video Review Tool
 * Manages video playback, clip creation, and UI interactions
 */

class VideoReviewApp {
    constructor() {
        // State
        this.currentVideo = null;
        this.clips = new Map(); // clipId -> clipData
        this.session = null;
        this.timeline = null;
        this.videoPlayer = null;
        
        // Configuration
        this.defaultClipDuration = 10;
        this.isLoading = false;
        this.currentPlaybackSpeed = 1.0; // Track current playback speed
        
        // Initialize
        this.init();
    }
    
    async init() {
        console.log('Initializing VideoReviewApp');
        
        try {
            // Initialize video player
            this.initVideoPlayer();
            
            // Initialize timeline
            this.initTimeline();
            
            // Bind UI events
            this.bindUIEvents();
            
            // Load initial data
            await this.loadInitialData();
            
            // Initialize playback speed from dropdown
            const speedDropdown = document.getElementById('playback-speed');
            this.currentPlaybackSpeed = parseFloat(speedDropdown.value);
            
            console.log('VideoReviewApp initialized successfully');
            
        } catch (error) {
            console.error('Failed to initialize app:', error);
            this.showError('Failed to initialize application: ' + error.message);
        }
    }
    
    initVideoPlayer() {
        this.videoPlayer = videojs('main-video', {
            fluid: false,
            responsive: false,
            controls: true,
            preload: 'auto',
            playbackRates: [0.5, 1, 1.25, 1.5, 2],
            width: 800,
            height: 240
        });
        
        // Bind video events
        this.videoPlayer.on('loadedmetadata', () => {
            this.updateVideoInfo();
        });
        
        this.videoPlayer.on('timeupdate', () => {
            this.updateTimeDisplay();
        });
        
        this.videoPlayer.on('error', (error) => {
            console.error('Video player error:', error);
            this.showError('Failed to load video');
        });
        
        console.log('Video player initialized');
    }
    
    initTimeline() {
        this.timeline = new InteractiveTimeline('timeline-canvas', this.videoPlayer);
        
        // Override timeline event handlers
        this.timeline.onClipMoved = (clipId, newTime) => {
            this.handleClipMoved(clipId, newTime);
        };
        
        this.timeline.onClipDeleted = (clipId) => {
            this.handleClipDeleted(clipId);
        };
        
        console.log('Timeline initialized');
    }
    
    bindUIEvents() {
        // Duration selector
        document.getElementById('clip-duration').addEventListener('change', (event) => {
            const value = event.target.value;
            const customInput = document.getElementById('custom-duration');
            
            if (value === 'custom') {
                customInput.style.display = 'inline-block';
                customInput.focus();
                this.defaultClipDuration = parseFloat(customInput.value) || 10;
            } else {
                customInput.style.display = 'none';
                this.defaultClipDuration = parseFloat(value);
            }
        });
        
        document.getElementById('custom-duration').addEventListener('change', (event) => {
            this.defaultClipDuration = parseFloat(event.target.value) || 10;
        });
        
        // Add clip button
        document.getElementById('add-clip-btn').addEventListener('click', () => {
            this.addClip();
        });
        
        // Navigation buttons
        document.getElementById('previous-video-btn').addEventListener('click', () => {
            this.previousVideo();
        });
        
        document.getElementById('next-video-btn').addEventListener('click', () => {
            this.nextVideo();
        });
        
        document.getElementById('skip-video-btn').addEventListener('click', () => {
            this.skipVideo();
        });
        
        document.getElementById('no-clips-btn').addEventListener('click', () => {
            this.markVideoAsNoClips();
        });
        
        document.getElementById('complete-btn').addEventListener('click', () => {
            this.completeVideo();
        });
        
        // Other controls
        document.getElementById('review-clips-btn').addEventListener('click', () => {
            this.reviewClips();
        });
        
        document.getElementById('export-clips-btn').addEventListener('click', () => {
            this.exportClips();
        });
        
        
        document.getElementById('clear-all-clips-btn').addEventListener('click', () => {
            this.clearAllClips();
        });
        
        document.getElementById('statistics-btn').addEventListener('click', () => {
            this.showStatistics();
        });
        
        // Playback speed
        document.getElementById('playback-speed').addEventListener('change', (event) => {
            const speed = parseFloat(event.target.value);
            this.currentPlaybackSpeed = speed; // Store the speed
            this.videoPlayer.playbackRate(speed);
        });
        
        // Modal events
        this.bindModalEvents();
        
        // Keyboard shortcuts
        document.addEventListener('keydown', (event) => {
            this.handleKeyboardShortcut(event);
        });
        
        console.log('UI events bound');
    }
    
    bindModalEvents() {
        // Clip edit modal
        const clipModal = document.getElementById('clip-edit-modal');
        const clipCloseButtons = clipModal.querySelectorAll('.modal-close');
        
        // Close modal events
        clipCloseButtons.forEach(button => {
            button.addEventListener('click', () => {
                this.closeClipEditModal();
            });
        });
        
        // Click outside to close
        clipModal.addEventListener('click', (event) => {
            if (event.target === clipModal) {
                this.closeClipEditModal();
            }
        });
        
        // Save clip button
        document.getElementById('save-clip-btn').addEventListener('click', () => {
            this.saveClipEdit();
        });
        
        // Delete clip button
        document.getElementById('delete-clip-btn').addEventListener('click', () => {
            this.deleteClipFromModal();
        });
        
        // Severity range
        document.getElementById('clip-severity').addEventListener('input', (event) => {
            document.getElementById('severity-value').textContent = event.target.value;
        });
        
        // Statistics modal
        const statsModal = document.getElementById('statistics-modal');
        const statsCloseButtons = statsModal.querySelectorAll('.modal-close');
        
        // Close statistics modal events
        statsCloseButtons.forEach(button => {
            button.addEventListener('click', () => {
                this.closeStatisticsModal();
            });
        });
        
        // Click outside to close statistics modal
        statsModal.addEventListener('click', (event) => {
            if (event.target === statsModal) {
                this.closeStatisticsModal();
            }
        });
    }
    
    handleKeyboardShortcut(event) {
        // Only handle shortcuts if not typing in an input
        if (event.target.tagName === 'INPUT' || event.target.tagName === 'TEXTAREA') {
            return;
        }
        
        switch (event.code) {
            case 'Space':
                event.preventDefault();
                if (this.videoPlayer.paused()) {
                    this.videoPlayer.play();
                } else {
                    this.videoPlayer.pause();
                }
                break;
                
            case 'KeyA':
                event.preventDefault();
                this.addClip();
                break;
                
            case 'KeyN':
                event.preventDefault();
                this.nextVideo();
                break;
                
            case 'KeyS':
                event.preventDefault();
                this.skipVideo();
                break;
                
            case 'ArrowLeft':
                event.preventDefault();
                this.videoPlayer.currentTime(this.videoPlayer.currentTime() - 5);
                break;
                
            case 'ArrowRight':
                event.preventDefault();
                this.videoPlayer.currentTime(this.videoPlayer.currentTime() + 5);
                break;
        }
    }
    
    async loadInitialData() {
        this.showLoading('Loading initial data...');
        
        try {
            // Get app status
            const statusResponse = await fetch('/api/status');
            const status = await statusResponse.json();
            this.session = status.session;
            
            // Load current video
            await this.loadCurrentVideo();
            
            this.hideLoading();
            
        } catch (error) {
            this.hideLoading();
            throw error;
        }
    }
    
    async loadCurrentVideo() {
        try {
            console.log('=== LOADING CURRENT VIDEO ===');
            
            // Refresh session data first
            const statusResponse = await fetch('/api/status');
            const status = await statusResponse.json();
            this.session = status.session;
            console.log('Session updated:', this.session);
            
            const response = await fetch('/api/videos/current');
            
            if (!response.ok) {
                throw new Error('Failed to load current video');
            }
            
            const videoData = await response.json();
            console.log('Received video data:', {
                video_id: videoData.video_id,
                video_title: videoData.video_title,
                clips_count: videoData.clips ? videoData.clips.length : 0
            });
            
            // FORCE complete timeline reset
            console.log('Before clearing - Timeline clips:', this.timeline.clips.size, 'App clips:', this.clips.size);
            
            // Double-clear approach: timeline method + direct DOM clearing
            this.timeline.clearAllClips();
            this.clips.clear();
            
            // Also directly clear the clips container DOM
            const clipsContainer = document.getElementById('timeline-clips');
            if (clipsContainer) {
                clipsContainer.innerHTML = '';
                console.log('Directly cleared timeline-clips container DOM');
            }
            
            console.log('After clearing - Timeline clips:', this.timeline.clips.size, 'App clips:', this.clips.size);
            
            // Set current video AFTER clearing
            this.currentVideo = videoData;
            
            // Update UI
            this.updateVideoDisplay();
            
            // Load video in player
            this.loadVideoInPlayer();
            
            // Load existing clips
            this.loadVideoClips();
            
            console.log('=== VIDEO LOADING COMPLETE ===');
            
        } catch (error) {
            console.error('Failed to load current video:', error);
            this.showError('No videos available for review');
        }
    }
    
    updateVideoDisplay() {
        if (!this.currentVideo) return;
        
        // Update header info
        const videoCounter = document.getElementById('video-counter');
        const categoryInfo = document.getElementById('category-info');
        
        if (this.session) {
            const current = this.session.current_video_index + 1;
            const total = this.session.total_videos;
            videoCounter.textContent = `Video ${current}/${total}`;
        }
        
        categoryInfo.textContent = `Category: ${this.currentVideo.video_category}`;
        
        // Update video info
        document.getElementById('video-title').textContent = this.currentVideo.video_title || 'Untitled Video';
        
        const keywords = this.currentVideo.safety_keywords || [];
        document.getElementById('video-keywords').textContent = keywords.length > 0 ? 
            `Keywords: ${keywords.join(', ')}` : 'Keywords: none';
    }
    
    loadVideoInPlayer() {
        if (!this.currentVideo) return;
        
        const videoUrl = `/api/videos/${this.currentVideo.video_id}/stream`;
        this.videoPlayer.src({ type: 'video/mp4', src: videoUrl });
        this.videoPlayer.load();
        
        // Reapply playback speed after video loads
        this.videoPlayer.on('loadedmetadata', () => {
            if (this.currentPlaybackSpeed !== 1.0) {
                this.videoPlayer.playbackRate(this.currentPlaybackSpeed);
                console.log(`Reapplied playback speed: ${this.currentPlaybackSpeed}x`);
            }
        });
    }
    
    loadVideoClips() {
        console.log('=== LOADING CLIPS FOR VIDEO ===');
        console.log('Video ID:', this.currentVideo.video_id);
        console.log('Video data clips:', this.currentVideo.clips);
        
        if (!this.currentVideo.clips || this.currentVideo.clips.length === 0) {
            console.log('No clips to load for this video - should be empty timeline');
            this.updateClipsDisplay();
            return;
        }
        
        console.log('Loading', this.currentVideo.clips.length, 'clips for video', this.currentVideo.video_id);
        
        // Add clips to timeline
        this.currentVideo.clips.forEach((clipData, index) => {
            console.log(`Loading clip ${index + 1}/${this.currentVideo.clips.length}:`, {
                clip_id: clipData.clip_id,
                video_id: clipData.video_id, 
                start_time: clipData.start_time,
                duration: clipData.duration
            });
            
            // Verify this clip belongs to current video
            if (clipData.video_id !== this.currentVideo.video_id) {
                console.error(`MISMATCH: Clip ${clipData.clip_id} belongs to video ${clipData.video_id} but current video is ${this.currentVideo.video_id}`);
            }
            
            this.clips.set(clipData.clip_id, clipData);
            this.timeline.addClip(clipData);
        });
        
        this.updateClipsDisplay();
        
        console.log('Final state - App clips:', this.clips.size, 'Timeline clips:', this.timeline.clips.size);
        console.log('=== CLIP LOADING COMPLETE ===');
    }
    
    updateVideoInfo() {
        const duration = this.videoPlayer.duration();
        if (duration && !isNaN(duration)) {
            document.getElementById('video-duration').textContent = 
                `Duration: ${this.formatTime(duration)}`;
            document.getElementById('total-time').textContent = this.formatTime(duration);
        }
    }
    
    updateTimeDisplay() {
        const currentTime = this.videoPlayer.currentTime();
        if (currentTime !== undefined && !isNaN(currentTime)) {
            document.getElementById('current-time').textContent = this.formatTime(currentTime);
        }
    }
    
    formatTime(seconds) {
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    
    // Clip management methods
    
    async addClip() {
        if (!this.currentVideo) {
            this.showError('No video loaded');
            return;
        }
        
        const currentTime = this.videoPlayer.currentTime() || 0;
        const duration = this.defaultClipDuration;
        const videoDuration = this.videoPlayer.duration() || 0;
        
        // Check if clip would exceed video duration
        if (currentTime + duration > videoDuration) {
            this.showError('Clip would extend beyond video end');
            return;
        }
        
        // Check for overlapping clips
        if (this.timeline.hasOverlappingClips(currentTime, duration)) {
            this.showError('Clip would overlap with existing clip');
            return;
        }
        
        // Create clip data
        const clipData = {
            clip_id: this.generateClipId(),
            video_id: this.currentVideo.video_id,
            video_path: this.currentVideo.video_path,
            video_category: this.currentVideo.video_category,
            start_time: currentTime,
            duration: duration,
            end_time: currentTime + duration,
            event_type: "", // Default to empty, user will select
            description: '',
            severity_level: 3,
            confidence: 1.0
        };
        
        try {
            // Save to server first
            const response = await fetch('/api/clips', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(clipData)
            });
            
            if (!response.ok) {
                throw new Error('Failed to save clip to server');
            }
            
            // Add clip to timeline (this makes it draggable)
            this.timeline.addClip(clipData);
            
            // Store locally
            this.clips.set(clipData.clip_id, clipData);
            
            // Update display
            this.updateClipsDisplay();
            
            console.log(`Added clip ${clipData.clip_id} at ${currentTime}s`);
            
        } catch (error) {
            console.error('Failed to add clip:', error);
            this.showError('Failed to add clip');
        }
    }
    
    async handleClipMoved(clipId, newTime) {
        const clipData = this.clips.get(clipId);
        if (!clipData) return;
        
        // Update clip data
        clipData.start_time = newTime;
        clipData.end_time = newTime + clipData.duration;
        
        // Save to server
        await this.saveClipToServer(clipData);
        
        // Update display
        this.updateClipsDisplay();
        
        console.log(`Clip ${clipId} moved to ${newTime}s`);
    }
    
    async handleClipDeleted(clipId) {
        if (confirm('Are you sure you want to delete this clip?')) {
            await this.deleteClip(clipId);
        }
    }
    
    async deleteClip(clipId) {
        try {
            // Delete from server first
            const response = await fetch(`/api/clips/${clipId}`, { method: 'DELETE' });
            if (!response.ok) {
                throw new Error('Failed to delete clip from server');
            }
            
            // Remove from timeline
            this.timeline.removeClip(clipId);
            
            // Remove from local storage
            this.clips.delete(clipId);
            
            // Update display
            this.updateClipsDisplay();
            
            console.log(`Deleted clip ${clipId}`);
            
        } catch (error) {
            console.error('Failed to delete clip:', error);
            this.showError('Failed to delete clip');
        }
    }
    
    updateClipsDisplay() {
        const clipsList = document.getElementById('clips-list');
        const clipsCount = document.getElementById('clips-count');
        const clearAllBtn = document.getElementById('clear-all-clips-btn');
        const totalClipsCounter = document.getElementById('clips-counter');
        
        // Update counts
        const clipCount = this.clips.size;
        clipsCount.textContent = clipCount;
        
        // Show/hide clear all button
        clearAllBtn.style.display = clipCount > 0 ? 'inline-block' : 'none';
        
        // Update total clips counter (this would need to be fetched from server for accuracy)
        // For now, just show current video clips
        
        // Clear and rebuild clips list
        clipsList.innerHTML = '';
        
        if (clipCount === 0) {
            clipsList.innerHTML = `
                <div class="no-clips-message">
                    <p>No clips marked yet. Click "Add Clip" and drag it to the desired position on the timeline.</p>
                    <div class="workflow-hint">
                        <strong>Workflow:</strong>
                        <ol>
                            <li>Set clip duration (default 10s)</li>
                            <li>Click "Add Clip" to create a draggable clip</li>
                            <li>Drag clip on timeline to see video preview</li>
                            <li>Drop when satisfied with content</li>
                            <li>Right-click clip to delete if needed</li>
                        </ol>
                    </div>
                </div>
            `;
            return;
        }
        
        // Sort clips by start time
        const sortedClips = Array.from(this.clips.values()).sort((a, b) => a.start_time - b.start_time);
        
        sortedClips.forEach(clip => {
            const clipElement = this.createClipListItem(clip);
            clipsList.appendChild(clipElement);
        });
    }
    
    createClipListItem(clip) {
        const div = document.createElement('div');
        div.className = 'clip-item';
        div.dataset.clipId = clip.clip_id;
        
        const startTime = this.formatTime(clip.start_time);
        const endTime = this.formatTime(clip.end_time);
        const description = clip.description || 'No description';
        const category = clip.event_type || clip.video_category;
        
        div.innerHTML = `
            <div class="clip-info">
                <div class="clip-icon severity-${clip.severity_level}"></div>
                <div class="clip-timing">${startTime}-${endTime} (${clip.duration}s)</div>
                <div class="clip-description-edit">
                    <label>Description:</label>
                    <input type="text" class="description-input" value="${clip.description || ''}" 
                           placeholder="Add description..." 
                           onchange="app.updateClipDescription('${clip.clip_id}', this.value)">
                </div>
                <div class="clip-event-type">
                    <label>Event Type:</label>
                    <select class="event-type-dropdown" onchange="app.updateClipEventType('${clip.clip_id}', this.value)">
                        <option value="">Select event type...</option>
                        <option value="collision_accident" ${clip.event_type === 'collision_accident' ? 'selected' : ''}>Collision/Accident</option>
                        <option value="near_miss" ${clip.event_type === 'near_miss' ? 'selected' : ''}>Near Miss</option>
                        <option value="traffic_violation" ${clip.event_type === 'traffic_violation' ? 'selected' : ''}>Traffic Violation</option>
                        <option value="risky_behavior" ${clip.event_type === 'risky_behavior' ? 'selected' : ''}>Risky Behavior</option>
                        <option value="safe_maneuver" ${clip.event_type === 'safe_maneuver' ? 'selected' : ''}>Safe Maneuver</option>
                        <option value="environmental_hazard" ${clip.event_type === 'environmental_hazard' ? 'selected' : ''}>Environmental Hazard</option>
                        <option value="normal_driving" ${clip.event_type === 'normal_driving' ? 'selected' : ''}>Normal Driving</option>
                    </select>
                </div>
            </div>
            <div class="clip-actions">
                <button class="clip-action-btn play" onclick="app.playClip('${clip.clip_id}')">Play</button>
                <button class="clip-action-btn delete" onclick="app.deleteClip('${clip.clip_id}')">Delete</button>
            </div>
        `;
        
        return div;
    }
    
    playClip(clipId) {
        const clip = this.clips.get(clipId);
        if (!clip) return;
        
        // Jump to clip start and play
        this.videoPlayer.currentTime(clip.start_time);
        this.videoPlayer.play();
        
        // Optionally pause at clip end
        const pauseAtEnd = () => {
            if (this.videoPlayer.currentTime() >= clip.end_time) {
                this.videoPlayer.pause();
                this.videoPlayer.off('timeupdate', pauseAtEnd);
            }
        };
        
        this.videoPlayer.on('timeupdate', pauseAtEnd);
    }
    
    editClip(clipId) {
        console.log('editClip called with clipId:', clipId);
        
        const clip = this.clips.get(clipId);
        if (!clip) {
            console.error('Clip not found for id:', clipId);
            return;
        }
        
        console.log('Found clip:', clip);
        
        // Get modal element
        const modal = document.getElementById('clip-edit-modal');
        if (!modal) {
            console.error('Modal element not found');
            return;
        }
        
        console.log('Modal element found:', modal);
        
        // Populate modal with clip data
        const eventTypeEl = document.getElementById('clip-event-type');
        const descriptionEl = document.getElementById('clip-description');
        const severityEl = document.getElementById('clip-severity');
        const severityValueEl = document.getElementById('severity-value');
        const startTimeEl = document.getElementById('clip-start-time');
        const endTimeEl = document.getElementById('clip-end-time');
        
        if (!eventTypeEl) {
            console.error('Event type element not found');
            return;
        }
        
        eventTypeEl.value = clip.event_type || '';
        descriptionEl.value = clip.description || '';
        severityEl.value = clip.severity_level || 3;
        severityValueEl.textContent = clip.severity_level || 3;
        startTimeEl.value = clip.start_time.toFixed(1);
        endTimeEl.value = clip.end_time.toFixed(1);
        
        // Store current clip ID for saving
        modal.dataset.clipId = clipId;
        
        // Show modal with multiple approaches
        console.log('Showing modal...');
        modal.classList.add('show');
        modal.style.display = 'block';
        
        console.log('Modal classes after show:', modal.className);
        console.log('Modal display style:', modal.style.display);
        console.log('Modal computed display:', window.getComputedStyle(modal).display);
        
        // Force a reflow
        modal.offsetHeight;
        
        console.log('Modal should be visible now');
    }
    
    closeClipEditModal() {
        document.getElementById('clip-edit-modal').classList.remove('show');
    }
    
    testModal() {
        console.log('Test modal button clicked');
        
        const modal = document.getElementById('clip-edit-modal');
        if (!modal) {
            console.error('Modal not found');
            alert('Modal element not found!');
            return;
        }
        
        console.log('Modal found, testing different show methods...');
        
        // Try multiple approaches
        modal.style.display = 'block';
        modal.classList.add('show');
        modal.style.visibility = 'visible';
        modal.style.opacity = '1';
        modal.style.zIndex = '9999';
        
        console.log('Modal styles set, checking visibility...');
        console.log('Display:', window.getComputedStyle(modal).display);
        console.log('Visibility:', window.getComputedStyle(modal).visibility);
        console.log('Opacity:', window.getComputedStyle(modal).opacity);
        
        alert('Modal should be visible now. Check console for debug info.');
    }
    
    async updateClipEventType(clipId, eventType) {
        console.log('Updating clip event type:', clipId, eventType);
        
        const clip = this.clips.get(clipId);
        if (!clip) {
            console.error('Clip not found:', clipId);
            return;
        }
        
        // Update the clip data
        clip.event_type = eventType;
        
        // Save to server
        await this.saveClipToServer(clip);
        
        // Update the timeline clip position (refresh display)
        this.timeline.updateClipPosition(clipId);
        
        console.log('Updated clip event type to:', eventType);
    }
    
    async updateClipDescription(clipId, description) {
        console.log('Updating clip description:', clipId, description);
        
        const clip = this.clips.get(clipId);
        if (!clip) {
            console.error('Clip not found:', clipId);
            return;
        }
        
        // Update the clip data
        clip.description = description;
        
        // Save to server
        await this.saveClipToServer(clip);
        
        console.log('Updated clip description to:', description);
    }
    
    async saveClipToServer(clip) {
        try {
            // Update the clip on the server - first delete then create
            // (since there's no PUT endpoint, we'll use the POST endpoint which handles updates)
            const response = await fetch('/api/clips', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(clip)
            });
            
            if (!response.ok) {
                console.error('Failed to save clip to server');
            }
        } catch (error) {
            console.error('Error saving clip to server:', error);
        }
    }
    
    saveClipEdit() {
        const modal = document.getElementById('clip-edit-modal');
        const clipId = modal.dataset.clipId;
        const clip = this.clips.get(clipId);
        
        if (!clip) return;
        
        // Update clip data from form
        clip.event_type = document.getElementById('clip-event-type').value;
        clip.description = document.getElementById('clip-description').value;
        clip.severity_level = parseInt(document.getElementById('clip-severity').value);
        
        const startTime = parseFloat(document.getElementById('clip-start-time').value);
        const endTime = parseFloat(document.getElementById('clip-end-time').value);
        
        if (startTime >= 0 && endTime > startTime) {
            clip.start_time = startTime;
            clip.end_time = endTime;
            clip.duration = endTime - startTime;
            
            // Update timeline position
            this.timeline.updateClipPosition(clipId);
        }
        
        // Update display
        this.updateClipsDisplay();
        
        // Close modal
        this.closeClipEditModal();
        
        console.log(`Updated clip ${clipId}`);
    }
    
    deleteClipFromModal() {
        const modal = document.getElementById('clip-edit-modal');
        const clipId = modal.dataset.clipId;
        
        if (confirm('Are you sure you want to delete this clip?')) {
            this.deleteClip(clipId);
            this.closeClipEditModal();
        }
    }
    
    async clearAllClips() {
        if (this.clips.size === 0) return;
        
        if (confirm(`Are you sure you want to delete all ${this.clips.size} clips?`)) {
            // Delete clips from server
            const clipIds = Array.from(this.clips.keys());
            for (const clipId of clipIds) {
                try {
                    const response = await fetch(`/api/clips/${clipId}`, { method: 'DELETE' });
                    if (!response.ok) {
                        console.error(`Failed to delete clip ${clipId} from server`);
                    }
                } catch (error) {
                    console.error(`Error deleting clip ${clipId}:`, error);
                }
            }
            
            // Clear local clips
            this.timeline.clearAllClips();
            this.clips.clear();
            this.updateClipsDisplay();
        }
    }
    
    // Navigation methods
    
    async previousVideo() {
        await this.saveCurrentVideoClips();
        
        try {
            const response = await fetch('/api/navigation/previous', { method: 'POST' });
            const result = await response.json();
            
            if (result.success) {
                await this.loadCurrentVideo();
            } else {
                this.showInfo(result.message || 'Already at first video');
            }
        } catch (error) {
            console.error('Failed to navigate to previous video:', error);
            this.showError('Failed to navigate to previous video');
        }
    }
    
    async nextVideo() {
        await this.saveCurrentVideoClips();
        
        try {
            const response = await fetch('/api/navigation/next', { method: 'POST' });
            const result = await response.json();
            
            if (result.success) {
                await this.loadCurrentVideo();
            } else {
                this.showInfo(result.message || 'Already at last video');
            }
        } catch (error) {
            console.error('Failed to navigate to next video:', error);
            this.showError('Failed to navigate to next video');
        }
    }
    
    async skipVideo() {
        if (confirm('Skip this video? This will mark it as reviewed but skipped.')) {
            await this.updateVideoStatus('skipped', 'Not useful for dataset');
            await this.nextVideo();
        }
    }
    
    async markVideoAsNoClips() {
        if (confirm('Mark this video as reviewed with no clips?')) {
            await this.updateVideoStatus('no_clips');
            await this.nextVideo();
        }
    }
    
    async completeVideo() {
        if (this.clips.size === 0) {
            if (!confirm('Complete this video with no clips marked?')) {
                return;
            }
            await this.updateVideoStatus('no_clips');
        } else {
            await this.saveCurrentVideoClips();
            await this.updateVideoStatus('has_clips');
        }
        
        await this.nextVideo();
    }
    
    async updateVideoStatus(status, skipReason = null) {
        if (!this.currentVideo) return;
        
        try {
            const response = await fetch(`/api/videos/${this.currentVideo.video_id}/status`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ status, skip_reason: skipReason })
            });
            
            if (!response.ok) {
                throw new Error('Failed to update video status');
            }
            
            console.log(`Updated video ${this.currentVideo.video_id} status to ${status}`);
            
        } catch (error) {
            console.error('Failed to update video status:', error);
            this.showError('Failed to update video status');
        }
    }
    
    async saveCurrentVideoClips() {
        if (!this.currentVideo || this.clips.size === 0) return;
        
        try {
            // Save all clips for current video
            for (const clip of this.clips.values()) {
                const response = await fetch('/api/clips', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(clip)
                });
                
                if (!response.ok) {
                    throw new Error('Failed to save clip');
                }
            }
            
            console.log(`Saved ${this.clips.size} clips for video ${this.currentVideo.video_id}`);
            
        } catch (error) {
            console.error('Failed to save clips:', error);
            this.showError('Failed to save clips');
        }
    }
    
    // Other methods
    
    reviewClips() {
        // TODO: Implement clip review mode
        this.showInfo('Clip review mode coming soon!');
    }
    
    async exportClips() {
        try {
            this.showLoading('Exporting clips...');
            
            const response = await fetch('/api/export');
            const exportData = await response.json();
            
            this.hideLoading();
            this.showSuccess(`Exported ${exportData.total_clips} clips successfully!`);
            
        } catch (error) {
            this.hideLoading();
            console.error('Failed to export clips:', error);
            this.showError('Failed to export clips');
        }
    }
    
    async showStatistics() {
        try {
            const modal = document.getElementById('statistics-modal');
            const loadingElement = document.getElementById('statistics-loading');
            const contentElement = document.getElementById('statistics-content');
            const errorElement = document.getElementById('statistics-error');
            
            // Show modal and loading state
            modal.classList.add('show');
            modal.style.display = 'block';
            loadingElement.style.display = 'flex';
            contentElement.style.display = 'none';
            errorElement.style.display = 'none';
            
            // Fetch statistics from server
            const response = await fetch('/api/statistics/clips');
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const stats = await response.json();
            
            // Populate statistics content
            this.populateStatistics(stats);
            
            // Hide loading and show content
            loadingElement.style.display = 'none';
            contentElement.style.display = 'block';
            
            console.log('Statistics loaded successfully');
            
        } catch (error) {
            console.error('Failed to load statistics:', error);
            
            // Hide loading and show error
            document.getElementById('statistics-loading').style.display = 'none';
            document.getElementById('statistics-content').style.display = 'none';
            document.getElementById('statistics-error').style.display = 'block';
        }
    }
    
    populateStatistics(stats) {
        // Update overall statistics
        document.getElementById('stat-total-clips').textContent = stats.total_clips || 0;
        document.getElementById('stat-total-videos').textContent = stats.total_videos_with_clips || 0;
        document.getElementById('stat-total-duration').textContent = `${(stats.total_clip_duration || 0).toFixed(1)}s`;
        document.getElementById('stat-avg-duration').textContent = `${(stats.average_clip_duration || 0).toFixed(1)}s`;
        
        // Populate category statistics
        this.populateStatsTable('category-stats', stats.clips_by_category, 'Category');
        
        // Populate event type statistics
        this.populateStatsTable('event-stats', stats.clips_by_event_type, 'Event Type');
        
        // Populate severity statistics with level names
        const severityLabels = {
            1: 'Minor (1)',
            2: 'Low (2)',
            3: 'Moderate (3)',
            4: 'High (4)',
            5: 'Critical (5)'
        };
        this.populateStatsTable('severity-stats', stats.clips_by_severity, 'Severity Level', severityLabels);
    }
    
    populateStatsTable(containerId, data, labelColumn, customLabels = null) {
        const container = document.getElementById(containerId);
        
        if (!data || Object.keys(data).length === 0) {
            container.innerHTML = '<p class="no-data">No data available</p>';
            return;
        }
        
        // Sort entries by count (descending)
        const sortedEntries = Object.entries(data).sort(([,a], [,b]) => b - a);
        
        const table = document.createElement('table');
        table.className = 'stats-table';
        
        // Create header
        const thead = document.createElement('thead');
        thead.innerHTML = `
            <tr>
                <th>${labelColumn}</th>
                <th>Count</th>
                <th>Percentage</th>
            </tr>
        `;
        table.appendChild(thead);
        
        // Create body
        const tbody = document.createElement('tbody');
        const total = Object.values(data).reduce((sum, count) => sum + count, 0);
        
        sortedEntries.forEach(([key, count]) => {
            const percentage = total > 0 ? ((count / total) * 100).toFixed(1) : '0.0';
            const displayKey = customLabels ? customLabels[key] || key : key;
            
            const row = document.createElement('tr');
            row.innerHTML = `
                <td><span class="category-name">${displayKey}</span></td>
                <td><span class="stat-number">${count}</span></td>
                <td>${percentage}%</td>
            `;
            tbody.appendChild(row);
        });
        
        table.appendChild(tbody);
        container.innerHTML = '';
        container.appendChild(table);
    }
    
    closeStatisticsModal() {
        const modal = document.getElementById('statistics-modal');
        modal.classList.remove('show');
        modal.style.display = 'none';
        
        // Reset modal state
        document.getElementById('statistics-loading').style.display = 'none';
        document.getElementById('statistics-content').style.display = 'none';
        document.getElementById('statistics-error').style.display = 'none';
        
        console.log('Statistics modal closed');
    }
    
    // Utility methods
    
    generateClipId() {
        return 'clip_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }
    
    showLoading(message = 'Loading...') {
        const overlay = document.getElementById('loading-overlay');
        const messageElement = document.getElementById('loading-message');
        messageElement.textContent = message;
        overlay.style.display = 'flex';
        this.isLoading = true;
    }
    
    hideLoading() {
        document.getElementById('loading-overlay').style.display = 'none';
        this.isLoading = false;
    }
    
    showError(message) {
        // Simple alert for now - could be enhanced with toast notifications
        alert('Error: ' + message);
        console.error(message);
    }
    
    showSuccess(message) {
        alert('Success: ' + message);
        console.log(message);
    }
    
    showInfo(message) {
        alert('Info: ' + message);
        console.log(message);
    }
}

// Initialize app when page loads
let app;
document.addEventListener('DOMContentLoaded', () => {
    app = new VideoReviewApp();
});

// Make app available globally for debugging
window.app = app;