/**
 * Interactive Timeline Component for DriveGuard Video Review Tool
 * Handles drag-and-drop clip marking with real-time video preview
 */

class InteractiveTimeline {
    constructor(canvasId, videoPlayer, options = {}) {
        this.canvas = document.getElementById(canvasId);
        this.ctx = this.canvas.getContext('2d');
        this.videoPlayer = videoPlayer;
        this.clipsContainer = document.getElementById('timeline-clips');
        this.playhead = document.getElementById('timeline-playhead');
        
        // Configuration
        this.options = {
            height: 60,
            padding: 8,
            clipHeight: 44,
            clipMinWidth: 30,
            pixelsPerSecond: 10,
            gridInterval: 30, // seconds
            ...options
        };
        
        // State
        this.videoDuration = 0;
        this.currentTime = 0;
        this.clips = new Map(); // clipId -> clipData
        this.dragging = null;
        this.dragOffset = { x: 0, y: 0 };
        this.isPreviewMode = false;
        this.originalVideoTime = 0;
        
        // Initialize
        this.setupCanvas();
        this.bindEvents();
        this.render();
        
        console.log('InteractiveTimeline initialized');
    }
    
    setupCanvas() {
        // Set canvas size based on container
        const container = this.canvas.parentElement;
        const rect = container.getBoundingClientRect();
        
        // Only set initial canvas size, don't override if already set by setVideoDuration
        if (this.canvas.width === 0) {
            this.canvas.width = rect.width;
        }
        this.canvas.height = this.options.height;
        
        // Set CSS size to prevent scaling
        this.canvas.style.width = this.canvas.width + 'px';
        this.canvas.style.height = this.options.height + 'px';
        
        // Don't update pixels per second here - it's now fixed at 2 pixels/second
    }
    
    bindEvents() {
        // Canvas events
        this.canvas.addEventListener('click', this.handleCanvasClick.bind(this));
        
        // Window resize
        window.addEventListener('resize', () => {
            this.setupCanvas();
            this.render();
            this.updateClipPositions();
        });
        
        // Video player events
        this.videoPlayer.on('loadedmetadata', () => {
            this.setVideoDuration(this.videoPlayer.duration());
        });
        
        this.videoPlayer.on('timeupdate', () => {
            if (!this.isPreviewMode) {
                this.setCurrentTime(this.videoPlayer.currentTime());
            }
        });
        
        // Drag and drop events will be bound to clip elements
    }
    
    handleCanvasClick(event) {
        const rect = this.canvas.getBoundingClientRect();
        const x = event.clientX - rect.left;
        const y = event.clientY - rect.top;
        
        // Convert click position to time
        const clickTime = this.pixelToTime(x);
        
        // Seek video to clicked position
        this.videoPlayer.currentTime(clickTime);
        
        console.log(`Timeline clicked at ${clickTime.toFixed(1)}s`);
    }
    
    setVideoDuration(duration) {
        this.videoDuration = duration;
        
        // Use a fixed scale for better visual representation
        // 5 pixels per second = 10 seconds takes 50 pixels
        // This provides better visibility for typical 10-second clips
        this.options.pixelsPerSecond = 5;
        
        // Calculate required canvas width
        const requiredWidth = (duration * this.options.pixelsPerSecond) + (this.options.padding * 2);
        const minWidth = this.canvas.parentElement.clientWidth;
        
        // Set canvas width to accommodate full video duration
        this.canvas.width = Math.max(requiredWidth, minWidth);
        this.canvas.style.width = this.canvas.width + 'px';
        
        // Update clips container to match canvas width
        this.clipsContainer.style.width = this.canvas.width + 'px';
        
        // Update inner container to match canvas width for proper scrolling
        const innerContainer = document.getElementById('timeline-inner');
        if (innerContainer) {
            innerContainer.style.width = this.canvas.width + 'px';
        }
        
        this.render();
        this.updateClipPositions();
        this.updateTimeMarkers();
        console.log(`Video duration set to ${duration}s, timeline width: ${this.canvas.width}px`);
    }
    
    updateTimeMarkers() {
        const markersContainer = document.getElementById('time-markers');
        if (!markersContainer) return;
        
        markersContainer.innerHTML = '';
        
        // Generate markers every 30 seconds
        const interval = 30; // seconds
        const markerCount = Math.ceil(this.videoDuration / interval) + 1;
        
        for (let i = 0; i < markerCount; i++) {
            const time = i * interval;
            if (time > this.videoDuration) break;
            
            const marker = document.createElement('span');
            marker.className = 'time-marker';
            marker.textContent = this.formatTime(time);
            
            // Position marker
            const x = this.timeToPixel(time);
            marker.style.position = 'absolute';
            marker.style.left = x + 'px';
            marker.style.transform = 'translateX(-50%)';
            
            markersContainer.appendChild(marker);
        }
    }
    
    setCurrentTime(time) {
        this.currentTime = time;
        this.updatePlayhead();
    }
    
    updatePlayhead() {
        const x = this.timeToPixel(this.currentTime);
        this.playhead.style.left = x + 'px';
    }
    
    timeToPixel(time) {
        return this.options.padding + (time * this.options.pixelsPerSecond);
    }
    
    pixelToTime(pixel) {
        return Math.max(0, (pixel - this.options.padding) / this.options.pixelsPerSecond);
    }
    
    render() {
        // Clear canvas
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        
        // Draw background
        this.ctx.fillStyle = '#f8f9fa';
        this.ctx.fillRect(0, 0, this.canvas.width, this.canvas.height);
        
        // Draw time grid
        this.drawTimeGrid();
        
        // Draw waveform placeholder (could be enhanced later)
        this.drawWaveformPlaceholder();
    }
    
    drawTimeGrid() {
        if (this.videoDuration === 0) return;
        
        this.ctx.strokeStyle = '#e9ecef';
        this.ctx.lineWidth = 1;
        
        // Draw vertical grid lines every 30 seconds
        for (let time = 0; time <= this.videoDuration; time += this.options.gridInterval) {
            const x = this.timeToPixel(time);
            
            this.ctx.beginPath();
            this.ctx.moveTo(x, 0);
            this.ctx.lineTo(x, this.canvas.height);
            this.ctx.stroke();
        }
        
        // Draw horizontal center line
        const centerY = this.canvas.height / 2;
        this.ctx.beginPath();
        this.ctx.moveTo(this.options.padding, centerY);
        this.ctx.lineTo(this.canvas.width - this.options.padding, centerY);
        this.ctx.stroke();
    }
    
    drawWaveformPlaceholder() {
        // Draw a simple sine wave pattern as placeholder for audio waveform
        this.ctx.strokeStyle = '#dee2e6';
        this.ctx.lineWidth = 2;
        this.ctx.beginPath();
        
        const centerY = this.canvas.height / 2;
        const amplitude = 15;
        const frequency = 0.02;
        
        for (let x = this.options.padding; x < this.canvas.width - this.options.padding; x++) {
            const y = centerY + Math.sin((x - this.options.padding) * frequency) * amplitude;
            if (x === this.options.padding) {
                this.ctx.moveTo(x, y);
            } else {
                this.ctx.lineTo(x, y);
            }
        }
        
        this.ctx.stroke();
    }
    
    addClip(clipData) {
        console.log('Timeline.addClip called with:', clipData);
        const clipElement = this.createClipElement(clipData);
        this.clipsContainer.appendChild(clipElement);
        this.clips.set(clipData.clip_id, { ...clipData, element: clipElement });
        this.updateClipPosition(clipData.clip_id);
        
        console.log(`Added clip ${clipData.clip_id} at ${clipData.start_time}s`);
        
        return clipData.clip_id;
    }
    
    createClipElement(clipData) {
        const clipElement = document.createElement('div');
        clipElement.className = `timeline-clip severity-${clipData.severity_level || 3}`;
        clipElement.dataset.clipId = clipData.clip_id;
        
        // Clip content
        const duration = clipData.duration || 10;
        clipElement.innerHTML = `
            <div class="clip-duration">${duration}s</div>
        `;
        
        // Add tooltip
        clipElement.title = `Click to preview clip (${duration}s)`;
        
        // Set up drag and drop
        clipElement.draggable = true;
        clipElement.addEventListener('dragstart', this.handleClipDragStart.bind(this));
        clipElement.addEventListener('drag', this.handleClipDrag.bind(this));
        clipElement.addEventListener('dragend', this.handleClipDragEnd.bind(this));
        
        // Context menu for deletion
        clipElement.addEventListener('contextmenu', (event) => {
            event.preventDefault();
            this.showClipContextMenu(clipData.clip_id, event);
        });
        
        // Click to jump to clip
        clipElement.addEventListener('click', (event) => {
            event.stopPropagation();
            this.jumpToClip(clipData.clip_id);
        });
        
        return clipElement;
    }
    
    handleClipDragStart(event) {
        const clipId = event.target.dataset.clipId;
        const clipData = this.clips.get(clipId);
        
        if (!clipData) return;
        
        this.dragging = clipId;
        event.target.classList.add('dragging');
        
        // Store offset from mouse to clip start
        const rect = event.target.getBoundingClientRect();
        const timelineRect = this.canvas.getBoundingClientRect();
        this.dragOffset.x = event.clientX - rect.left;
        
        // Set drag effect
        event.dataTransfer.effectAllowed = 'move';
        event.dataTransfer.setData('text/plain', clipId);
        
        // Enter preview mode
        this.enterPreviewMode();
        
        console.log(`Started dragging clip ${clipId}`);
    }
    
    handleClipDrag(event) {
        if (!this.dragging || event.clientX === 0) return; // Skip invalid drag events
        
        const timelineRect = this.canvas.getBoundingClientRect();
        const mouseX = event.clientX - timelineRect.left - this.dragOffset.x;
        const newTime = this.pixelToTime(mouseX);
        
        // Constrain to valid time range
        const clipData = this.clips.get(this.dragging);
        const maxTime = this.videoDuration - (clipData.duration || 10);
        const constrainedTime = Math.max(0, Math.min(newTime, maxTime));
        
        // Update clip position visually
        const clipElement = clipData.element;
        const newX = this.timeToPixel(constrainedTime);
        clipElement.style.left = newX + 'px';
        
        // Update video preview
        this.updateVideoPreview(constrainedTime);
        
        // Update temporary clip data
        clipData.start_time = constrainedTime;
        clipData.end_time = constrainedTime + (clipData.duration || 10);
    }
    
    handleClipDragEnd(event) {
        if (!this.dragging) return;
        
        const clipElement = event.target;
        clipElement.classList.remove('dragging');
        
        // Get final position
        const clipData = this.clips.get(this.dragging);
        const finalTime = clipData.start_time;
        
        // Exit preview mode
        this.exitPreviewMode();
        
        // Notify of clip position change
        this.onClipMoved(this.dragging, finalTime);
        
        this.dragging = null;
        console.log(`Finished dragging clip ${this.dragging} to ${finalTime.toFixed(1)}s`);
    }
    
    enterPreviewMode() {
        this.isPreviewMode = true;
        this.originalVideoTime = this.videoPlayer.currentTime();
        
        // Show preview overlay
        const overlay = document.getElementById('preview-overlay');
        if (overlay) {
            overlay.style.display = 'block';
        }
        
        console.log('Entered preview mode');
    }
    
    exitPreviewMode() {
        this.isPreviewMode = false;
        
        // Hide preview overlay
        const overlay = document.getElementById('preview-overlay');
        if (overlay) {
            overlay.style.display = 'none';
        }
        
        console.log('Exited preview mode');
    }
    
    updateVideoPreview(time) {
        if (!this.isPreviewMode) return;
        
        // Seek video to preview time
        this.videoPlayer.currentTime(time);
        
        // Update preview overlay text
        const previewTime = document.getElementById('preview-time');
        if (previewTime) {
            previewTime.textContent = `Preview: ${this.formatTime(time)}`;
        }
    }
    
    updateClipPosition(clipId) {
        const clipData = this.clips.get(clipId);
        if (!clipData) return;
        
        const x = this.timeToPixel(clipData.start_time);
        const duration = clipData.duration || 10;
        const calculatedWidth = duration * this.options.pixelsPerSecond;
        const width = Math.max(this.options.clipMinWidth, calculatedWidth);
        
        console.log(`Clip ${clipId}: duration=${duration}s, pixelsPerSecond=${this.options.pixelsPerSecond}, calculatedWidth=${calculatedWidth}px, finalWidth=${width}px`);
        
        const clipElement = clipData.element;
        clipElement.style.left = x + 'px';
        clipElement.style.width = width + 'px';
        clipElement.style.height = this.options.clipHeight + 'px';
        clipElement.style.top = this.options.padding + 'px';
    }
    
    updateClipPositions() {
        this.clips.forEach((clipData, clipId) => {
            this.updateClipPosition(clipId);
        });
    }
    
    removeClip(clipId) {
        const clipData = this.clips.get(clipId);
        if (!clipData) return false;
        
        // Remove DOM element
        if (clipData.element && clipData.element.parentNode) {
            clipData.element.parentNode.removeChild(clipData.element);
        }
        
        // Remove from clips map
        this.clips.delete(clipId);
        
        console.log(`Removed clip ${clipId}`);
        return true;
    }
    
    jumpToClip(clipId) {
        const clipData = this.clips.get(clipId);
        if (!clipData) return;
        
        // Seek video to clip start time
        this.videoPlayer.currentTime(clipData.start_time);
        
        // Play the clip and pause at the end
        this.videoPlayer.play();
        
        // Set up pause at clip end
        const pauseAtEnd = () => {
            if (this.videoPlayer.currentTime() >= clipData.end_time) {
                this.videoPlayer.pause();
                this.videoPlayer.off('timeupdate', pauseAtEnd);
            }
        };
        
        this.videoPlayer.on('timeupdate', pauseAtEnd);
        
        console.log(`Playing clip ${clipId} from ${clipData.start_time}s to ${clipData.end_time}s`);
    }
    
    showClipContextMenu(clipId, event) {
        // Create simple context menu for clip deletion
        const existingMenu = document.querySelector('.clip-context-menu');
        if (existingMenu) {
            existingMenu.remove();
        }
        
        const menu = document.createElement('div');
        menu.className = 'clip-context-menu';
        menu.style.position = 'fixed';
        menu.style.left = event.clientX + 'px';
        menu.style.top = event.clientY + 'px';
        menu.style.background = 'white';
        menu.style.border = '1px solid #ccc';
        menu.style.borderRadius = '4px';
        menu.style.padding = '0.5rem';
        menu.style.boxShadow = '0 2px 8px rgba(0,0,0,0.2)';
        menu.style.zIndex = '1000';
        
        const deleteOption = document.createElement('button');
        deleteOption.textContent = 'Delete Clip';
        deleteOption.className = 'btn-danger btn-small';
        deleteOption.onclick = () => {
            this.onClipDeleted(clipId);
            menu.remove();
        };
        
        menu.appendChild(deleteOption);
        document.body.appendChild(menu);
        
        // Remove menu when clicking elsewhere
        setTimeout(() => {
            const removeMenu = () => {
                menu.remove();
                document.removeEventListener('click', removeMenu);
            };
            document.addEventListener('click', removeMenu);
        }, 100);
    }
    
    formatTime(seconds) {
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    
    // Event callbacks (to be overridden by parent)
    onClipMoved(clipId, newTime) {
        console.log(`Clip ${clipId} moved to ${newTime}s`);
        // Override this method to handle clip position changes
    }
    
    onClipDeleted(clipId) {
        console.log(`Clip ${clipId} deletion requested`);
        // Override this method to handle clip deletion
    }
    
    // Public API methods
    getAllClips() {
        const clips = [];
        this.clips.forEach((clipData, clipId) => {
            clips.push({
                clip_id: clipId,
                start_time: clipData.start_time,
                end_time: clipData.end_time,
                duration: clipData.duration
            });
        });
        return clips.sort((a, b) => a.start_time - b.start_time);
    }
    
    clearAllClips() {
        console.log(`Timeline clearAllClips called - currently have ${this.clips.size} clips`);
        
        let removedCount = 0;
        this.clips.forEach((clipData, clipId) => {
            console.log(`Removing clip ${clipId} from timeline`);
            if (clipData.element && clipData.element.parentNode) {
                clipData.element.parentNode.removeChild(clipData.element);
                removedCount++;
            }
        });
        
        this.clips.clear();
        console.log(`Timeline cleared: removed ${removedCount} DOM elements, clips.size is now ${this.clips.size}`);
        
        // Force check for any remaining clip elements in the DOM
        const remainingClips = document.querySelectorAll('.timeline-clip');
        if (remainingClips.length > 0) {
            console.warn(`WARNING: ${remainingClips.length} clip elements still in DOM after clearing!`);
            remainingClips.forEach((element, index) => {
                console.log(`Orphaned clip ${index}:`, element);
                element.remove(); // Force remove any stragglers
            });
        }
    }
    
    hasOverlappingClips(startTime, duration, excludeClipId = null) {
        const endTime = startTime + duration;
        
        for (const [clipId, clipData] of this.clips) {
            if (clipId === excludeClipId) continue;
            
            const clipEnd = clipData.start_time + (clipData.duration || 10);
            
            // Check for overlap
            if (startTime < clipEnd && endTime > clipData.start_time) {
                return true;
            }
        }
        
        return false;
    }
    
    getClipAtTime(time) {
        for (const [clipId, clipData] of this.clips) {
            const clipEnd = clipData.start_time + (clipData.duration || 10);
            if (time >= clipData.start_time && time <= clipEnd) {
                return clipId;
            }
        }
        return null;
    }
}