# DriveGuard Ground Truth Annotation Tool

A web-based annotation tool for creating ground truth data for dashcam video safety evaluation. This application provides a comprehensive 5-step pipeline for annotating driving scenarios and generating safety assessments.

## Overview

The Ground Truth Annotation Tool is part of the DriveGuard evaluation framework, designed to create high-quality ground truth annotations for dashcam videos. It integrates real AI agents from the DriveGuard pipeline to generate, analyze, and assess driving scenarios.

### Key Features

- 🎥 **Video Analysis**: Frame-by-frame annotation using DriveGuard's VideoAnnotator AI
- 📝 **Scene Extraction**: Automatic breakdown into atomic scene descriptions
- ⚠️ **Violation Detection**: AI-powered traffic rule violation analysis
- 🚨 **Accident Analysis**: Risk assessment using historical accident data
- 📊 **Safety Assessment**: Comprehensive evaluation with DrivingMentor AI
- 💾 **Data Persistence**: Automatic saving and session management
- 📤 **Export Functionality**: Ground truth data export in JSON format

## Architecture

### 5-Step Annotation Pipeline

```
Video Input → 1. Annotation → 2. Scene Extraction → 3. Violation Analysis → 4. Accident Analysis → 5. Safety Assessment
```

1. **Generate Annotation**: VideoAnnotator AI creates narrative descriptions from video frames
2. **Extract Scenes**: Break down annotations into atomic scene descriptions
3. **Analyze Violations**: TrafficRuleChecker identifies rule violations using Milvus vector database
4. **Analyze Accidents**: AccidentRetriever assesses risks using Neo4j graph database
5. **Generate Assessment**: DrivingMentor creates comprehensive safety evaluation

### Technology Stack

**Backend:**
- **FastAPI**: High-performance async web framework
- **DriveGuard AI Agents**: VideoAnnotator, TrafficRuleChecker, AccidentRetriever, DrivingMentor
- **LLM Integration**: OpenAI GPT-4o (multimodal), GPT-4.1 (reasoning)
- **Databases**: Milvus (vector search), Neo4j (graph queries)
- **Python 3.12+**: Modern Python with async/await support

**Frontend:**
- **Vanilla JavaScript**: Lightweight, no framework dependencies
- **Video.js**: Professional video player component
- **CSS Grid/Flexbox**: Modern responsive layout
- **WebAPI**: Fetch-based communication with backend

## Quick Start

### Prerequisites

- Python 3.12+
- uv package manager
- OpenAI API key
- Neo4j database (for accident analysis)
- Milvus database (for violation detection)

### Installation

1. **Clone and navigate to the application:**
   ```bash
   cd evaluation/make_dataset/s4_annotate_ground_truth
   ```

2. **Install dependencies:**
   ```bash
   uv sync
   ```

3. **Configure environment:**
   ```bash
   # Create config/.env file with:
   OPENAI_API_KEY=your_openai_key
   NEO4J_URI=your_neo4j_uri
   NEO4J_USERNAME=your_username
   NEO4J_PASSWORD=your_password
   MILVUS_HOST=your_milvus_host
   MILVUS_PORT=19530
   ```

4. **Prepare video data:**
   ```bash
   # Place video clips in data/clips/
   # Videos should follow naming: NNNN_category_description_videoid_time.mp4
   ```

### Running the Application

**Start the web server:**
```bash
uv run python -m evaluation.make_dataset.s4_annotate_ground_truth.app
```

**Access the application:**
```
http://localhost:8000
```

**For background operation:**
```bash
nohup uv run python -m evaluation.make_dataset.s4_annotate_ground_truth.app > server.log 2>&1 &
```

## Usage Guide

### Basic Workflow

1. **Select Video**: Use Previous/Next buttons to navigate through available clips
2. **Generate Annotation**: Click "Generate with AI" to create narrative description
3. **Extract Scenes**: Automatically break down annotation into atomic scenes
4. **Analyze Violations**: AI identifies traffic rule violations in each scene  
5. **Analyze Accidents**: AI assesses accident risks and consequences
6. **Generate Assessment**: Create comprehensive safety evaluation with scores

### Navigation

- **Previous/Next Clip**: Navigate sequentially through video clips
- **Session Management**: Automatic progress tracking and resume capability
- **Refresh**: Reload current data and session state

### Data Management

- **Automatic Saving**: All changes saved immediately to ground truth files
- **Export Functionality**: Download complete ground truth data as JSON
- **Session Persistence**: Work resumption across browser sessions

## API Reference

### Core Endpoints

```http
GET  /                                    # Main application interface
GET  /api/health                          # Health check
GET  /api/clips                           # List all video clips
GET  /api/clips/{video_id}                # Get specific clip info
GET  /api/clips/{video_id}/stream         # Stream video content
GET  /api/clips/{video_id}/ground-truth   # Get ground truth data
```

### Pipeline Endpoints

```http
POST /api/clips/{video_id}/step/1/generate    # Generate annotation
POST /api/clips/{video_id}/step/1/optimize    # Optimize annotation
POST /api/clips/{video_id}/step/2/extract     # Extract scenes
POST /api/clips/{video_id}/step/3/analyze     # Analyze violations
POST /api/clips/{video_id}/step/4/analyze     # Analyze accidents
POST /api/clips/{video_id}/step/5/generate    # Generate assessment
PUT  /api/clips/{video_id}/step/{step}        # Save step data
```

### Session Management

```http
GET  /api/session/current                     # Current session info
POST /api/session/navigate                    # Navigate to clip/step
GET  /api/session/statistics                  # Session statistics
GET  /api/session/clip-status                 # Clip processing status
```

### Data Export

```http
POST /api/clips/{video_id}/export             # Create export
GET  /api/clips/{video_id}/download-export    # Download export file
POST /api/backup                              # Create data backup
```

## Data Format

### Ground Truth Structure

```json
{
  "video_id": "0001_example_video",
  "video_path": "/path/to/video.mp4",
  "ground_truth": {
    "annotation": "Narrative description...",
    "scenes": [
      "Scene 1 description",
      "Scene 2 description"
    ],
    "violations": [
      {
        "scene": "Scene description",
        "violation": "found|not_found",
        "reason": "Explanation"
      }
    ],
    "accidents": [
      {
        "scene": "Scene description", 
        "accident": "found|not_found",
        "consequence": "Risk description"
      }
    ],
    "assessment": {
      "safety_score": 7,
      "risk_level": "medium",
      "overall_evaluation": "Assessment text",
      "strengths": ["Positive behavior 1", "Positive behavior 2"],
      "weaknesses": ["Issue 1", "Issue 2"],
      "improvement_advice": ["Advice 1", "Advice 2"]
    }
  },
  "evaluation_criteria": {
    "annotation_quality": "How accurate is the scene description?",
    "scene_extraction": "Are all important scenes identified?",
    "violation_detection": "Are traffic violations correctly identified?",
    "accident_assessment": "Are accident risks properly evaluated?",
    "safety_scoring": "Is the safety score appropriate?",
    "advice_relevance": "Are recommendations actionable and relevant?"
  }
}
```

## File Structure

```
s4_annotate_ground_truth/
├── app.py                    # Main FastAPI application
├── main.py                   # Entry point
├── core/                     # Core system components
│   ├── config.py            # Configuration settings
│   ├── models.py            # Data models
│   ├── session_manager.py   # Session handling
│   ├── progress_tracker.py  # Progress tracking
│   └── video_manager.py     # Video management
├── managers/                 # Business logic managers
│   ├── ground_truth_manager.py    # Ground truth data management
│   ├── pipeline_orchestrator.py   # Pipeline coordination
│   ├── progress_tracker.py        # Progress management
│   ├── session_manager.py         # Session management
│   └── video_discovery.py         # Video discovery logic
├── pipeline/                 # AI pipeline components
│   ├── annotation_generator.py    # Step 1: Video annotation
│   ├── scene_extractor.py         # Step 2: Scene extraction
│   ├── violation_analyzer.py      # Step 3: Violation analysis
│   ├── accident_analyzer.py       # Step 4: Accident analysis
│   └── assessment_generator.py    # Step 5: Safety assessment
├── static/                   # Frontend assets
│   ├── index.html           # Main UI
│   ├── css/                 # Stylesheets
│   └── js/                  # JavaScript components
└── data/                    # Application data
    ├── clips/               # Video files
    ├── ground_truth/        # Annotation files
    └── sessions/            # Session data
```

## Configuration

### Environment Variables

```bash
# OpenAI API
OPENAI_API_KEY=your_openai_api_key

# Neo4j Database (for accident analysis)
NEO4J_URI=bolt://localhost:7687
NEO4J_USERNAME=neo4j
NEO4J_PASSWORD=your_password

# Milvus Database (for violation detection)  
MILVUS_HOST=localhost
MILVUS_PORT=19530

# Application Settings (optional)
GT_ANNOTATION_ENVIRONMENT=production
GT_ANNOTATION_DEBUG=false
```

### Application Settings

Key settings in `core/config.py`:

- `video_directory`: Location of video clips (default: `./data/clips`)
- `ground_truth_directory`: Ground truth storage (default: `./data/ground_truth`)
- `session_directory`: Session data (default: `./data/sessions`)
- `max_video_size_mb`: Maximum video size (default: 500MB)
- `supported_formats`: Supported video formats (mp4, avi, mov, webm)

## AI Agents Integration

### VideoAnnotator (Step 1)
- **Model**: OpenAI GPT-4o (multimodal)
- **Function**: Frame-by-frame video analysis
- **Output**: Narrative driving scenario descriptions

### TrafficRuleChecker (Step 3)  
- **Model**: OpenAI GPT-4.1 + Milvus vector database
- **Function**: Traffic rule violation detection
- **Database**: Similarity search against traffic rule embeddings

### AccidentRetriever (Step 4)
- **Model**: OpenAI GPT-4.1 + Neo4j graph database  
- **Function**: Accident risk assessment
- **Database**: Graph queries against historical accident scenarios

### DrivingMentor (Step 5)
- **Model**: Multi-agent workflow with GPT-4.1
- **Function**: Comprehensive safety assessment
- **Output**: Structured evaluation with scores and recommendations

## Troubleshooting

### Common Issues

**Application won't start:**
- Check Python version (3.12+ required)
- Verify all dependencies installed: `uv sync`
- Check environment configuration in `config/.env`

**AI agents not working:**
- Verify OpenAI API key is valid and has credits
- Check database connections (Neo4j, Milvus)
- Review server logs for specific error messages

**Videos not loading:**
- Ensure videos are in `data/clips/` directory
- Check file permissions and supported formats
- Verify file naming follows convention: `NNNN_category_description.mp4`

**Frontend not updating:**
- Hard refresh browser (Ctrl+F5 or Cmd+Shift+R)
- Check browser developer tools for errors
- Verify static files are being served correctly

### Logging

Application logs are written to:
- Console output (when running in foreground)
- `server.log` (when running with nohup)

Log levels can be configured in `core/config.py`.

## Development

### Architecture Principles

- **Production-Ready**: No test code, mock implementations, or debug artifacts
- **Fail-Fast**: Clear error reporting when dependencies unavailable
- **Real AI Integration**: Uses actual DriveGuard pipeline agents
- **Stateless Components**: Easy to understand and maintain
- **Responsive Design**: Works across desktop and tablet devices

### Code Organization

- **Separation of Concerns**: Clear boundaries between UI, API, and AI components
- **Async/Await**: Non-blocking operations for better performance
- **Error Handling**: Comprehensive error reporting and recovery
- **Data Validation**: Pydantic models for type safety
- **Clean APIs**: RESTful endpoints with consistent response formats

## License

Part of the DriveGuard project. See main project license for details.

## Support

For issues and questions:
1. Check the troubleshooting section above
2. Review server logs for error details
3. Verify environment configuration
4. Ensure all prerequisites are properly installed

---

**Generated with Claude Code**