"""
Pydantic data models for the Ground Truth Annotation application
"""

from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Any, Union
from datetime import datetime
from enum import Enum
from pathlib import Path


class StepStatus(str, Enum):
    """Step completion status"""
    PENDING = "pending"
    IN_PROGRESS = "in_progress"
    COMPLETED = "completed"
    ERROR = "error"


class SafetyRating(str, Enum):
    """Safety rating levels"""
    EXCELLENT = "excellent"
    GOOD = "good"
    FAIR = "fair"
    POOR = "poor"
    DANGEROUS = "dangerous"


class ViolationSeverity(str, Enum):
    """Traffic violation severity levels"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


class RiskLevel(str, Enum):
    """Accident risk levels"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


# Core data models
class ClipInfo(BaseModel):
    """Information about a video clip"""
    video_id: str
    video_path: str
    category: Optional[str] = None
    duration: Optional[float] = None
    sequence_number: Optional[int] = None
    metadata_source: Optional[str] = "extraction"
    
    @validator('video_path')
    def validate_video_path(cls, v):
        if not Path(v).exists():
            raise ValueError(f"Video file does not exist: {v}")
        return v


class Scene(BaseModel):
    """A scene extracted from video annotation"""
    description: str = Field(..., min_length=10)
    start_time: float = Field(..., ge=0)
    end_time: float = Field(..., ge=0)
    scene_type: str = "general"
    
    @validator('end_time')
    def validate_time_order(cls, v, values):
        if 'start_time' in values and v <= values['start_time']:
            raise ValueError('End time must be greater than start time')
        return v


class TrafficViolation(BaseModel):
    """A traffic violation detected in the video"""
    violation_type: str = Field(..., min_length=1)
    description: str = Field(..., min_length=10)
    severity: ViolationSeverity = ViolationSeverity.MEDIUM
    timestamp: float = Field(..., ge=0)
    confidence_score: float = Field(0.5, ge=0.0, le=1.0)


class AccidentScenario(BaseModel):
    """An accident scenario identified in the video"""
    accident_type: str = Field(..., min_length=1)
    description: str = Field(..., min_length=10)
    risk_level: RiskLevel = RiskLevel.MEDIUM
    timestamp: float = Field(..., ge=0)
    likelihood_score: float = Field(0.5, ge=0.0, le=1.0)
    severity_score: float = Field(0.5, ge=0.0, le=1.0)
    contributing_factors: List[str] = []


class DrivingAssessment(BaseModel):
    """Final driving assessment"""
    overall_score: int = Field(..., ge=0, le=100)
    safety_rating: SafetyRating = SafetyRating.GOOD
    feedback: str = Field(..., min_length=20)
    recommendations: str = Field(..., min_length=20)
    violations_summary: str = ""
    accidents_summary: str = ""


class GroundTruthData(BaseModel):
    """Complete ground truth data for a video clip"""
    video_id: str
    annotation: Optional[str] = None
    scenes: List[Scene] = []
    violations: List[TrafficViolation] = []
    accidents: List[AccidentScenario] = []
    assessment: Optional[DrivingAssessment] = None
    created_at: datetime = Field(default_factory=datetime.now)
    updated_at: datetime = Field(default_factory=datetime.now)


# Progress tracking models
class StepProgress(BaseModel):
    """Progress information for a pipeline step"""
    step: int = Field(..., ge=1, le=5)
    status: StepStatus = StepStatus.PENDING
    completed_at: Optional[datetime] = None
    error_message: Optional[str] = None


class ClipProgress(BaseModel):
    """Progress tracking for a video clip"""
    video_id: str
    steps_completed: Dict[str, StepProgress] = {}
    total_steps: int = 5
    completed_steps: int = 0
    is_complete: bool = False
    current_step: Optional[int] = None
    last_updated: datetime = Field(default_factory=datetime.now)


# Session management models
class SessionInfo(BaseModel):
    """Session information"""
    session_id: str
    started_at: datetime = Field(default_factory=datetime.now)
    current_video_id: Optional[str] = None
    current_step: int = 1
    total_clips: int = 0
    clips_completed: int = 0
    last_activity: datetime = Field(default_factory=datetime.now)


# API response models
class APIResponse(BaseModel):
    """Base API response"""
    success: bool = True
    error: Optional[str] = None


class SessionResponse(APIResponse):
    """Session API response"""
    session: Optional[SessionInfo] = None


class ClipListResponse(APIResponse):
    """Clip list API response"""
    clips: List[ClipInfo] = []


class GroundTruthResponse(APIResponse):
    """Ground truth API response"""
    data: Optional[GroundTruthData] = None


# Step-specific data models for API
class AnnotationData(BaseModel):
    """Annotation step data"""
    annotation: str = Field(..., min_length=50)


class ScenesData(BaseModel):
    """Scenes step data"""
    scenes: List[Scene] = []


class ViolationsData(BaseModel):
    """Violations step data"""
    violations: List[TrafficViolation] = []


class AccidentsData(BaseModel):
    """Accidents step data"""
    accidents: List[AccidentScenario] = []


class AssessmentData(BaseModel):
    """Assessment step data"""
    assessment: DrivingAssessment