"""
Video discovery and management
"""

import asyncio
import json
from pathlib import Path
from typing import List, Optional, Dict, Any
import logging

from .models import ClipInfo

logger = logging.getLogger(__name__)


class VideoManager:
    """Manages video clip discovery and metadata"""
    
    def __init__(self, video_directory: Path):
        self.video_directory = Path(video_directory)
        self.metadata_cache: Dict[str, ClipInfo] = {}
        self._clips: List[ClipInfo] = []
        
    async def get_clips(self) -> List[ClipInfo]:
        """Get all available video clips, sorted by sequence number"""
        if not self._clips:
            await self.discover_clips()
        
        # Sort clips by sequence number, then by video_id
        # Clips without sequence numbers go to the end (using 9999 as fallback)
        sorted_clips = sorted(self._clips, key=lambda x: (x.sequence_number or 9999, x.video_id))
        
        return sorted_clips
    
    async def get_clip_by_id(self, video_id: str) -> Optional[ClipInfo]:
        """Get a specific clip by ID"""
        clips = await self.get_clips()
        for clip in clips:
            if clip.video_id == video_id:
                return clip
        return None
    
    async def discover_clips(self) -> None:
        """Discover video clips in the directory"""
        try:
            self._clips = []
            
            if not self.video_directory.exists():
                logger.warning(f"Video directory does not exist: {self.video_directory}")
                return
            
            # Check for metadata files first
            metadata_dir = self.video_directory / "metadata"
            if metadata_dir.exists():
                await self._load_from_metadata(metadata_dir)
            else:
                # Fallback to direct file discovery
                await self._discover_video_files()
            
            logger.info(f"Discovered {len(self._clips)} video clips")
            
        except Exception as e:
            logger.error(f"Failed to discover clips: {e}")
            raise
    
    async def _load_from_metadata(self, metadata_dir: Path) -> None:
        """Load clips from metadata files"""
        try:
            # Look for individual metadata files
            for metadata_file in metadata_dir.glob("*_clips_metadata.json"):
                await self._load_metadata_file(metadata_file)
            
            # Also check for general extraction metadata
            extraction_meta = metadata_dir / "extraction_metadata.json"
            if extraction_meta.exists():
                await self._load_extraction_metadata(extraction_meta)
                
        except Exception as e:
            logger.error(f"Failed to load metadata: {e}")
            # Fallback to file discovery
            await self._discover_video_files()
    
    async def _load_metadata_file(self, metadata_file: Path) -> None:
        """Load clips from a specific metadata file"""
        try:
            with open(metadata_file, 'r') as f:
                metadata = json.load(f)
            
            # Handle different metadata formats
            clips_data = metadata.get('clips', [])
            if not clips_data and 'extracted_clips' in metadata:
                clips_data = metadata['extracted_clips']
            
            for clip_data in clips_data:
                clip_info = self._create_clip_info(clip_data)
                if clip_info:
                    self._clips.append(clip_info)
                    
        except Exception as e:
            logger.error(f"Failed to load metadata file {metadata_file}: {e}")
    
    async def _load_extraction_metadata(self, extraction_file: Path) -> None:
        """Load clips from extraction metadata"""
        try:
            with open(extraction_file, 'r') as f:
                metadata = json.load(f)
            
            # Extract clips from various possible structures
            clips_data = []
            
            if 'videos' in metadata:
                for video_id, video_data in metadata['videos'].items():
                    if 'clips' in video_data:
                        clips_data.extend(video_data['clips'])
            
            for clip_data in clips_data:
                clip_info = self._create_clip_info(clip_data)
                if clip_info:
                    self._clips.append(clip_info)
                    
        except Exception as e:
            logger.error(f"Failed to load extraction metadata: {e}")
    
    def _create_clip_info(self, clip_data: Dict[str, Any]) -> Optional[ClipInfo]:
        """Create ClipInfo from metadata"""
        try:
            # Handle different metadata formats
            video_id = clip_data.get('video_id') or clip_data.get('clip_id')
            video_path = clip_data.get('video_path') or clip_data.get('file_path') or clip_data.get('output_path')
            
            # Use filename as video_id if available (cleaner ID)
            if 'filename' in clip_data:
                video_id = Path(clip_data['filename']).stem
            
            if not video_id or not video_path:
                return None
            
            # Ensure video path is absolute and exists
            if not Path(video_path).is_absolute():
                video_path = self.video_directory / video_path
            
            if not Path(video_path).exists():
                logger.warning(f"Video file not found: {video_path}")
                return None
            
            # Extract category from clip_data if available, otherwise infer from video_id
            category = clip_data.get('category')
            if not category and 'clip_data' in clip_data and isinstance(clip_data['clip_data'], dict):
                category = clip_data['clip_data'].get('video_category')
            if not category:
                category = self._infer_category(video_id)
            
            # Get duration from clip_data if available
            duration = clip_data.get('duration')
            if not duration and 'clip_data' in clip_data and isinstance(clip_data['clip_data'], dict):
                duration = clip_data['clip_data'].get('duration')
            
            # Extract sequence number - first try metadata, then extract from video_id
            sequence_number = clip_data.get('sequence_number')
            if sequence_number is None:
                sequence_number = self._extract_sequence_number(video_id)
            
            return ClipInfo(
                video_id=video_id,
                video_path=str(video_path),
                category=category,
                duration=duration,
                sequence_number=sequence_number,
                metadata_source=clip_data.get('source', 'metadata')
            )
            
        except Exception as e:
            logger.error(f"Failed to create clip info: {e}")
            return None
    
    async def _discover_video_files(self) -> None:
        """Fallback method to discover video files directly"""
        try:
            video_extensions = {'.mp4', '.avi', '.mov', '.webm'}
            
            for video_file in self.video_directory.rglob('*'):
                if video_file.is_file() and video_file.suffix.lower() in video_extensions:
                    video_id = video_file.stem
                    
                    clip_info = ClipInfo(
                        video_id=video_id,
                        video_path=str(video_file),
                        category=self._infer_category(video_id),
                        sequence_number=self._extract_sequence_number(video_id),
                        metadata_source="file_discovery"
                    )
                    
                    self._clips.append(clip_info)
                    
        except Exception as e:
            logger.error(f"Failed to discover video files: {e}")
    
    def _infer_category(self, video_id: str) -> str:
        """Infer category from video ID"""
        video_id_lower = video_id.lower()
        
        if 'accident' in video_id_lower:
            return 'accident'
        elif 'turn' in video_id_lower:
            return 'turning'
        elif 'lane' in video_id_lower:
            return 'lane_change'
        elif 'speed' in video_id_lower:
            return 'speeding'
        elif 'light' in video_id_lower or 'signal' in video_id_lower:
            return 'traffic_signals'
        elif 'intersection' in video_id_lower:
            return 'intersection'
        else:
            return 'general'

    def _extract_sequence_number(self, video_id: str) -> Optional[int]:
        """Extract sequence number from video ID.
        
        Args:
            video_id (str): Video ID to parse.
            
        Returns:
            Optional[int]: Sequence number if found, None otherwise.
        """
        try:
            # Look for pattern like "0000_..." at the beginning
            if video_id and len(video_id) >= 4 and video_id[:4].isdigit():
                return int(video_id[:4])
        except (ValueError, IndexError):
            pass
        return None
    
    async def refresh(self) -> None:
        """Refresh the clip list"""
        self._clips = []
        self.metadata_cache = {}
        await self.discover_clips()
    
    def get_stats(self) -> Dict[str, Any]:
        """Get statistics about discovered clips"""
        categories = {}
        for clip in self._clips:
            cat = clip.category or 'unknown'
            categories[cat] = categories.get(cat, 0) + 1
        
        return {
            'total_clips': len(self._clips),
            'categories': categories,
            'has_metadata': len([c for c in self._clips if c.metadata_source != 'file_discovery'])
        }