"""Progress Tracker for detailed progress tracking per clip and step.

This manager handles detailed progress persistence and recovery for individual clips.
"""

import json
import logging
from pathlib import Path
from typing import Dict, List, Optional, Any
from datetime import datetime

# Import models from parent module
import sys
sys.path.append(str(Path(__file__).parent.parent))
from models import ClipProgress, StepProgress, GlobalStatistics, StepStatistics

logger = logging.getLogger(__name__)


class ProgressTracker:
    """Detailed progress tracking for individual clips."""
    
    def __init__(self, session_dir: str = "data/evaluation/sessions"):
        """Initialize the progress tracker.
        
        Args:
            session_dir (str): Directory to store session data.
        """
        self.session_dir = Path(session_dir)
        self.clips_dir = self.session_dir / "clips"
        self.clips_dir.mkdir(parents=True, exist_ok=True)
        
        # Progress cache
        self._progress_cache: Dict[str, ClipProgress] = {}
    
    def load_clip_progress(self, video_id: str) -> ClipProgress:
        """Load progress for specific clip.
        
        Args:
            video_id (str): Video clip identifier.
            
        Returns:
            ClipProgress: Progress data for the clip.
        """
        # Check cache first
        if video_id in self._progress_cache:
            return self._progress_cache[video_id]
        
        progress_file = self.clips_dir / f"{video_id}_progress.json"
        
        if progress_file.exists():
            try:
                with open(progress_file, 'r', encoding='utf-8') as f:
                    progress_data = json.load(f)
                
                # Convert datetime strings back to datetime objects
                progress_data['started_at'] = datetime.fromisoformat(progress_data['started_at'])
                progress_data['last_updated'] = datetime.fromisoformat(progress_data['last_updated'])
                
                # Convert step progress
                steps = {}
                for step_num, step_data in progress_data.get('steps', {}).items():
                    if step_data.get('started_at'):
                        step_data['started_at'] = datetime.fromisoformat(step_data['started_at'])
                    if step_data.get('completed_at'):
                        step_data['completed_at'] = datetime.fromisoformat(step_data['completed_at'])
                    
                    steps[int(step_num)] = StepProgress(**step_data)
                
                progress_data['steps'] = steps
                clip_progress = ClipProgress(**progress_data)
                
                # Cache the result
                self._progress_cache[video_id] = clip_progress
                
                return clip_progress
                
            except Exception as e:
                logger.warning(f"Failed to load progress for {video_id}: {e}")
        
        # Create new progress if not found
        return self._create_new_progress(video_id)
    
    def _create_new_progress(self, video_id: str) -> ClipProgress:
        """Create new progress tracking for a clip.
        
        Args:
            video_id (str): Video clip identifier.
            
        Returns:
            ClipProgress: New progress instance.
        """
        now = datetime.now()
        
        clip_progress = ClipProgress(
            video_id=video_id,
            current_step=1,
            overall_status="pending",
            started_at=now,
            last_updated=now,
            steps={},
            total_time_spent=0.0
        )
        
        # Cache and save
        self._progress_cache[video_id] = clip_progress
        self._save_clip_progress(video_id, clip_progress)
        
        logger.info(f"Created new progress tracking for {video_id}")
        return clip_progress
    
    def _save_clip_progress(self, video_id: str, progress: ClipProgress) -> None:
        """Save clip progress to disk.
        
        Args:
            video_id (str): Video clip identifier.
            progress (ClipProgress): Progress data to save.
        """
        try:
            progress.last_updated = datetime.now()
            progress_file = self.clips_dir / f"{video_id}_progress.json"
            
            # Convert to dict for JSON serialization
            progress_dict = progress.dict()
            
            with open(progress_file, 'w', encoding='utf-8') as f:
                json.dump(progress_dict, f, indent=2, default=str, ensure_ascii=False)
                
        except Exception as e:
            logger.error(f"Failed to save progress for {video_id}: {e}")
    
    def start_step(self, video_id: str, step: int) -> None:
        """Mark step as started.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
        """
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}")
            return
        
        progress = self.load_clip_progress(video_id)
        
        # Create step progress if not exists
        if step not in progress.steps:
            step_names = {
                1: "annotation",
                2: "scenes", 
                3: "violations",
                4: "accidents",
                5: "assessment"
            }
            
            progress.steps[step] = StepProgress(
                step_number=step,
                step_name=step_names[step],
                status="pending"
            )
        
        # Mark step as in progress
        progress.steps[step].status = "in_progress"
        progress.steps[step].started_at = datetime.now()
        progress.current_step = step
        progress.overall_status = "in_progress"
        
        # Update cache and save
        self._progress_cache[video_id] = progress
        self._save_clip_progress(video_id, progress)
        
        logger.info(f"Started step {step} for {video_id}")
    
    def complete_step(self, video_id: str, step: int, data: Optional[Dict[str, Any]] = None) -> None:
        """Mark step as completed with data.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
            data (Dict[str, Any], optional): Step data to save.
        """
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}")
            return
        
        progress = self.load_clip_progress(video_id)
        
        # Ensure step exists
        if step not in progress.steps:
            self.start_step(video_id, step)
            progress = self.load_clip_progress(video_id)  # Reload after starting
        
        # Mark step as completed
        progress.steps[step].status = "completed"
        progress.steps[step].completed_at = datetime.now()
        if data:
            progress.steps[step].data = data
        
        # Check if all steps are completed
        completed_steps = sum(1 for s in progress.steps.values() if s.status == "completed")
        if completed_steps == 5:
            progress.overall_status = "completed"
        
        # Update cache and save
        self._progress_cache[video_id] = progress
        self._save_clip_progress(video_id, progress)
        
        logger.info(f"Completed step {step} for {video_id}")
    
    def save_step_data(self, video_id: str, step: int, data: Dict[str, Any]) -> None:
        """Save step data without completing the step.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
            data (Dict[str, Any]): Step data to save.
        """
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}")
            return
        
        progress = self.load_clip_progress(video_id)
        
        # Ensure step exists
        if step not in progress.steps:
            self.start_step(video_id, step)
            progress = self.load_clip_progress(video_id)
        
        # Save data and increment save count
        progress.steps[step].data = data
        progress.steps[step].save_count += 1
        
        # Update cache and save
        self._progress_cache[video_id] = progress
        self._save_clip_progress(video_id, progress)
        
    
    def mark_manual_edit(self, video_id: str, step: int) -> None:
        """Mark that user made manual edits to this step.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
        """
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}")
            return
        
        progress = self.load_clip_progress(video_id)
        
        if step in progress.steps:
            progress.steps[step].manual_edits = True
            
            # Update cache and save
            self._progress_cache[video_id] = progress
            self._save_clip_progress(video_id, progress)
            
            logger.info(f"Marked manual edit for step {step} of {video_id}")
    
    def increment_llm_generations(self, video_id: str, step: int) -> None:
        """Increment LLM generation count for a step.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
        """
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}")
            return
        
        progress = self.load_clip_progress(video_id)
        
        if step in progress.steps:
            progress.steps[step].llm_generations += 1
            
            # Update cache and save
            self._progress_cache[video_id] = progress
            self._save_clip_progress(video_id, progress)
    
    def is_step_completed(self, video_id: str, step: int) -> bool:
        """Check if specific step is completed.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
            
        Returns:
            bool: True if step is completed.
        """
        progress = self.load_clip_progress(video_id)
        return (step in progress.steps and 
                progress.steps[step].status == "completed")
    
    def is_clip_completed(self, video_id: str) -> bool:
        """Check if all 5 steps are completed.
        
        Args:
            video_id (str): Video clip identifier.
            
        Returns:
            bool: True if all steps are completed.
        """
        progress = self.load_clip_progress(video_id)
        return progress.overall_status == "completed"
    
    def get_step_status(self, video_id: str, step: int) -> str:
        """Get status of a specific step.
        
        Args:
            video_id (str): Video clip identifier.
            step (int): Step number (1-5).
            
        Returns:
            str: Step status (pending, in_progress, completed).
        """
        progress = self.load_clip_progress(video_id)
        if step in progress.steps:
            return progress.steps[step].status
        return "pending"
    
    def get_all_clip_progress(self) -> List[ClipProgress]:
        """Get progress for all clips.
        
        Returns:
            List[ClipProgress]: Progress data for all clips.
        """
        progress_files = list(self.clips_dir.glob("*_progress.json"))
        all_progress = []
        
        for progress_file in progress_files:
            try:
                video_id = progress_file.stem.replace("_progress", "")
                progress = self.load_clip_progress(video_id)
                all_progress.append(progress)
            except Exception as e:
                logger.warning(f"Failed to load progress from {progress_file}: {e}")
        
        return all_progress
    
    def get_global_statistics(self) -> GlobalStatistics:
        """Get global progress statistics.
        
        Returns:
            GlobalStatistics: Global application statistics.
        """
        all_progress = self.get_all_clip_progress()
        
        total_clips = len(all_progress)
        clips_completed = sum(1 for p in all_progress if p.overall_status == "completed")
        clips_in_progress = sum(1 for p in all_progress if p.overall_status == "in_progress")
        clips_pending = total_clips - clips_completed - clips_in_progress
        
        completion_percentage = (clips_completed / total_clips * 100) if total_clips > 0 else 0.0
        total_time_spent = sum(p.total_time_spent for p in all_progress)
        
        # Calculate step statistics
        step_stats = []
        step_names = ["annotation", "scenes", "violations", "accidents", "assessment"]
        
        for step_num in range(1, 6):
            step_name = step_names[step_num - 1]
            completed_count = 0
            in_progress_count = 0
            pending_count = 0
            total_time = 0.0
            manual_edit_count = 0
            
            for progress in all_progress:
                if step_num in progress.steps:
                    step = progress.steps[step_num]
                    if step.status == "completed":
                        completed_count += 1
                    elif step.status == "in_progress":
                        in_progress_count += 1
                    else:
                        pending_count += 1
                    
                    if step.manual_edits:
                        manual_edit_count += 1
                    
                    # Calculate time spent (rough estimate)
                    if step.started_at and step.completed_at:
                        duration = (step.completed_at - step.started_at).total_seconds() / 60.0
                        total_time += duration
                else:
                    pending_count += 1
            
            average_time = total_time / completed_count if completed_count > 0 else 0.0
            manual_edit_percentage = (manual_edit_count / total_clips * 100) if total_clips > 0 else 0.0
            
            step_stat = StepStatistics(
                step_number=step_num,
                step_name=step_name,
                completed_count=completed_count,
                in_progress_count=in_progress_count,
                pending_count=pending_count,
                average_time=average_time,
                manual_edit_percentage=manual_edit_percentage
            )
            step_stats.append(step_stat)
        
        return GlobalStatistics(
            total_clips=total_clips,
            clips_completed=clips_completed,
            clips_in_progress=clips_in_progress,
            clips_pending=clips_pending,
            completion_percentage=completion_percentage,
            total_time_spent=total_time_spent,
            step_statistics=step_stats,
            last_updated=datetime.now()
        )
    
    def reset_clip_progress(self, video_id: str) -> None:
        """Reset progress for a specific clip.
        
        Args:
            video_id (str): Video clip identifier.
        """
        progress_file = self.clips_dir / f"{video_id}_progress.json"
        
        # Remove from cache
        if video_id in self._progress_cache:
            del self._progress_cache[video_id]
        
        # Remove file
        if progress_file.exists():
            progress_file.unlink()
        
        logger.info(f"Reset progress for {video_id}")
    
    def create_backup(self, backup_name: Optional[str] = None) -> str:
        """Create backup of all progress data.
        
        Args:
            backup_name (str, optional): Custom backup name.
            
        Returns:
            str: Path to the backup file.
        """
        if not backup_name:
            backup_name = f"progress_backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        
        backup_dir = self.session_dir / "backups"
        backup_dir.mkdir(exist_ok=True)
        
        backup_file = backup_dir / f"{backup_name}.json"
        
        # Collect all progress data
        all_progress = {}
        for progress in self.get_all_clip_progress():
            all_progress[progress.video_id] = progress.dict()
        
        # Save backup
        backup_data = {
            'created_at': datetime.now().isoformat(),
            'clip_count': len(all_progress),
            'progress_data': all_progress
        }
        
        with open(backup_file, 'w', encoding='utf-8') as f:
            json.dump(backup_data, f, indent=2, default=str, ensure_ascii=False)
        
        logger.info(f"Created progress backup: {backup_file}")
        return str(backup_file)