"""Session Manager for tracking user session state and navigation.

This manager handles session persistence and navigation between clips and steps.
"""

import json
import uuid
import logging
from pathlib import Path
from typing import Optional, Tuple, Dict, Any
from datetime import datetime

# Import models from parent module
import sys
sys.path.append(str(Path(__file__).parent.parent))
from models import SessionState, VideoClip

logger = logging.getLogger(__name__)


class SessionManager:
    """Manages session state and navigation between clips/steps."""
    
    def __init__(self, session_dir: str = "data/evaluation/sessions"):
        """Initialize the session manager.
        
        Args:
            session_dir (str): Directory to store session data.
        """
        self.session_dir = Path(session_dir)
        self.session_dir.mkdir(parents=True, exist_ok=True)
        self.session_file = self.session_dir / "current_session.json"
        
        # Current session state
        self._current_session: Optional[SessionState] = None
        
        # Load existing session or create new one
        self._load_or_create_session()
    
    def _load_or_create_session(self) -> None:
        """Load existing session or create a new one."""
        if self.session_file.exists():
            try:
                with open(self.session_file, 'r', encoding='utf-8') as f:
                    session_data = json.load(f)
                
                # Convert ISO datetime strings back to datetime objects
                session_data['started_at'] = datetime.fromisoformat(session_data['started_at'])
                session_data['last_updated'] = datetime.fromisoformat(session_data['last_updated'])
                
                self._current_session = SessionState(**session_data)
                logger.info(f"Loaded existing session: {self._current_session.session_id}")
                
            except Exception as e:
                logger.warning(f"Failed to load existing session: {e}")
                self._create_new_session()
        else:
            self._create_new_session()
    
    def _create_new_session(self) -> None:
        """Create a new session."""
        session_id = str(uuid.uuid4())
        now = datetime.now()
        
        self._current_session = SessionState(
            session_id=session_id,
            started_at=now,
            current_video_id="",  # Will be set when first clip is loaded
            current_step=1,
            total_clips=0,
            clips_completed=0,
            clips_in_progress=0,
            clips_pending=0,
            last_updated=now
        )
        
        self._save_session()
        logger.info(f"Created new session: {session_id}")
    
    def _save_session(self) -> None:
        """Save current session to disk."""
        if not self._current_session:
            return
        
        try:
            self._current_session.last_updated = datetime.now()
            
            # Convert to dict for JSON serialization
            session_dict = self._current_session.dict()
            
            with open(self.session_file, 'w', encoding='utf-8') as f:
                json.dump(session_dict, f, indent=2, default=str, ensure_ascii=False)
                
        except Exception as e:
            logger.error(f"Failed to save session: {e}")
    
    def get_current_position(self) -> Tuple[str, int]:
        """Get current video_id and step number.
        
        Returns:
            Tuple[str, int]: Current video ID and step number.
        """
        if not self._current_session:
            return "", 1
        return self._current_session.current_video_id, self._current_session.current_step
    
    def set_position(self, video_id: str, step: int) -> None:
        """Set current position.
        
        Args:
            video_id (str): Video ID to navigate to.
            step (int): Step number to navigate to (1-5).
        """
        if not self._current_session:
            return
        
        # Validate step number
        if not (1 <= step <= 5):
            logger.warning(f"Invalid step number: {step}. Must be 1-5.")
            return
        
        self._current_session.current_video_id = video_id
        self._current_session.current_step = step
        self._save_session()
        
        logger.info(f"Navigated to {video_id}, step {step}")
    
    def advance_step(self, video_id: str) -> int:
        """Advance to next step for current clip.
        
        Args:
            video_id (str): Video ID to advance.
            
        Returns:
            int: New step number.
        """
        if not self._current_session:
            return 1
        
        current_step = self._current_session.current_step
        new_step = min(current_step + 1, 5)  # Cap at step 5
        
        if new_step != current_step:
            self.set_position(video_id, new_step)
            logger.info(f"Advanced {video_id} from step {current_step} to step {new_step}")
        
        return new_step
    
    def previous_step(self, video_id: str) -> int:
        """Go back to previous step for current clip.
        
        Args:
            video_id (str): Video ID to go back.
            
        Returns:
            int: New step number.
        """
        if not self._current_session:
            return 1
        
        current_step = self._current_session.current_step
        new_step = max(current_step - 1, 1)  # Cap at step 1
        
        if new_step != current_step:
            self.set_position(video_id, new_step)
            logger.info(f"Went back {video_id} from step {current_step} to step {new_step}")
        
        return new_step
    
    def get_next_clip(self, current_video_id: str, clips: list) -> Optional[str]:
        """Find next incomplete clip.
        
        Args:
            current_video_id (str): Current video ID.
            clips (list): List of available clips.
            
        Returns:
            Optional[str]: Next clip video ID, or None if no more clips.
        """
        if not clips:
            return None
        
        # Find current clip index
        current_index = None
        for i, clip in enumerate(clips):
            if clip.video_id == current_video_id:
                current_index = i
                break
        
        # If current clip not found, start from beginning
        if current_index is None:
            return clips[0].video_id if clips else None
        
        # Look for next clip
        if current_index + 1 < len(clips):
            return clips[current_index + 1].video_id
        
        return None  # No more clips
    
    def get_previous_clip(self, current_video_id: str, clips: list) -> Optional[str]:
        """Find previous clip.
        
        Args:
            current_video_id (str): Current video ID.
            clips (list): List of available clips.
            
        Returns:
            Optional[str]: Previous clip video ID, or None if no previous clips.
        """
        if not clips:
            return None
        
        # Find current clip index
        current_index = None
        for i, clip in enumerate(clips):
            if clip.video_id == current_video_id:
                current_index = i
                break
        
        # If current clip not found, go to last clip
        if current_index is None:
            return clips[-1].video_id if clips else None
        
        # Look for previous clip
        if current_index > 0:
            return clips[current_index - 1].video_id
        
        return None  # No previous clips
    
    def update_statistics(self, total_clips: int, clips_completed: int, 
                         clips_in_progress: int, clips_pending: int) -> None:
        """Update session statistics.
        
        Args:
            total_clips (int): Total number of clips.
            clips_completed (int): Number of completed clips.
            clips_in_progress (int): Number of clips in progress.
            clips_pending (int): Number of pending clips.
        """
        if not self._current_session:
            return
        
        self._current_session.total_clips = total_clips
        self._current_session.clips_completed = clips_completed
        self._current_session.clips_in_progress = clips_in_progress
        self._current_session.clips_pending = clips_pending
        
        self._save_session()
    
    def get_session_state(self) -> Optional[SessionState]:
        """Get current session state.
        
        Returns:
            Optional[SessionState]: Current session state.
        """
        return self._current_session
    
    def reset_session(self) -> None:
        """Reset current session (create new one)."""
        logger.info("Resetting session...")
        self._create_new_session()
    
    def get_session_duration(self) -> float:
        """Get session duration in minutes.
        
        Returns:
            float: Duration in minutes.
        """
        if not self._current_session:
            return 0.0
        
        duration = datetime.now() - self._current_session.started_at
        return duration.total_seconds() / 60.0
    
    def export_session_summary(self) -> Dict[str, Any]:
        """Export session summary for reporting.
        
        Returns:
            Dict[str, Any]: Session summary data.
        """
        if not self._current_session:
            return {}
        
        return {
            'session_id': self._current_session.session_id,
            'started_at': self._current_session.started_at.isoformat(),
            'duration_minutes': self.get_session_duration(),
            'current_position': {
                'video_id': self._current_session.current_video_id,
                'step': self._current_session.current_step
            },
            'progress': {
                'total_clips': self._current_session.total_clips,
                'completed': self._current_session.clips_completed,
                'in_progress': self._current_session.clips_in_progress,
                'pending': self._current_session.clips_pending,
                'completion_percentage': (
                    (self._current_session.clips_completed / self._current_session.total_clips * 100)
                    if self._current_session.total_clips > 0 else 0.0
                )
            }
        }
    
    async def get_next_unprocessed_clip(self, clip_ids: list, progress_tracker) -> Optional[str]:
        """Find the next unprocessed clip.
        
        Args:
            clip_ids (list): List of all available clip IDs.
            progress_tracker: Progress tracker instance to check completion status.
            
        Returns:
            Optional[str]: Next unprocessed clip ID, or None if all clips are processed.
        """
        for clip_id in clip_ids:
            if not progress_tracker.is_clip_completed(clip_id):
                return clip_id
        return None
    
    async def navigate_to(self, video_id: str, step: int) -> None:
        """Navigate to specific video and step.
        
        Args:
            video_id (str): Video ID to navigate to.
            step (int): Step number to navigate to (1-5).
        """
        self.set_position(video_id, step)
    
    async def get_processed_clips(self, clip_ids: list, progress_tracker) -> list:
        """Get list of completed clip IDs.
        
        Args:
            clip_ids (list): List of all available clip IDs.
            progress_tracker: Progress tracker instance to check completion status.
            
        Returns:
            list: List of completed clip IDs.
        """
        return [clip_id for clip_id in clip_ids if progress_tracker.is_clip_completed(clip_id)]
    
    async def get_unprocessed_clips(self, clip_ids: list, progress_tracker) -> list:
        """Get list of unprocessed clip IDs.
        
        Args:
            clip_ids (list): List of all available clip IDs.
            progress_tracker: Progress tracker instance to check completion status.
            
        Returns:
            list: List of unprocessed clip IDs.
        """
        return [clip_id for clip_id in clip_ids if not progress_tracker.is_clip_completed(clip_id)]