"""Pydantic models for the Ground Truth Annotation Web Application."""

from datetime import datetime
from typing import Optional, Dict, List, Any
from pydantic import BaseModel, Field


# Core Video Clip Model
class VideoClip(BaseModel):
    """Represents a video clip with metadata and progress information."""
    
    video_id: str = Field(..., description="Unique identifier for the video clip")
    video_path: str = Field(..., description="Full path to the video file")
    sequence_number: Optional[int] = Field(None, description="Sequential number if available")
    category: str = Field("unknown", description="Video category")
    event_type: str = Field("unmarked", description="Event type")
    duration: float = Field(10.0, description="Duration in seconds")
    severity_level: int = Field(3, description="Severity level 1-5", ge=1, le=5)
    metadata_source: str = Field(..., description="Source of metadata (rich_metadata, ground_truth, file_scan, system_output)")
    has_full_context: bool = Field(False, description="Whether rich metadata is available")
    
    # Optional rich metadata fields
    original_video_id: Optional[str] = Field(None, description="Original video ID before clip extraction")
    start_time: Optional[float] = Field(None, description="Start time in original video")
    confidence: Optional[float] = Field(None, description="Confidence score of the clip marking")
    marked_at: Optional[datetime] = Field(None, description="When the clip was marked")
    file_size: Optional[int] = Field(None, description="File size in bytes")


# Progress Tracking Models
class StepProgress(BaseModel):
    """Tracks progress for individual pipeline steps."""
    
    step_number: int = Field(..., description="Step number (1-5)", ge=1, le=5)
    step_name: str = Field(..., description="Step name (annotation, scenes, violations, accidents, assessment)")
    status: str = Field("pending", description="Status (pending, in_progress, completed)")
    started_at: Optional[datetime] = Field(None, description="When step was started")
    completed_at: Optional[datetime] = Field(None, description="When step was completed")
    data: Optional[Dict[str, Any]] = Field(None, description="Step-specific data")
    manual_edits: bool = Field(False, description="Whether user made manual changes")
    llm_generations: int = Field(0, description="Number of LLM generations used")
    save_count: int = Field(0, description="Number of times saved")
    
    class Config:
        json_encoders = {
            datetime: lambda v: v.isoformat()
        }


class ClipProgress(BaseModel):
    """Tracks overall progress for a single clip through all steps."""
    
    video_id: str = Field(..., description="Video clip identifier")
    current_step: int = Field(1, description="Current step (1-5)", ge=1, le=5)
    overall_status: str = Field("pending", description="Overall status (pending, in_progress, completed)")
    started_at: datetime = Field(..., description="When clip processing was started")
    last_updated: datetime = Field(..., description="Last update timestamp")
    steps: Dict[int, StepProgress] = Field(default_factory=dict, description="Step number -> StepProgress")
    total_time_spent: float = Field(0.0, description="Total time spent in minutes")
    
    class Config:
        json_encoders = {
            datetime: lambda v: v.isoformat()
        }


class SessionState(BaseModel):
    """Tracks global session state and statistics."""
    
    session_id: str = Field(..., description="Unique session identifier")
    started_at: datetime = Field(..., description="Session start time")
    current_video_id: str = Field(..., description="Currently active video ID")
    current_step: int = Field(1, description="Current step number", ge=1, le=5)
    total_clips: int = Field(0, description="Total number of clips discovered")
    clips_completed: int = Field(0, description="Number of clips completed")
    clips_in_progress: int = Field(0, description="Number of clips in progress")
    clips_pending: int = Field(0, description="Number of clips pending")
    last_updated: datetime = Field(..., description="Last session update")
    
    class Config:
        json_encoders = {
            datetime: lambda v: v.isoformat()
        }


# Ground Truth Data Models
class GroundTruthAnnotation(BaseModel):
    """Step 1: Annotation data."""
    annotation: str = Field(..., description="Narrative description of the driving scenario")


class GroundTruthScenes(BaseModel):
    """Step 2: Scene extraction data."""
    scenes: List[str] = Field(..., description="List of atomic scene descriptions")


class ViolationAnalysis(BaseModel):
    """Individual violation analysis for a scene."""
    scene: str = Field(..., description="Scene description")
    violation: str = Field(..., description="Violation status (found, not_found)")
    reason: str = Field(..., description="Violation reason or explanation")


class AccidentAnalysis(BaseModel):
    """Individual accident analysis for a scene."""
    scene: str = Field(..., description="Scene description")
    accident: str = Field(..., description="Accident risk status (found, not_found)")
    consequence: str = Field(..., description="Accident consequence or explanation")


class GroundTruthViolations(BaseModel):
    """Step 3: Violation analysis data."""
    violations: List[ViolationAnalysis] = Field(..., description="Violation analysis per scene")


class GroundTruthAccidents(BaseModel):
    """Step 4: Accident analysis data."""
    accidents: List[AccidentAnalysis] = Field(..., description="Accident analysis per scene")


class SafetyAssessment(BaseModel):
    """Safety assessment data."""
    safety_score: int = Field(..., description="Safety score 1-10", ge=1, le=10)
    risk_level: str = Field(..., description="Risk level (low, medium, high, critical)")
    overall_evaluation: str = Field(..., description="Overall evaluation summary")
    strengths: List[str] = Field(..., description="List of positive driving behaviors")
    weaknesses: List[str] = Field(..., description="List of concerning behaviors")
    improvement_advice: List[str] = Field(..., description="List of improvement recommendations")


class GroundTruthAssessment(BaseModel):
    """Step 5: Assessment data."""
    assessment: SafetyAssessment = Field(..., description="Complete safety assessment")


class GroundTruthData(BaseModel):
    """Complete ground truth data structure compatible with existing pipeline."""
    
    video_id: str = Field(..., description="Video clip identifier")
    video_path: str = Field(..., description="Path to video file")
    ground_truth: Dict[str, Any] = Field(..., description="Ground truth data in standard format")
    
    # Standard ground_truth structure:
    # {
    #   "annotation": str,           # Step 1: Narrative annotation
    #   "scenes": List[str],         # Step 2: List of atomic scenes
    #   "violations": List[dict],    # Step 3: Violation analysis per scene
    #   "accidents": List[dict],     # Step 4: Accident risk analysis per scene
    #   "assessment": dict           # Step 5: Final safety assessment
    # }


# API Request/Response Models
class NavigationRequest(BaseModel):
    """Request to navigate to specific clip and step."""
    video_id: str = Field(..., description="Target video ID")
    step: int = Field(..., description="Target step number", ge=1, le=5)


class GenerateAnnotationRequest(BaseModel):
    """Request to generate annotation for a clip."""
    video_id: str = Field(..., description="Video clip ID")
    optimize: bool = Field(False, description="Whether to optimize existing annotation")
    user_feedback: Optional[str] = Field(None, description="User feedback for optimization")


class SaveStepRequest(BaseModel):
    """Request to save step data."""
    video_id: str = Field(..., description="Video clip ID")
    step: int = Field(..., description="Step number", ge=1, le=5)
    data: Dict[str, Any] = Field(..., description="Step-specific data to save")


# Response Models
class APIResponse(BaseModel):
    """Standard API response format."""
    success: bool = Field(..., description="Whether the operation succeeded")
    message: Optional[str] = Field(None, description="Optional message")
    data: Optional[Dict[str, Any]] = Field(None, description="Optional response data")


class VideoListResponse(BaseModel):
    """Response for listing videos."""
    clips: List[VideoClip] = Field(..., description="List of video clips")
    total_count: int = Field(..., description="Total number of clips")
    completed_count: int = Field(..., description="Number of completed clips")
    in_progress_count: int = Field(..., description="Number of clips in progress")
    pending_count: int = Field(..., description="Number of pending clips")


class SessionStatusResponse(BaseModel):
    """Response for session status."""
    session: SessionState = Field(..., description="Current session state")
    current_clip: Optional[VideoClip] = Field(None, description="Current video clip")
    current_progress: Optional[ClipProgress] = Field(None, description="Current clip progress")


class StepExecutionResponse(BaseModel):
    """Response for pipeline step execution."""
    success: bool = Field(..., description="Whether step execution succeeded")
    step: int = Field(..., description="Step number that was executed")
    video_id: str = Field(..., description="Video ID")
    data: Optional[Dict[str, Any]] = Field(None, description="Generated step data")
    message: Optional[str] = Field(None, description="Optional message")
    execution_time: Optional[float] = Field(None, description="Execution time in seconds")


# Configuration Models
class AppConfig(BaseModel):
    """Application configuration."""
    
    # Directories
    dashcam_dir: str = Field("data/dashcam", description="Directory containing video clips")
    metadata_dir: str = Field("data/dashcam/metadata", description="Directory containing clip metadata")
    ground_truth_dir: str = Field("data/evaluation/ground_truth", description="Ground truth output directory")
    system_output_dir: str = Field("data/evaluation/system_outputs", description="System outputs directory")
    session_dir: str = Field("data/evaluation/sessions", description="Session data directory")
    
    # Application settings
    auto_save_interval: int = Field(30, description="Auto-save interval in seconds")
    backup_enabled: bool = Field(True, description="Whether to create automatic backups")
    video_extensions: List[str] = Field([".mp4", ".avi", ".mov", ".mkv"], description="Supported video extensions")
    
    # LLM settings
    default_model: Optional[str] = Field(None, description="Default LLM model to use")
    enable_caching: bool = Field(True, description="Whether to cache LLM responses")
    
    class Config:
        env_file = ".env"
        env_prefix = "GROUND_TRUTH_"


# Statistics Models
class StepStatistics(BaseModel):
    """Statistics for a specific step across all clips."""
    step_number: int = Field(..., description="Step number")
    step_name: str = Field(..., description="Step name")
    completed_count: int = Field(0, description="Number of clips with this step completed")
    in_progress_count: int = Field(0, description="Number of clips with this step in progress")
    pending_count: int = Field(0, description="Number of clips with this step pending")
    average_time: float = Field(0.0, description="Average time spent on this step")
    manual_edit_percentage: float = Field(0.0, description="Percentage of clips with manual edits")


class GlobalStatistics(BaseModel):
    """Global application statistics."""
    total_clips: int = Field(0, description="Total number of clips")
    clips_completed: int = Field(0, description="Clips fully completed")
    clips_in_progress: int = Field(0, description="Clips currently in progress")
    clips_pending: int = Field(0, description="Clips not yet started")
    completion_percentage: float = Field(0.0, description="Overall completion percentage")
    total_time_spent: float = Field(0.0, description="Total time spent across all clips")
    step_statistics: List[StepStatistics] = Field(..., description="Statistics per step")
    last_updated: datetime = Field(..., description="When statistics were last calculated")
    
    class Config:
        json_encoders = {
            datetime: lambda v: v.isoformat()
        }