"""
Step 4: Accident risk analysis
Adapted for the web application interface
"""

import asyncio
from typing import List, Optional, Dict, Any
from pathlib import Path
import logging
import json
import sys

# Add project root to sys.path
root_path = Path(__file__).parents[4]
src_path = root_path / 'src'
if str(src_path) not in sys.path:
    sys.path.insert(0, str(src_path))
if str(root_path) not in sys.path:
    sys.path.insert(0, str(root_path))

from ..core.models import AccidentScenario, RiskLevel

logger = logging.getLogger(__name__)


class AccidentAnalyzer:
    """
    Step 4: Analyze potential accident scenarios from scenes
    """
    
    def __init__(self):
        self._accident_agent = None
        self._llm = None
        self._agent_available = True
    
    async def _get_accident_agent(self):
        """Lazy load the accident analyzer agent with proper error handling"""
        if self._accident_agent is None and self._agent_available:
            try:
                # Import the module without executing module-level code
                import src.llm.agent.traffic_accident_retriever as accident_module
                
                # Check if we can create the agent
                if hasattr(accident_module, 'graph'):
                    # Try to compile the agent
                    self._accident_agent = accident_module.graph.compile()
                    logger.info("Successfully compiled traffic_accident_agent from graph")
                elif hasattr(accident_module, 'traffic_accident_agent'):
                    # Use pre-compiled agent if available
                    self._accident_agent = accident_module.traffic_accident_agent
                    logger.info("Loaded pre-compiled traffic_accident_agent")
                else:
                    raise AttributeError("No graph or traffic_accident_agent found in module")
                    
            except ImportError as e:
                logger.warning(f"Could not import accident retriever module: {e}")
                self._agent_available = False
            except Exception as e:
                logger.warning(f"Could not initialize accident agent (likely database unavailable): {e}")
                self._agent_available = False
                
        return self._accident_agent
    
    async def _get_llm(self):
        """Lazy load LLM for scene analysis"""
        if self._llm is None:
            try:
                from src.llm.llms import get_llm
                from src.utils.settings import settings
                self._llm = get_llm(settings.app.llm['main'])
                logger.info("Loaded LLM for accident analysis")
            except ImportError as e:
                logger.error(f"Could not load LLM: {e}")
                raise RuntimeError("Failed to load LLM")
        return self._llm
    
    def _create_accident_analysis_prompt(self, scene: str, annotation: str = "") -> str:
        """Create prompt for analyzing accident risks in a scene"""
        return f"""You are an expert traffic safety analyst evaluating accident risks in dashcam footage.

Scene to analyze: {scene}

{f"Full annotation context: {annotation}" if annotation else ""}

For this specific scene, determine:
1. Is there accident risk or potential? (found/not_found)
2. If found, describe the potential consequence

Rules for analysis:
- Only mark "found" if there's genuine collision risk or near-miss potential
- Describe realistic consequences (e.g., "Rear-end collision", "Side-impact crash", "Loss of vehicle control")
- Consider the full context when making your determination
- If no accident risk exists, clearly state so

Respond in JSON format:
{{"accident": "found" or "not_found", "consequence": "potential accident description or 'No possible accident is found'"}}"""
    
    async def analyze_accidents(
        self, 
        video_path: str, 
        scenes: List[str]
    ) -> List[AccidentScenario]:
        """Analyze accident scenarios from scenes
        
        Args:
            video_path: Path to the video file
            scenes: List of scene descriptions as strings
            
        Returns:
            List of AccidentScenario objects
        """
        try:
            # Ensure video path exists (optional check)
            video_path_obj = Path(video_path)
            if not video_path_obj.exists():
                logger.warning(f"Video file not found: {video_path}")
            
            if not scenes:
                logger.info("No scenes provided for accident analysis")
                return []
            
            logger.info(f"Analyzing accident scenarios from {len(scenes)} scenes")
            
            # Try to get the accident agent
            agent = await self._get_accident_agent()
            
            # Get LLM for fallback or primary analysis
            llm = await self._get_llm()
            
            # Analyze each scene
            accidents = []
            loop = asyncio.get_event_loop()
            
            for scene in scenes:
                accident_found = 'not_found'
                consequence_desc = 'No possible accident is found'
                
                # Try using the accident agent if available
                if agent:
                    try:
                        agent_result = await loop.run_in_executor(
                            None,
                            agent.invoke,
                            {"scene": scene}
                        )
                        
                        # Extract accident info from agent result
                        # The agent returns 'consequences' which contains the analysis
                        if 'consequences' in agent_result:
                            consequences = agent_result['consequences']
                            
                            # Handle if consequences is a structured object
                            if hasattr(consequences, 'accident') and hasattr(consequences, 'consequence'):
                                accident_found = consequences.accident
                                consequence_desc = consequences.consequence
                            elif isinstance(consequences, dict):
                                accident_found = consequences.get('accident', 'not_found')
                                consequence_desc = consequences.get('consequence', 'No possible accident is found')
                            elif isinstance(consequences, str):
                                # If it's a string, determine if accident risk was found
                                if consequences and 'no possible accident' not in consequences.lower():
                                    accident_found = 'found'
                                    consequence_desc = consequences
                        
                        
                    except Exception as e:
                        logger.warning(f"Accident agent failed for scene: {e}")
                        agent = None  # Disable agent for remaining scenes
                
                # If agent failed or unavailable, use LLM analysis
                if not agent or accident_found == 'not_found':
                    try:
                        prompt = self._create_accident_analysis_prompt(scene)
                        llm_response = await loop.run_in_executor(
                            None,
                            llm.invoke,
                            prompt
                        )
                        
                        analysis_result = json.loads(llm_response.content.strip())
                        accident_found = analysis_result.get('accident', 'not_found')
                        consequence_desc = analysis_result.get('consequence', 'No possible accident is found')
                        
                        
                    except json.JSONDecodeError:
                        logger.error(f"Failed to parse LLM response for scene: {scene}")
                    except Exception as e:
                        logger.error(f"LLM analysis failed: {e}")
                
                # Determine risk level based on found status
                if accident_found == 'found':
                    # Determine risk level based on keywords in consequence
                    consequence_lower = consequence_desc.lower()
                    if any(word in consequence_lower for word in ['critical', 'severe', 'fatal', 'head-on']):
                        risk_level = RiskLevel.CRITICAL
                    elif any(word in consequence_lower for word in ['high risk', 'serious', 'major']):
                        risk_level = RiskLevel.HIGH
                    elif any(word in consequence_lower for word in ['moderate', 'potential']):
                        risk_level = RiskLevel.MEDIUM
                    else:
                        risk_level = RiskLevel.LOW
                else:
                    risk_level = RiskLevel.LOW
                
                # Create accident scenario object
                accident = AccidentScenario(
                    accident_type='collision' if accident_found == 'found' else 'none',
                    description=f"{scene}: {accident_found} - {consequence_desc}",
                    risk_level=risk_level,
                    timestamp=0.0,  # No timestamp info in string scenes
                    likelihood_score=0.7 if accident_found == 'found' else 0.1,
                    severity_score=0.8 if risk_level in [RiskLevel.HIGH, RiskLevel.CRITICAL] else 0.4,
                    contributing_factors=[]
                )
                
                accidents.append(accident)
            
            logger.info(f"Found {sum(1 for a in accidents if 'found' in a.description)} accident risks out of {len(scenes)} scenes")
            return accidents
            
        except Exception as e:
            logger.error(f"Failed to analyze accidents: {e}")
            raise RuntimeError(f"Failed to analyze accidents: {str(e)}")
    
    def format_accidents_for_json(self, scenes: List[str], accidents: List[AccidentScenario]) -> List[Dict[str, Any]]:
        """Format accidents in the expected JSON structure
        
        Args:
            scenes: Original scene descriptions
            accidents: Analyzed accident scenarios
            
        Returns:
            List of accident dictionaries in expected format
        """
        formatted_accidents = []
        
        for i, (scene, accident) in enumerate(zip(scenes, accidents)):
            # Extract accident status and consequence from description
            desc_parts = accident.description.split(': ', 1)
            if len(desc_parts) > 1 and ' - ' in desc_parts[1]:
                status_consequence = desc_parts[1].split(' - ', 1)
                accident_status = status_consequence[0]
                consequence = status_consequence[1]
            else:
                accident_status = 'not_found'
                consequence = 'No possible accident is found'
            
            formatted_accidents.append({
                "scene": scene,
                "accident": accident_status,
                "consequence": consequence
            })
        
        return formatted_accidents