"""
Step 1: Generate frame-by-frame video annotations
Adapted for the web application interface
"""

import asyncio
from typing import Optional
from pathlib import Path
import sys
import os
import json
import logging

# Add project root to path for imports
root_path = Path(__file__).parents[4]
src_path = root_path / 'src'
if str(src_path) not in sys.path:
    sys.path.insert(0, str(src_path))

# Also add project root for relative imports
if str(root_path) not in sys.path:
    sys.path.insert(0, str(root_path))

logger = logging.getLogger(__name__)


class AnnotationGenerator:
    """
    Step 1: Generate frame-by-frame video annotations
    """
    
    def __init__(self):
        self._video_annotator = None
        self._llm = None
    
    async def _get_video_annotator(self):
        """Lazy load the video annotator"""
        if self._video_annotator is None:
            import sys
            from pathlib import Path
            
            # Add the project root to path (not just src)
            project_root = Path(__file__).parents[4]
            if str(project_root) not in sys.path:
                sys.path.insert(0, str(project_root))
            
            # Import using absolute path from project root
            from src.llm.agent.dashcam_annotation import VideoAnnotator
            # Use GPT-4o for multimodal video analysis
            self._video_annotator = VideoAnnotator(model_id='openai:gpt-4o')
            logger.info("🎥 Loaded VideoAnnotator from DriveGuard pipeline with GPT-4o")
        return self._video_annotator
    
    async def _get_llm(self):
        """Lazy load the LLM for optimization"""
        if self._llm is None:
            import sys
            from pathlib import Path
            
            # Add the project root to path (not just src)
            project_root = Path(__file__).parents[4]
            if str(project_root) not in sys.path:
                sys.path.insert(0, str(project_root))
            
            # Import using absolute path from project root
            from src.llm.llms import get_llm
            from src.utils.settings import settings
            # Use GPT-4.1 for complex reasoning tasks like optimization
            self._llm = get_llm('openai:gpt-4.1')
            logger.info("🧠 Loaded GPT-4.1 from DriveGuard pipeline for annotation optimization")
        return self._llm
    
    async def generate_annotation(
        self, 
        video_path: str, 
        optimize: bool = False, 
        user_feedback: Optional[str] = None
    ) -> str:
        """Generate annotation for a video"""
        try:
            # Ensure video path exists
            if not Path(video_path).exists():
                raise FileNotFoundError(f"Video file not found: {video_path}")
            
            logger.info(f"Generating annotation for {video_path}")
            
            # Try to load existing system annotation first
            video_id = Path(video_path).stem
            system_annotation = await self._load_system_annotation(video_id)
            
            if system_annotation:
                # Found cached annotation - use it directly without modification
                logger.info("✅ Using cached system annotation (no AI calls needed)")
                annotation = system_annotation
            else:
                # Generate from scratch using video annotator
                logger.info("🎥 VIDEO ANALYSIS: Generating annotation from scratch using VideoAnnotator")
                annotator = await self._get_video_annotator()
                
                # Run annotation generation in thread pool to avoid blocking
                loop = asyncio.get_event_loop()
                annotation = await loop.run_in_executor(
                    None, 
                    annotator.annotate_video, 
                    video_path
                )
            
            # If optimization is requested, apply it
            if optimize and user_feedback:
                logger.info("Optimizing annotation with user feedback")
                annotation = await self.optimize_annotation(
                    video_path, annotation, user_feedback
                )
            
            logger.info(f"Generated annotation ({len(annotation)} chars)")
            return annotation
            
        except Exception as e:
            logger.error(f"Failed to generate annotation: {e}")
            raise RuntimeError(f"Failed to generate annotation: {str(e)}")
    
    async def optimize_annotation(
        self, 
        video_path: str, 
        current_annotation: str, 
        user_instructions: str
    ) -> str:
        """Optimize existing annotation with user instructions"""
        try:
            logger.info(f"Optimizing annotation with user instructions: {user_instructions[:50]}...")
            
            llm = await self._get_llm()
            
            # Create optimization prompt
            prompt = self._create_optimization_prompt(
                current_annotation, user_instructions, video_path
            )
            
            # Run optimization in thread pool
            loop = asyncio.get_event_loop()
            optimized_annotation = await loop.run_in_executor(
                None,
                self._invoke_llm,
                llm,
                prompt
            )
            
            logger.info(f"Optimized annotation ({len(optimized_annotation)} chars)")
            return optimized_annotation
            
        except Exception as e:
            logger.error(f"Failed to optimize annotation: {e}")
            raise RuntimeError(f"Failed to optimize annotation: {str(e)}")
    
    async def _improve_system_annotation(self, system_annotation: str, video_path: str) -> str:
        """Improve system annotation to create ground truth"""
        try:
            llm = await self._get_llm()
            
            prompt = self._create_improvement_prompt(system_annotation, video_path)
            
            loop = asyncio.get_event_loop()
            improved_annotation = await loop.run_in_executor(
                None,
                self._invoke_llm,
                llm,
                prompt
            )
            
            return improved_annotation
            
        except Exception as e:
            logger.error(f"Failed to improve system annotation: {e}")
            return system_annotation  # Fallback to original
    
    async def _load_system_annotation(self, video_id: str) -> Optional[str]:
        """Load system annotation if available from s5_generate_system_output"""
        try:
            # Model mapping: prioritize outputs based on current model
            # The annotation generator uses 'openai:gpt-4o' by default, but directories use model names only
            model_priority = [
                'gpt-4o',     # Prefer GPT-4o outputs (matches VideoAnnotator)
                'gpt-4.1',    # Fallback to GPT-4.1 if available
            ]
            
            # Check the structured system output locations
            annotation_base = root_path / "data" / "evaluation" / "system_outputs" / "annotation"
            
            for model_dir in model_priority:
                system_output_file = annotation_base / model_dir / f"{video_id}.json"
                
                if system_output_file.exists():
                    try:
                        with open(system_output_file, 'r', encoding='utf-8') as f:
                            sys_data = json.load(f)
                        
                        # The annotation is stored in the 'content' field
                        annotation = sys_data.get('content', '')
                        
                        if annotation:
                            # Calculate token savings (rough estimate: 1 token ≈ 4 chars)
                            estimated_tokens = len(annotation) // 4
                            generation_time = sys_data.get('generation_time', 15)
                            
                            logger.info(f"✅ CACHED: Loaded existing annotation from {model_dir}")
                            logger.info(f"💰 TOKEN SAVINGS: ~{estimated_tokens:,} tokens saved")
                            logger.info(f"⏱️  TIME SAVINGS: {generation_time:.1f} seconds saved")
                            logger.info(f"📄 Source: {system_output_file.relative_to(root_path)}")
                            
                            return annotation
                    
                    except json.JSONDecodeError as e:
                        logger.warning(f"Invalid JSON in {system_output_file}: {e}")
                    except Exception as e:
                        logger.warning(f"Error loading {system_output_file}: {e}")
            
            # No cached annotation found
            logger.info(f"🚀 GENERATING: No cached annotation found for {video_id}")
            logger.info(f"💡 This will use ~4,000 tokens and take ~15 seconds")
            return None
            
        except Exception as e:
            logger.warning(f"Failed to load system annotation for {video_id}: {e}")
            return None
    
    def _create_improvement_prompt(self, system_annotation: str, video_path: str) -> str:
        """Create prompt for improving system annotation - matches evaluation/2_1_generate_ground_truth_annotations.py"""
        video_name = Path(video_path).name
        
        return f"""You are an expert traffic safety analyst creating ground truth annotations for dashcam video evaluation.

Your task is to transform the structured system-generated annotation below into a coherent narrative ground truth annotation.

Video: {video_name}

System Annotation (Structured Format):
{system_annotation}

EXAMPLE TRANSFORMATION:

System Format: "**Traffic Signals and Traffic Signs:** Lane markings with white dashed lines... **Interactions and Intentions:** Ego-vehicle in second lane, black SUV ahead, silver SUV adjacent... **Unsafe Behaviors:** No speeding observed... **Road Features:** Multi-lane highway, dry conditions..."

Ground Truth Format: "The ego vehicle was traveling in the second lane from the left on a multi-lane highway with clear, dry conditions. A black SUV was directly ahead in the same lane, with a silver SUV visible in the adjacent right lane. The ego vehicle maintained a safe following distance and steady speed, with no aggressive driving behaviors observed. All vehicles maintained proper lane discipline throughout the sequence."

INSTRUCTIONS:
1. **Convert to narrative format**: Transform the structured categories into a flowing chronological story
2. **Maintain chronological order**: Describe events as they unfold in time sequence
3. **Focus on the ego vehicle**: Center the narrative on what the ego vehicle is doing and experiencing
4. **Include critical interactions**: Highlight significant interactions with other vehicles, infrastructure, or hazards
5. **Preserve technical accuracy**: Keep all factual information from the system output
6. **Use objective language**: Avoid subjective interpretations, stick to observable facts
7. **Be concise but complete**: Create a comprehensive but readable narrative
8. **Include safety-relevant details**: Emphasize behaviors, positions, and events that matter for safety evaluation

Transform the system annotation above into a narrative ground truth annotation:"""
    
    def _create_optimization_prompt(self, current_annotation: str, user_instructions: str, video_path: str) -> str:
        """Create prompt for optimizing annotation based on user feedback - matches evaluation/2_2_optimize_ground_truth_annotation.py"""
        video_name = Path(video_path).name
        
        return f"""You are an expert traffic safety analyst helping to refine ground truth annotations for dashcam video evaluation.

Video: {video_name}

Current Ground Truth Annotation:
{current_annotation}

User Revision Instructions:
{user_instructions}

Your task is to revise the annotation according to the user's specific instructions while maintaining:
1. **Factual accuracy**: Only modify what the user specifically requests
2. **Narrative flow**: Keep the chronological, story-like format
3. **Technical precision**: Use accurate traffic safety terminology  
4. **Objective tone**: Maintain professional, factual language
5. **Completeness**: Ensure no important safety details are lost

Instructions for revision:
- Follow the user's instructions precisely
- If instructions are unclear, make the most reasonable interpretation
- Preserve all information not explicitly mentioned for change
- Maintain the same narrative style and structure
- Keep the annotation focused on observable facts

Provide the revised annotation:"""
    
    def _invoke_llm(self, llm, prompt: str) -> str:
        """Synchronous LLM invocation for thread pool execution"""
        try:
            response = llm.invoke(prompt)
            return response.content.strip() if hasattr(response, 'content') else str(response).strip()
        except Exception as e:
            logger.error(f"LLM invocation failed: {e}")
            raise