"""
Step 5: Driving assessment generation
Adapted for the web application interface
"""

import asyncio
from typing import List, Optional, Dict, Any
from pathlib import Path
import logging
import json

logger = logging.getLogger(__name__)


class AssessmentGenerator:
    """
    Step 5: Generate comprehensive driving assessment
    """
    
    def __init__(self):
        self._driving_mentor = None
        self._llm = None
    
    async def _get_driving_mentor(self):
        """Load the real DrivingMentor (no fallback)"""
        if self._driving_mentor is None:
            try:
                # Import from DriveGuard pipeline
                import sys
                root_path = Path(__file__).parents[4]
                src_path = root_path / 'src'
                if str(src_path) not in sys.path:
                    sys.path.insert(0, str(src_path))
                
                logger.info("🚀 Loading real DrivingMentor from DriveGuard pipeline...")
                from src.llm.agent.driving_suggestion import DrivingMentor
                
                # Test instantiation
                self._driving_mentor = DrivingMentor()
                logger.info("✅ Successfully loaded real DrivingMentor!")
                
            except Exception as e:
                logger.error(f"❌ Failed to load DrivingMentor: {type(e).__name__}: {e}")
                logger.error("💥 No fallback available - DrivingMentor is required")
                raise RuntimeError(f"DrivingMentor unavailable: {e}") from e
        
        return self._driving_mentor
    
    async def _get_llm(self):
        """Lazy load the LLM for optimization"""
        if self._llm is None:
            import sys
            
            # Add the project root to path (not just src)
            project_root = Path(__file__).parents[4]
            if str(project_root) not in sys.path:
                sys.path.insert(0, str(project_root))
            
            # Import using absolute path from project root
            from src.llm.llms import get_llm
            # Use GPT-4.1 for complex reasoning tasks like optimization
            self._llm = get_llm('openai:gpt-4.1')
            logger.info("🧠 Loaded GPT-4.1 from DriveGuard pipeline for assessment optimization")
        return self._llm
    
    async def generate_assessment(
        self,
        video_path: str,
        annotation: str,
        scenes: List[Dict[str, Any]],
        violations: List[Dict[str, Any]],
        accidents: List[Dict[str, Any]]
    ) -> Dict[str, Any]:
        """Generate comprehensive driving assessment using real DrivingMentor"""
        # Ensure video path exists
        if not Path(video_path).exists():
            raise FileNotFoundError(f"Video file not found: {video_path}")
        
        logger.info(f"🎯 Generating REAL assessment for {Path(video_path).name}")
        logger.info(f"📊 Input: {len(scenes)} scenes, {len(violations)} violations, {len(accidents)} accidents")
        
        # Load the real DrivingMentor (will raise exception if fails)
        driving_mentor = await self._get_driving_mentor()
        
        logger.info("🤖 Using REAL DrivingMentor for assessment generation")
        
        # Run assessment generation in thread pool
        loop = asyncio.get_event_loop()
        assessment_data = await loop.run_in_executor(
            None,
            driving_mentor.assess_driving,
            annotation  # DrivingMentor only needs annotation
        )
        
        # Validate and return real DrivingMentor format
        if isinstance(assessment_data, dict):
            logger.info(f"✅ Generated REAL assessment: score={assessment_data.get('safety_score', 'unknown')}/10, risk={assessment_data.get('risk_level', 'unknown')}")
            return assessment_data
        else:
            logger.error(f"❌ Unexpected assessment data type: {type(assessment_data)}")
            raise ValueError(f"Invalid assessment data format from DrivingMentor: {type(assessment_data)}")
    
    async def optimize_assessment(
        self, 
        video_path: str, 
        current_assessment: Dict[str, Any], 
        user_instructions: str
    ) -> Dict[str, Any]:
        """Optimize existing assessment with user instructions"""
        try:
            logger.info(f"Optimizing assessment with user instructions: {user_instructions[:50]}...")
            
            llm = await self._get_llm()
            
            # Create optimization prompt
            prompt = self._create_optimization_prompt(
                current_assessment, user_instructions, video_path
            )
            
            # Run optimization in thread pool
            loop = asyncio.get_event_loop()
            optimized_response = await loop.run_in_executor(
                None,
                self._invoke_llm,
                llm,
                prompt
            )
            
            # Parse the optimized assessment response
            optimized_assessment = self._parse_assessment_response(optimized_response)
            
            logger.info(f"✅ Optimized assessment: score={optimized_assessment.get('safety_score', 'unknown')}/10, risk={optimized_assessment.get('risk_level', 'unknown')}")
            return optimized_assessment
            
        except Exception as e:
            logger.error(f"Failed to optimize assessment: {e}")
            raise RuntimeError(f"Failed to optimize assessment: {str(e)}")
    
    def _create_optimization_prompt(self, current_assessment: Dict[str, Any], user_instructions: str, video_path: str) -> str:
        """Create prompt for optimizing assessment based on user feedback"""
        video_name = Path(video_path).name
        
        # Convert current assessment to readable format
        current_assessment_text = json.dumps(current_assessment, indent=2)
        
        return f"""You are an expert driving safety analyst helping to refine a comprehensive driving assessment based on user feedback.

Video: {video_name}

Current Assessment:
{current_assessment_text}

User Revision Instructions:
{user_instructions}

Your task is to revise the assessment according to the user's specific instructions while maintaining:

1. **Safety Score Accuracy (1-10)**: 
   - 1-4: Critical risk (accidents involving ego vehicle)
   - 5-7: High risk (near misses, violations) 
   - 8: Medium risk (minor violations, no accidents)
   - 9-10: Low risk (safe driving, defensive actions)

2. **Risk Level Consistency**: Must match safety score ranges:
   - Critical (1-4), High (5-7), Medium (8), Low (9-10)

3. **Assessment Structure**: Return a JSON object with exactly these fields:
   - safety_score: integer from 1-10
   - risk_level: "critical", "high", "medium", or "low"
   - overall_evaluation: comprehensive text evaluation
   - strengths: array of positive driving behaviors
   - weaknesses: array of identified issues or risky behaviors  
   - improvement_advice: array of specific actionable recommendations

4. **Quality Standards**:
   - Factual accuracy based on the video evidence
   - Professional, objective tone
   - Specific, actionable recommendations
   - Consistent with traffic safety principles

Please return ONLY a valid JSON object with the optimized assessment. Do not include any other text."""
    
    def _invoke_llm(self, llm, prompt: str) -> str:
        """Synchronous LLM invocation for thread pool execution"""
        try:
            response = llm.invoke(prompt)
            return response.content.strip() if hasattr(response, 'content') else str(response).strip()
        except Exception as e:
            logger.error(f"LLM invocation failed: {e}")
            raise
    
    def _parse_assessment_response(self, response: str) -> Dict[str, Any]:
        """Parse LLM response into assessment dictionary"""
        try:
            # Try to extract JSON from the response
            response = response.strip()
            
            # Handle cases where LLM might add extra text around JSON
            if response.startswith('```json'):
                response = response[7:]
            if response.endswith('```'):
                response = response[:-3]
            
            # Parse JSON
            assessment = json.loads(response)
            
            # Validate required fields
            required_fields = ['safety_score', 'risk_level', 'overall_evaluation', 'strengths', 'weaknesses', 'improvement_advice']
            for field in required_fields:
                if field not in assessment:
                    raise ValueError(f"Missing required field: {field}")
            
            # Validate safety score
            if not isinstance(assessment['safety_score'], int) or not (1 <= assessment['safety_score'] <= 10):
                raise ValueError(f"Invalid safety_score: {assessment['safety_score']}")
            
            # Validate risk level
            valid_risk_levels = ['critical', 'high', 'medium', 'low']
            if assessment['risk_level'] not in valid_risk_levels:
                raise ValueError(f"Invalid risk_level: {assessment['risk_level']}")
            
            # Ensure arrays are lists
            for field in ['strengths', 'weaknesses', 'improvement_advice']:
                if not isinstance(assessment[field], list):
                    assessment[field] = [str(assessment[field])] if assessment[field] else []
            
            return assessment
            
        except json.JSONDecodeError as e:
            logger.error(f"Failed to parse assessment JSON: {e}")
            logger.error(f"Response was: {response[:200]}...")
            raise ValueError(f"Invalid JSON response from LLM: {e}")
        except Exception as e:
            logger.error(f"Failed to parse assessment response: {e}")
            raise
    
