"""
Step 2: Scene extraction from video annotations
Adapted for the web application interface
"""

import asyncio
from typing import List
from pathlib import Path
import logging

logger = logging.getLogger(__name__)


class SceneExtractor:
    """
    Step 2: Extract atomic scenes from video annotations
    """

    def __init__(self):
        self._scene_extractor = None

    async def _get_scene_extractor(self):
        """Lazy load the scene extractor"""
        if self._scene_extractor is None:
            # Import from DriveGuard pipeline
            import sys
            root_path = Path(__file__).parents[4]
            sys.path.insert(0, str(root_path))

            from src.llm.agent.scene_extraction import SceneExtractor as DriveGuardSceneExtractor
            self._scene_extractor = DriveGuardSceneExtractor()
            logger.info("Loaded SceneExtractor from DriveGuard pipeline")
        return self._scene_extractor

    async def extract_scenes(
        self,
        video_path: str,
        annotation: str
    ) -> List[str]:
        """Extract scenes from annotation"""
        try:
            # Ensure video path exists
            if not Path(video_path).exists():
                raise FileNotFoundError(f"Video file not found: {video_path}")

            if not annotation or not annotation.strip():
                raise ValueError("Annotation is required for scene extraction")

            logger.info("Extracting scenes from annotation (%d chars)", len(annotation))

            extractor = await self._get_scene_extractor()

            # Run scene extraction in thread pool
            loop = asyncio.get_event_loop()
            scenes = await loop.run_in_executor(
                None,
                extractor.extract,
                annotation
            )

            # The real SceneExtractor returns a list of strings
            logger.info("Extracted %d scenes", len(scenes))
            return scenes

        except Exception as e:
            logger.error("Failed to extract scenes: %s", e)
            raise RuntimeError(f"Failed to extract scenes: {str(e)}") from e
