"""
Step 3: Traffic violation analysis
Adapted for the web application interface
"""

import asyncio
from typing import List, Optional, Dict, Any
from pathlib import Path
import logging
import json
import sys
import re

# Add project root to sys.path
root_path = Path(__file__).parents[4]
src_path = root_path / 'src'
if str(src_path) not in sys.path:
    sys.path.insert(0, str(src_path))
if str(root_path) not in sys.path:
    sys.path.insert(0, str(root_path))

from ..core.models import TrafficViolation, ViolationSeverity

logger = logging.getLogger(__name__)


class ViolationAnalyzer:
    """
    Step 3: Analyze traffic violations from scenes
    """
    
    def __init__(self):
        self._traffic_rule_agent = None
        self._llm = None
        self._agent_available = True
    
    async def _get_traffic_rule_agent(self):
        """Lazy load the traffic rule agent with proper error handling"""
        if self._traffic_rule_agent is None and self._agent_available:
            try:
                # Import the module without executing module-level code
                import src.llm.agent.traffic_rule_checker as rule_checker_module
                
                # Check if we can create the agent
                if hasattr(rule_checker_module, 'graph'):
                    # Try to compile the agent
                    self._traffic_rule_agent = rule_checker_module.graph.compile()
                    logger.info("Successfully compiled traffic_rule_agent from graph")
                elif hasattr(rule_checker_module, 'traffic_rule_agent'):
                    # Use pre-compiled agent if available
                    self._traffic_rule_agent = rule_checker_module.traffic_rule_agent
                    logger.info("Loaded pre-compiled traffic_rule_agent")
                else:
                    raise AttributeError("No graph or traffic_rule_agent found in module")
                    
            except ImportError as e:
                logger.warning(f"Could not import traffic rule checker module: {e}")
                self._agent_available = False
            except Exception as e:
                logger.warning(f"Could not initialize traffic rule agent (likely database unavailable): {e}")
                self._agent_available = False
                
        return self._traffic_rule_agent
    
    async def _get_llm(self):
        """Lazy load LLM for scene analysis"""
        if self._llm is None:
            try:
                from src.llm.llms import get_llm
                from src.utils.settings import settings
                self._llm = get_llm(settings.app.llm['main'])
                logger.info("Loaded LLM for scene analysis")
            except ImportError as e:
                logger.error(f"Could not load LLM: {e}")
                raise RuntimeError("Failed to load LLM")
        return self._llm
    
    def _extract_json_from_response(self, response_text: str) -> Optional[Dict[str, Any]]:
        """Extract valid JSON from LLM response with robust parsing"""
        try:
            # First, try direct parsing (fastest path)
            return json.loads(response_text.strip())
        except json.JSONDecodeError:
            pass
        
        try:
            # Extract JSON object using regex (handles extra text)
            json_match = re.search(r'\{.*\}', response_text, re.DOTALL)
            if json_match:
                json_str = json_match.group(0)
                return json.loads(json_str)
        except json.JSONDecodeError:
            pass
        
        try:
            # Try to extract content between first { and last }
            start_idx = response_text.find('{')
            end_idx = response_text.rfind('}')
            if start_idx != -1 and end_idx != -1 and start_idx < end_idx:
                json_str = response_text[start_idx:end_idx + 1]
                return json.loads(json_str)
        except json.JSONDecodeError:
            pass
        
        # Try to parse line by line to find JSON
        lines = response_text.split('\n')
        for i, line in enumerate(lines):
            if line.strip().startswith('{'):
                # Find the matching closing brace
                brace_count = 0
                json_lines = []
                for j in range(i, len(lines)):
                    json_lines.append(lines[j])
                    for char in lines[j]:
                        if char == '{':
                            brace_count += 1
                        elif char == '}':
                            brace_count -= 1
                            if brace_count == 0:
                                try:
                                    json_str = '\n'.join(json_lines)
                                    return json.loads(json_str)
                                except json.JSONDecodeError:
                                    break
                    if brace_count == 0:
                        break
        
        logger.warning(f"Could not extract valid JSON from response: {response_text[:200]}...")
        return None
    
    def _fallback_parse_response(self, response_text: str) -> tuple[str, str]:
        """Fallback parsing when JSON extraction fails"""
        response_lower = response_text.lower()
        
        # Look for violation keywords
        violation_indicators = ['violation', 'found', 'illegal', 'improper', 'failed to', 'exceeded', 'unsafe']
        no_violation_indicators = ['no violation', 'not found', 'no traffic rule violation', 'compliant', 'legal']
        
        # Check for no violation first (more specific)
        for indicator in no_violation_indicators:
            if indicator in response_lower:
                return 'not_found', 'No traffic rule violation found'
        
        # Check for violation indicators
        for indicator in violation_indicators:
            if indicator in response_lower:
                # Try to extract a meaningful reason
                sentences = response_text.split('.')
                for sentence in sentences:
                    if indicator in sentence.lower():
                        return 'found', sentence.strip()
                return 'found', 'Traffic violation detected but details unclear'
        
        # Default to no violation if unclear
        logger.warning(f"Unclear response, defaulting to no violation: {response_text[:100]}...")
        return 'not_found', 'Response unclear - no violation assumed'
    
    def _create_violation_analysis_prompt(self, scene: str, annotation: str = "") -> str:
        """Create prompt for analyzing violations in a scene"""
        return f"""You are an expert traffic safety analyst evaluating violations in dashcam footage.

Scene to analyze: {scene}

{f"Full annotation context: {annotation}" if annotation else ""}

For this specific scene, determine:
1. Is there a traffic violation? (found/not_found)
2. If found, provide a specific, factual reason

Rules for analysis:
- Only mark "found" if there's a clear, observable traffic law violation
- Be specific about what rule was violated (e.g., "Failed to yield", "Exceeded speed limit", "Improper lane change")
- Consider the full context when making your determination
- If no violation exists, clearly state so

IMPORTANT: Respond ONLY with valid JSON in this exact format:
{{"violation": "found", "reason": "specific violation description"}}
or
{{"violation": "not_found", "reason": "No traffic rule violation found"}}

Do not include any other text before or after the JSON object."""
    
    async def analyze_violations(
        self, 
        video_path: str, 
        scenes: List[str]
    ) -> List[TrafficViolation]:
        """Analyze traffic violations from scenes
        
        Args:
            video_path: Path to the video file
            scenes: List of scene descriptions as strings
            
        Returns:
            List of TrafficViolation objects
        """
        try:
            # Ensure video path exists (optional check)
            video_path_obj = Path(video_path)
            if not video_path_obj.exists():
                logger.warning(f"Video file not found: {video_path}")
            
            if not scenes:
                logger.info("No scenes provided for violation analysis")
                return []
            
            logger.info(f"Analyzing violations from {len(scenes)} scenes")
            
            # Try to get the traffic rule agent
            agent = await self._get_traffic_rule_agent()
            
            # Get LLM for fallback or primary analysis
            llm = await self._get_llm()
            
            # Analyze each scene
            violations = []
            loop = asyncio.get_event_loop()
            
            for scene in scenes:
                violation_found = 'not_found'
                reason = 'No traffic rule violation found'
                
                # Try using the traffic rule agent if available
                if agent:
                    try:
                        agent_result = await loop.run_in_executor(
                            None,
                            agent.invoke,
                            {"query": scene}
                        )
                        
                        # Extract violation info from agent result
                        if 'result' in agent_result:
                            violation_found = agent_result['result'].get('violation', 'not_found')
                            reason = agent_result['result'].get('reason', 'No traffic rule violation found')
                        else:
                            # Handle different response structures
                            violation_found = agent_result.get('violation', 'not_found')
                            reason = agent_result.get('reason', 'No traffic rule violation found')
                        
                        
                    except Exception as e:
                        logger.warning(f"Traffic rule agent failed for scene: {e}")
                        agent = None  # Disable agent for remaining scenes
                
                # If agent failed or unavailable, use LLM analysis
                if not agent or violation_found == 'not_found':
                    try:
                        prompt = self._create_violation_analysis_prompt(scene)
                        llm_response = await loop.run_in_executor(
                            None,
                            llm.invoke,
                            prompt
                        )
                        
                        # Use robust JSON extraction
                        response_text = llm_response.content.strip() if hasattr(llm_response, 'content') else str(llm_response).strip()
                        analysis_result = self._extract_json_from_response(response_text)
                        
                        if analysis_result:
                            violation_found = analysis_result.get('violation', 'not_found')
                            reason = analysis_result.get('reason', 'No traffic rule violation found')
                            logger.debug(f"LLM analysis successful for scene: {scene[:50]}...")
                        else:
                            # Fallback: try to extract information from raw text
                            logger.warning(f"Using fallback parsing for scene: {scene[:50]}...")
                            violation_found, reason = self._fallback_parse_response(response_text)
                        
                    except Exception as e:
                        logger.error(f"LLM analysis failed for scene '{scene[:50]}...': {e}")
                        # Continue with default values (no violation)
                        violation_found = 'not_found'
                        reason = f'Analysis failed: {str(e)}'
                
                # Create violation object matching expected format
                violation = TrafficViolation(
                    violation_type='traffic_rule_violation',
                    description=f"{scene}: {violation_found} - {reason}",
                    severity=ViolationSeverity.HIGH if violation_found == 'found' else ViolationSeverity.LOW,
                    timestamp=0.0,  # No timestamp info in string scenes
                    confidence_score=0.8 if violation_found == 'found' else 0.2
                )
                
                violations.append(violation)
            
            logger.info(f"Found {sum(1 for v in violations if 'found' in v.description)} violations out of {len(scenes)} scenes")
            return violations
            
        except Exception as e:
            logger.error(f"Failed to analyze violations: {e}")
            raise RuntimeError(f"Failed to analyze violations: {str(e)}")
    
    def format_violations_for_json(self, scenes: List[str], violations: List[TrafficViolation]) -> List[Dict[str, Any]]:
        """Format violations in the expected JSON structure
        
        Args:
            scenes: Original scene descriptions
            violations: Analyzed violations
            
        Returns:
            List of violation dictionaries in expected format
        """
        formatted_violations = []
        
        for i, (scene, violation) in enumerate(zip(scenes, violations)):
            # Extract violation status and reason from description
            desc_parts = violation.description.split(': ', 1)
            if len(desc_parts) > 1 and ' - ' in desc_parts[1]:
                status_reason = desc_parts[1].split(' - ', 1)
                violation_status = status_reason[0]
                reason = status_reason[1]
            else:
                violation_status = 'not_found'
                reason = 'No traffic rule violation found'
            
            formatted_violations.append({
                "scene": scene,
                "violation": violation_status,
                "reason": reason
            })
        
        return formatted_violations