/**
 * Main Application Controller
 * Coordinates all components and manages application state
 */

class GroundTruthApp {
    constructor() {
        // Application state
        this.state = {
            currentClip: null,
            currentClipIndex: 0,
            currentStep: 1,
            clips: [],
            processedClips: [],
            unprocessedClips: [],
            session: null,
            loading: false
        };
        
        // Component instances
        this.apiClient = new APIClient();
        this.progressTracker = new ProgressTracker();
        this.videoPlayer = null;
        
        // Step components
        this.stepComponents = {
            1: null, // Will be initialized as StepAnnotation
            2: null, // Will be initialized as StepScenes
            3: null, // Will be initialized as StepViolations
            4: null, // Will be initialized as StepAccidents
            5: null  // Will be initialized as StepAssessment
        };
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }
    
    async init() {
        console.log('Initializing Ground Truth Annotation App...');
        
        try {
            // Initialize DOM elements
            this.initElements();
            
            // Initialize components
            await this.initComponents();
            
            // Set up event listeners
            this.setupEventListeners();
            
            // Load initial data
            await this.loadInitialData();
            
            // Set up auto-refresh
            this.setupAutoRefresh();
            
            console.log('✅ App initialized successfully');
            
        } catch (error) {
            console.error('❌ Failed to initialize app:', error);
            this.showError('Failed to initialize application: ' + error.message);
        }
    }
    
    initElements() {
        // Cache commonly used DOM elements
        this.elements = {
            // Header elements
            refreshBtn: document.getElementById('refresh-btn'),
            
            // Clip navigation
            prevClipBtn: document.getElementById('prev-clip-btn'),
            nextClipBtn: document.getElementById('next-clip-btn'),
            clipCounter: document.getElementById('clip-counter'),
            markCompleteBtn: document.getElementById('mark-complete-btn'),
            
            // Content elements
            currentClipTitle: document.getElementById('current-clip-title'),
            clipMetadata: document.getElementById('clip-metadata'),
            videoPlayer: document.getElementById('main-video-player'),
            videoPlaceholder: document.getElementById('video-placeholder'),
            
            // Pipeline elements (step navigation removed - all steps visible)
            
            // Step containers
            stepContainers: Array.from(document.querySelectorAll('.step-container')),
            
            // Loading overlay
            loadingOverlay: document.getElementById('loading-overlay')
        };
    }
    
    async initComponents() {
        // Initialize video player
        this.videoPlayer = new VideoPlayerComponent(this.elements.videoPlayer);
        
        // Initialize scoring reference component
        this.scoringReference = new ScoringReferenceComponent();
        
        // Initialize step components
        this.stepComponents[1] = new StepAnnotationComponent(document.getElementById('step-1'), this);
        this.stepComponents[2] = new StepScenesComponent(document.getElementById('step-2'), this);
        this.stepComponents[3] = new StepViolationsComponent(document.getElementById('step-3'), this);
        this.stepComponents[4] = new StepAccidentsComponent(document.getElementById('step-4'), this);
        this.stepComponents[5] = new StepAssessmentComponent(document.getElementById('step-5'), this);
        
        // Initialize all components
        await this.videoPlayer.init();
        for (let i = 1; i <= 5; i++) {
            if (this.stepComponents[i]) {
                await this.stepComponents[i].init();
            }
        }
        
        // Connect assessment component with scoring reference
        if (this.stepComponents[5] && this.scoringReference) {
            this.stepComponents[5].scoringReference = this.scoringReference;
        }
    }
    
    setupEventListeners() {
        // Header controls
        if (this.elements.refreshBtn) {
            this.elements.refreshBtn.addEventListener('click', () => this.refresh());
        }
        
        // Clip navigation
        if (this.elements.prevClipBtn) {
            this.elements.prevClipBtn.addEventListener('click', () => this.previousClip());
        }
        if (this.elements.nextClipBtn) {
            this.elements.nextClipBtn.addEventListener('click', () => this.nextClip());
        }
        if (this.elements.markCompleteBtn) {
            this.elements.markCompleteBtn.addEventListener('click', () => this.markClipComplete());
        }
        
        // Step navigation removed - all steps visible simultaneously
        
        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => this.handleKeyboardShortcuts(e));
        
        // Window events
        window.addEventListener('beforeunload', (e) => this.handleBeforeUnload(e));
        window.addEventListener('resize', () => this.handleResize());
    }
    
    async loadInitialData() {
        this.showLoading('Loading application data...');
        
        try {
            // Load session state
            await this.loadSession();
            
            // Load clips and progress status
            await this.loadClips();
            await this.loadClipStatus();
            
            // Always start from the first clip for sequential processing
            // (ignoring session state to ensure consistent starting point)
            await this.loadFirstClip();
            
            this.updateUI();
            
        } finally {
            this.hideLoading();
        }
    }
    
    async loadSession() {
        try {
            const response = await this.apiClient.get('/api/session/current');
            if (response.success !== false) {
                this.state.session = response.session;
                console.log('Session loaded:', this.state.session);
            }
        } catch (error) {
            console.warn('Failed to load session:', error);
        }
    }
    
    async loadClips() {
        try {
            const response = await this.apiClient.get('/api/clips');
            if (response.clips) {
                this.state.clips = response.clips;
                console.log(`Loaded ${this.state.clips.length} clips`);
            }
        } catch (error) {
            console.error('Failed to load clips:', error);
            this.showError('Failed to load video clips');
        }
    }
    
    async loadClipStatus() {
        try {
            const response = await this.apiClient.get('/api/session/clip-status');
            if (response.success && response.data) {
                this.state.processedClips = response.data.processed;
                this.state.unprocessedClips = response.data.unprocessed;
                console.log(`Processed: ${this.state.processedClips.length}, Unprocessed: ${this.state.unprocessedClips.length}`);
            }
        } catch (error) {
            console.error('Failed to load clip status:', error);
        }
    }
    
    async loadFirstClip() {
        if (this.state.clips.length === 0) return;
        
        try {
            // Try to get the next unprocessed clip for smart resume
            const response = await this.apiClient.get('/api/session/next-unprocessed');
            
            if (response.success && response.video_id) {
                // Smart resume: start from first incomplete clip
                await this.loadClip(response.video_id);
                console.log(`Smart resume: Loaded first incomplete clip: ${response.video_id}`);
                this.showSuccess('Resumed from first incomplete clip');
            } else {
                // All clips are complete, or API failed - start from first clip
                await this.loadClip(this.state.clips[0].video_id);
                console.log(`Loaded first clip (fallback or all complete): ${this.state.clips[0].video_id}`);
                
                if (response.success && !response.video_id) {
                    this.showSuccess('🎉 All clips are completed! Starting from first clip for review.');
                }
            }
        } catch (error) {
            console.warn('Smart resume failed, using fallback:', error);
            // Fallback to first clip if API call fails
            await this.loadClip(this.state.clips[0].video_id);
            console.log(`Fallback: Loaded first clip: ${this.state.clips[0].video_id}`);
        }
    }
    
    async loadClip(videoId) {
        if (this.state.currentClip?.video_id === videoId) {
            return; // Already loaded
        }
        
        this.showLoading('Loading clip...');
        
        try {
            // Find clip in state and update index
            const clipIndex = this.state.clips.findIndex(c => c.video_id === videoId);
            if (clipIndex === -1) {
                throw new Error(`Clip not found: ${videoId}`);
            }
            
            this.state.currentClip = this.state.clips[clipIndex];
            this.state.currentClipIndex = clipIndex;
            
            // Load video using streaming endpoint
            await this.videoPlayer.loadVideo(`/api/clips/${videoId}/stream`);
            
            // Load clip data for all steps
            await this.loadClipData();
            
            // Navigate to clip in session
            await this.navigateToClip(videoId);
            
            // Update UI
            this.updateClipUI();
            this.updateClipNavigation(); // Update clip counter and navigation buttons
            
            console.log('Clip loaded:', videoId);
            
        } catch (error) {
            console.error('Failed to load clip:', error);
            this.showError(`Failed to load clip: ${error.message}`);
        } finally {
            this.hideLoading();
        }
    }
    
    async loadClipData() {
        if (!this.state.currentClip) return;
        
        const videoId = this.state.currentClip.video_id;
        
        try {
            // Load progress
            const progressResponse = await this.apiClient.get(`/api/clips/${videoId}/progress`);
            if (progressResponse.success !== false) {
                this.progressTracker.updateClipProgress(videoId, progressResponse.data);
            }
            
            // Load ground truth data
            const gtResponse = await this.apiClient.get(`/api/clips/${videoId}/ground-truth`);
            if (gtResponse.success !== false) {
                this.state.currentClip.groundTruth = gtResponse.data.ground_truth;
            }
            
            // Update step components with data
            for (let step = 1; step <= 5; step++) {
                if (this.stepComponents[step]) {
                    await this.stepComponents[step].loadData(this.state.currentClip.groundTruth);
                }
            }
            
            // Update mark complete button after data is loaded
            this.updateMarkCompleteButton();
            
            // Add a small delay and update again to ensure all components have finished updating
            setTimeout(() => {
                this.updateMarkCompleteButton();
                console.log('Button state updated after component loading');
            }, 100);
            
        } catch (error) {
            console.warn('Failed to load some clip data:', error);
        }
    }
    
    async navigateToClip(videoId, step = null) {
        if (!videoId) return;
        
        try {
            await this.apiClient.post('/api/session/navigate', {
                video_id: videoId,
                step: step || this.state.currentStep
            });
        } catch (error) {
            console.warn('Failed to navigate to clip in session:', error);
        }
    }
    
    setCurrentStep(step) {
        if (step < 1 || step > 5) return;
        
        this.state.currentStep = step;
        this.updateStepUI();
        
        // All steps are now visible simultaneously - no need to hide/show containers
        
        // Navigate in session
        if (this.state.currentClip) {
            this.navigateToClip(this.state.currentClip.video_id, step);
        }
    }
    
    // Step navigation methods removed - all steps visible simultaneously
    
    async previousClip() {
        // Sequential navigation
        if (this.state.currentClipIndex > 0) {
            const previousClip = this.state.clips[this.state.currentClipIndex - 1];
            await this.loadClip(previousClip.video_id);
            this.updateClipNavigation();
        }
    }
    
    async nextClip() {
        // Sequential navigation
        if (this.state.currentClipIndex < this.state.clips.length - 1) {
            const nextClip = this.state.clips[this.state.currentClipIndex + 1];
            await this.loadClip(nextClip.video_id);
            this.updateClipNavigation();
        }
    }
    
    async markClipComplete() {
        if (!this.state.currentClip) {
            this.showError('No clip selected');
            return;
        }
        
        const videoId = this.state.currentClip.video_id;
        
        // Check if all steps are really completed
        const allStepsCompleted = this.checkAllStepsCompleted();
        if (!allStepsCompleted) {
            this.showError('Cannot mark complete: Please complete all 5 steps first (annotation, scenes, violations, accidents, assessment)');
            return;
        }
        
        // Confirm with user
        const confirmed = confirm(`Mark clip "${videoId}" as completed?\n\nThis will:\n• Mark the clip as fully annotated\n• Update completion statistics\n• Move focus to next incomplete clip`);
        if (!confirmed) {
            return;
        }
        
        try {
            this.showLoading('Marking clip complete...');
            
            const response = await this.apiClient.post(`/api/clips/${videoId}/mark-complete`);
            
            if (response.success) {
                this.showSuccess(`Clip marked as complete! (${response.data.completed_count}/${response.data.total_clips} clips done)`);
                
                // Update session statistics
                await this.loadSession();
                this.updateHeaderUI();
                
                // Automatically move to next unprocessed clip
                await this.jumpToNextUnprocessed();
                
            } else {
                throw new Error(response.error || 'Failed to mark clip complete');
            }
            
        } catch (error) {
            console.error('Failed to mark clip complete:', error);
            this.showError('Failed to mark clip complete: ' + error.message);
        } finally {
            this.hideLoading();
        }
    }
    
    checkAllStepsCompleted() {
        // Check if all 5 step components report they have valid data
        // First check ground truth data (most reliable), then component state
        const groundTruth = this.state.currentClip?.groundTruth;
        
        for (let step = 1; step <= 5; step++) {
            const component = this.stepComponents[step];
            if (!component) continue;
            
            // Different steps have different ways to check completion
            switch (step) {
                case 1: // Annotation
                    if (!groundTruth?.annotation && !component.currentAnnotation) {
                        console.log('Step 1 incomplete: no annotation found');
                        return false;
                    }
                    break;
                case 2: // Scenes
                    const hasScenes = (groundTruth?.scenes && groundTruth.scenes.length > 0) || 
                                     (component.currentScenes && component.currentScenes.length > 0);
                    if (!hasScenes) {
                        console.log('Step 2 incomplete: no scenes found');
                        return false;
                    }
                    break;
                case 3: // Violations
                    const hasViolations = (groundTruth?.violations && groundTruth.violations.length > 0) || 
                                         (component.currentViolations && component.currentViolations.length > 0);
                    if (!hasViolations) {
                        console.log('Step 3 incomplete: no violations found');
                        return false;
                    }
                    break;
                case 4: // Accidents
                    const hasAccidents = (groundTruth?.accidents && groundTruth.accidents.length > 0) || 
                                        (component.currentAccidents && component.currentAccidents.length > 0);
                    if (!hasAccidents) {
                        console.log('Step 4 incomplete: no accidents found');
                        return false;
                    }
                    break;
                case 5: // Assessment
                    const hasAssessment = groundTruth?.assessment || component.currentAssessment;
                    if (!hasAssessment) {
                        console.log('Step 5 incomplete: no assessment found');
                        return false;
                    }
                    break;
            }
        }
        console.log('All steps completed!');
        return true;
    }
    
    async jumpToNextUnprocessed() {
        try {
            const response = await this.apiClient.get('/api/session/next-unprocessed');
            
            if (response.success && response.video_id) {
                // Load the next unprocessed clip
                await this.loadClip(response.video_id);
                console.log(`Jumped to next unprocessed clip: ${response.video_id}`);
            } else {
                // All clips are processed!
                this.showSuccess('🎉 All clips have been completed! Great work!');
                console.log('All clips are completed');
            }
        } catch (error) {
            console.warn('Failed to jump to next unprocessed clip:', error);
            // Continue with current clip - not a critical error
        }
    }
    
    
    // Clip list rendering removed - using clip navigation instead
    
    // Search functionality removed - using clip navigation instead
    
    updateUI() {
        this.updateHeaderUI();
        this.updateProgressUI();
        this.updateClipUI();
        this.updateStepUI();
        this.updateClipNavigation();
    }
    
    updateHeaderUI() {
        // Header UI updates removed - session stats were not working properly
        // Keeping this method stub in case future header updates are needed
    }
    
    updateProgressUI() {
        // Progress overview removed with sidebar - no longer needed
        return;
    }
    
    updateClipUI() {
        if (this.state.currentClip) {
            this.elements.currentClipTitle.textContent = this.state.currentClip.video_id;
            
            // Update metadata
            const metadata = [
                `<div class="metadata-item"><i class="fas fa-tag"></i> ${this.state.currentClip.category}</div>`,
                `<div class="metadata-item"><i class="fas fa-clock"></i> ${this.state.currentClip.duration || 10}s</div>`,
                `<div class="metadata-item"><i class="fas fa-database"></i> ${this.state.currentClip.metadata_source}</div>`
            ];
            
            if (this.state.currentClip.sequence_number) {
                metadata.unshift(
                    `<div class="metadata-item"><i class="fas fa-hashtag"></i> ${this.state.currentClip.sequence_number}</div>`
                );
            }
            
            this.elements.clipMetadata.innerHTML = metadata.join('');
            
            // Show video, hide placeholder
            this.elements.videoPlaceholder.style.display = 'none';
        } else {
            this.elements.currentClipTitle.textContent = 'Select a video clip to begin';
            this.elements.clipMetadata.innerHTML = '';
            this.elements.videoPlaceholder.style.display = 'flex';
        }
    }
    
    // updateClipListSelection removed - no clip list to update
    
    updateStepUI() {
        // Update step status indicators
        for (let step = 1; step <= 5; step++) {
            const statusElement = document.getElementById(`step-${step}-status`);
            if (statusElement) {
                // This would be updated based on actual progress data
                const status = step === this.state.currentStep ? 'in-progress' : 
                              step < this.state.currentStep ? 'completed' : 'pending';
                
                statusElement.textContent = status;
                statusElement.className = `step-status ${status}`;
            }
        }
    }
    
    advanceToNextStep() {
        if (this.state.currentStep < 5) {
            this.state.currentStep++;
            this.updateStepUI();
            console.log(`✅ Advanced to step ${this.state.currentStep}`);
        }
    }
    
    updateClipNavigation() {
        if (this.elements.clipCounter) {
            // Show sequential counter
            const current = this.state.currentClipIndex + 1;
            const total = this.state.clips.length;
            this.elements.clipCounter.textContent = `${current} / ${total}`;
        }
        
        if (this.elements.prevClipBtn) {
            this.elements.prevClipBtn.disabled = this.state.currentClipIndex === 0;
            this.elements.prevClipBtn.innerHTML = '<i class="fas fa-chevron-left"></i> Previous Clip';
        }
        
        if (this.elements.nextClipBtn) {
            this.elements.nextClipBtn.disabled = this.state.currentClipIndex === this.state.clips.length - 1;
            this.elements.nextClipBtn.innerHTML = 'Next Clip <i class="fas fa-chevron-right"></i>';
        }
        
        // Update mark complete button state
        this.updateMarkCompleteButton();
    }
    
    updateMarkCompleteButton() {
        if (!this.elements.markCompleteBtn) return;
        
        const allStepsCompleted = this.checkAllStepsCompleted();
        const hasCurrentClip = !!this.state.currentClip;
        
        if (hasCurrentClip && allStepsCompleted) {
            this.elements.markCompleteBtn.disabled = false;
            this.elements.markCompleteBtn.innerHTML = '<i class="fas fa-check-circle"></i> Mark Complete';
            this.elements.markCompleteBtn.title = 'Mark this clip as fully annotated';
        } else if (hasCurrentClip) {
            this.elements.markCompleteBtn.disabled = true;
            this.elements.markCompleteBtn.innerHTML = '<i class="fas fa-clock"></i> Complete All Steps';
            this.elements.markCompleteBtn.title = 'Please complete all 5 steps first';
        } else {
            this.elements.markCompleteBtn.disabled = true;
            this.elements.markCompleteBtn.innerHTML = '<i class="fas fa-check-circle"></i> Mark Complete';
            this.elements.markCompleteBtn.title = 'No clip selected';
        }
    }
    
    handleKeyboardShortcuts(event) {
        // Handle global keyboard shortcuts
        const { ctrlKey, metaKey, key } = event;
        const cmdOrCtrl = ctrlKey || metaKey;
        
        switch (key) {
            case 'r':
                if (cmdOrCtrl) {
                    event.preventDefault();
                    this.refresh();
                }
                break;
                
            case 's':
                if (cmdOrCtrl) {
                    event.preventDefault();
                    this.saveCurrentStep();
                }
                break;
        }
    }
    
    async saveCurrentStep() {
        const stepComponent = this.stepComponents[this.state.currentStep];
        if (stepComponent && typeof stepComponent.save === 'function') {
            await stepComponent.save();
        }
    }
    
    handleBeforeUnload(event) {
        // Check for unsaved changes
        if (this.hasUnsavedChanges()) {
            event.preventDefault();
            event.returnValue = 'You have unsaved changes. Are you sure you want to leave?';
        }
    }
    
    hasUnsavedChanges() {
        // Check if any step component has unsaved changes
        for (let step = 1; step <= 5; step++) {
            const component = this.stepComponents[step];
            if (component && typeof component.hasUnsavedChanges === 'function' && component.hasUnsavedChanges()) {
                return true;
            }
        }
        return false;
    }
    
    handleResize() {
        // Handle responsive behavior
        if (this.videoPlayer && typeof this.videoPlayer.handleResize === 'function') {
            this.videoPlayer.handleResize();
        }
    }
    
    setupAutoRefresh() {
        // Refresh session data every 30 seconds
        setInterval(async () => {
            try {
                await this.loadSession();
                this.updateHeaderUI();
            } catch (error) {
                console.warn('Auto-refresh failed:', error);
            }
        }, 30000);
    }
    
    async refresh() {
        this.showLoading('Refreshing...');
        
        try {
            await this.loadSession();
            await this.loadClips();
            await this.loadClipStatus();
            
            if (this.state.currentClip) {
                await this.loadClipData();
            }
            
            this.updateUI();
            
            // Force update mark complete button
            this.updateMarkCompleteButton();
            
        } catch (error) {
            console.error('Refresh failed:', error);
            this.showError('Failed to refresh data');
        } finally {
            this.hideLoading();
        }
    }
    
    // Manual button refresh for debugging
    refreshMarkCompleteButton() {
        console.log('Manually refreshing mark complete button...');
        this.updateMarkCompleteButton();
    }
    
    // Cross-step communication methods
    notifyStepDataChanged(stepNumber, data) {
        console.log(`Step ${stepNumber} data changed:`, data);
        
        // When Step 2 (scenes) changes, update Steps 3 and 4
        if (stepNumber === 2 && data.scenes) {
            this.syncScenesWithOtherSteps(data.scenes);
        }
        
        // Update mark complete button whenever any step changes
        this.updateMarkCompleteButton();
    }
    
    syncScenesWithOtherSteps(scenes) {
        console.log('Syncing scenes with Steps 3 and 4:', scenes);
        
        // Update Step 3 (Violations) with new scenes
        if (this.stepComponents[3] && typeof this.stepComponents[3].syncWithScenes === 'function') {
            this.stepComponents[3].syncWithScenes(scenes);
        }
        
        // Update Step 4 (Accidents) with new scenes
        if (this.stepComponents[4] && typeof this.stepComponents[4].syncWithScenes === 'function') {
            this.stepComponents[4].syncWithScenes(scenes);
        }
    }

    showLoading(message = 'Loading...') {
        this.state.loading = true;
        if (this.elements.loadingOverlay) {
            this.elements.loadingOverlay.querySelector('p').textContent = message;
            this.elements.loadingOverlay.style.display = 'flex';
        }
    }
    
    hideLoading() {
        this.state.loading = false;
        if (this.elements.loadingOverlay) {
            this.elements.loadingOverlay.style.display = 'none';
        }
    }
    
    showError(message) {
        console.error('App Error:', message);
        // Could implement toast notifications or error modal here
        alert('Error: ' + message);
    }
    
    showSuccess(message) {
        console.log('Success:', message);
        // Could implement toast notifications here
    }
}

// Initialize application when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.app = new GroundTruthApp();
});

// Export for modules that need app reference
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GroundTruthApp;
}