/**
 * Safety Scoring Reference Component
 * Handles the standalone scoring criteria reference section
 */

class ScoringReferenceComponent {
    constructor() {
        this.isExpanded = false;
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Scoring Reference component initialized');
    }

    initElements() {
        this.elements = {
            toggle: document.querySelector('#scoring-reference-toggle'),
            panel: document.querySelector('#scoring-reference-panel'),
            assessmentCriteriaToggle: document.querySelector('#score-criteria-toggle'),
        };
    }

    setupEventListeners() {
        // Main reference toggle
        if (this.elements.toggle) {
            this.elements.toggle.addEventListener('click', () => {
                this.toggleReference();
            });
        }
        
        // Assessment form criteria toggle (scrolls to reference)
        if (this.elements.assessmentCriteriaToggle) {
            this.elements.assessmentCriteriaToggle.addEventListener('click', () => {
                this.scrollToReference();
            });
        }
    }

    toggleReference() {
        if (!this.elements.panel || !this.elements.toggle) return;
        
        this.isExpanded = !this.isExpanded;
        
        if (this.isExpanded) {
            this.showReference();
        } else {
            this.hideReference();
        }
    }

    showReference() {
        if (this.elements.panel) {
            this.elements.panel.style.display = 'block';
        }
        
        if (this.elements.toggle) {
            this.elements.toggle.classList.add('expanded');
        }
        
        this.isExpanded = true;
    }

    hideReference() {
        if (this.elements.panel) {
            this.elements.panel.style.display = 'none';
        }
        
        if (this.elements.toggle) {
            this.elements.toggle.classList.remove('expanded');
        }
        
        this.isExpanded = false;
    }

    scrollToReference() {
        // Simply show the reference panel if it's hidden (no scrolling needed since it's in the same step)
        if (!this.isExpanded) {
            this.showReference();
        }
        
        // Add a brief highlight effect to draw attention
        const referenceSection = document.querySelector('.scoring-reference-section');
        if (referenceSection) {
            referenceSection.style.boxShadow = '0 0 20px rgba(99, 102, 241, 0.3)';
            setTimeout(() => {
                referenceSection.style.boxShadow = '';
            }, 2000);
        }
    }

    highlightScore(score) {
        if (!this.elements.panel || !this.isExpanded) return;
        
        // Remove all previous highlights
        const criteriaItems = this.elements.panel.querySelectorAll('.criteria-item');
        criteriaItems.forEach(item => {
            item.classList.remove('active');
        });
        
        // Highlight current score
        const activeItem = this.elements.panel.querySelector(`[data-score="${score}"]`);
        if (activeItem) {
            activeItem.classList.add('active');
            
            // Scroll the highlighted item into view within the panel
            activeItem.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'nearest' 
            });
        }
        
        // Highlight current risk section
        const sections = this.elements.panel.querySelectorAll('.criteria-section');
        sections.forEach(section => section.classList.remove('active-section'));
        
        const riskLevel = this.getRiskLevelForScore(score);
        const activeSection = this.elements.panel.querySelector(`.${riskLevel}-risk`);
        if (activeSection) {
            activeSection.classList.add('active-section');
        }
    }

    getRiskLevelForScore(score) {
        if (score >= 1 && score <= 4) return 'critical';
        if (score >= 5 && score <= 7) return 'high';
        if (score === 8) return 'medium';
        if (score >= 9 && score <= 10) return 'low';
        return 'medium'; // fallback
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.ScoringReferenceComponent = ScoringReferenceComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ScoringReferenceComponent;
}