/**
 * Step 4: Accidents Component
 * Handles accident risk analysis and editing
 */

class StepAccidentsComponent {
    constructor(container, app) {
        this.container = container;
        this.app = app;
        this.currentAccidents = [];
        this.hasUnsavedChanges = false;
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Step Accidents component initialized');
    }

    initElements() {
        this.elements = {
            analyzeBtn: this.container.querySelector('#analyze-accidents-btn'),
            accidentsList: this.container.querySelector('#accidents-list'),
            saveBtn: this.container.querySelector('#save-accidents-btn'),
            resetBtn: this.container.querySelector('#reset-accidents-btn'),
            status: this.container.querySelector('#step-4-status')
        };
    }

    setupEventListeners() {
        // Analyze accidents button
        if (this.elements.analyzeBtn) {
            this.elements.analyzeBtn.addEventListener('click', () => this.analyzeAccidents());
        }

        // Save button
        if (this.elements.saveBtn) {
            this.elements.saveBtn.addEventListener('click', () => this.save());
        }

        // Reset button
        if (this.elements.resetBtn) {
            this.elements.resetBtn.addEventListener('click', () => this.reset());
        }
    }

    async loadData(groundTruthData) {
        // Always reset state regardless of data presence
        if (groundTruthData && groundTruthData.accidents) {
            this.currentAccidents = groundTruthData.accidents;
        } else {
            this.currentAccidents = []; // Clear previous clip's data
        }
        
        this.renderAccidents();
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    async analyzeAccidents() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        console.log('Starting accidents analysis for:', videoId);

        try {
            this.setLoading(true, 'Analyzing accidents...');
            
            const response = await this.app.apiClient.analyzeAccidents(videoId);
            console.log('API response received:', response);

            if (response.success) {
                // Validate response data structure
                if (!response.data || !Array.isArray(response.data.accidents)) {
                    throw new Error('Invalid response format: accidents data is missing or invalid');
                }

                console.log('Processing accidents data:', response.data.accidents);
                
                try {
                    // Process the response data
                    this.currentAccidents = response.data.accidents;
                    console.log('Set currentAccidents:', this.currentAccidents);
                    
                    this.renderAccidents();
                    console.log('Rendered accidents to UI');
                    
                    this.hasUnsavedChanges = true;
                    this.updateStatus('completed');
                    
                    // Update app state if available
                    if (this.app.state.currentClip.groundTruth) {
                        this.app.state.currentClip.groundTruth.accidents = this.currentAccidents;
                        console.log('Updated app state');
                    }
                    
                    this.app.showSuccess('Accidents analyzed successfully');
                    console.log('✅ Accidents analysis completed successfully');
                    
                } catch (processingError) {
                    console.error('Error processing successful response:', processingError);
                    throw new Error('Failed to process analysis results: ' + processingError.message);
                }
            } else {
                throw new Error(response.error || 'Failed to analyze accidents');
            }

        } catch (error) {
            console.error('Failed to analyze accidents:', error);
            this.app.showError('Failed to analyze accidents: ' + error.message);
            this.updateStatus('error');
        } finally {
            console.log('Resetting loading state');
            this.setLoading(false);
        }
    }

    renderAccidents() {
        if (!this.elements.accidentsList) return;

        if (this.currentAccidents.length === 0) {
            this.elements.accidentsList.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-car-crash"></i>
                    <p>No accident risks identified</p>
                    <small>Use the "Analyze Accidents" button to assess potential accident scenarios, or ensure scenes are extracted in Step 2</small>
                </div>
            `;
            return;
        }

        const html = this.currentAccidents.map((accident, index) => {
            const accidentStatus = accident.accident || 'not_found';
            const isFound = accidentStatus === 'found';
            
            return `
                <div class="accident-item-compact ${isFound ? 'found' : 'not-found'}" data-accident-index="${index}">
                    <div class="accident-header-compact">
                        <span class="accident-number">Scene ${index + 1}</span>
                        <span class="accident-status-indicator ${isFound ? 'found' : 'not-found'}">
                            <i class="fas ${isFound ? 'fa-exclamation-triangle' : 'fa-check-circle'}"></i>
                            ${isFound ? 'Accident Risk Found' : 'No Accident Risk'}
                        </span>
                    </div>
                    <div class="accident-content-compact">
                        <div class="form-row-inline">
                            <label>Scene:</label>
                            <span class="scene-text-compact">${accident.scene || 'Scene description not available'}</span>
                        </div>
                        <div class="form-row-inline">
                            <label>Status:</label>
                            <select 
                                class="form-control-compact accident-status-select" 
                                onchange="app.stepComponents[4].updateAccident(${index}, 'accident', this.value)"
                            >
                                <option value="not_found" ${accidentStatus === 'not_found' ? 'selected' : ''}>No Accident Risk</option>
                                <option value="found" ${accidentStatus === 'found' ? 'selected' : ''}>Accident Risk Found</option>
                            </select>
                        </div>
                        <div class="form-row-block">
                            <label>Consequence:</label>
                            <textarea 
                                class="form-control-compact accident-consequence" 
                                rows="2"
                                placeholder="${isFound ? 'Describe the accident consequence...' : 'Explain why no accident risk was found...'}"
                                onchange="app.stepComponents[4].updateAccident(${index}, 'consequence', this.value)"
                            >${accident.consequence || ''}</textarea>
                        </div>
                    </div>
                </div>
            `;
        }).join('');

        this.elements.accidentsList.innerHTML = html;
    }


    updateAccident(index, field, value) {
        if (index >= 0 && index < this.currentAccidents.length) {
            this.currentAccidents[index][field] = value;
            this.hasUnsavedChanges = true;
            this.updateUI();
            
            // Re-render if risk level changed (affects styling)
            if (field === 'risk_level' || field === 'accident_type') {
                this.renderAccidents();
            }
        }
    }


    // Sync with scenes from Step 2
    syncWithScenes(scenes) {
        console.log('Step 4: Syncing with scenes from Step 2:', scenes);
        
        if (!Array.isArray(scenes)) {
            console.warn('Step 4: Invalid scenes data received');
            return;
        }
        
        // Preserve existing accident data where possible
        const existingAccidents = [...this.currentAccidents];
        const newAccidents = [];
        
        scenes.forEach((scene, index) => {
            const sceneText = typeof scene === 'string' ? scene : (scene.description || scene);
            
            // Try to find existing accident for this scene
            let existingAccident = null;
            
            // First try to match by scene text
            existingAccident = existingAccidents.find(a => a.scene === sceneText);
            
            // If not found, try to match by index (if within range)
            if (!existingAccident && existingAccidents[index]) {
                existingAccident = existingAccidents[index];
            }
            
            if (existingAccident) {
                // Update scene text but preserve accident status and consequence
                newAccidents.push({
                    scene: sceneText,
                    accident: existingAccident.accident || 'not_found',
                    consequence: existingAccident.consequence || ''
                });
            } else {
                // Create new accident entry
                newAccidents.push({
                    scene: sceneText,
                    accident: 'not_found',
                    consequence: ''
                });
            }
        });
        
        this.currentAccidents = newAccidents;
        this.renderAccidents();
        this.hasUnsavedChanges = true;
        this.updateUI();
        
        console.log('Step 4: Synced accidents:', this.currentAccidents);
    }

    // Removed: addNewAccident, deleteAccident, duplicateAccident
    // These methods are no longer needed as scenes are managed by Step 2


    async save() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.hasUnsavedChanges) {
            this.app.showSuccess('No changes to save');
            return;
        }

        // Validate accidents
        const errors = this.getValidationErrors();
        if (errors.length > 0) {
            this.app.showError('Please fix the following errors:\n' + errors.join('\n'));
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Saving accidents...');

            const response = await this.app.apiClient.saveStepData(videoId, 4, {
                accidents: this.currentAccidents
            });

            if (response.success) {
                this.hasUnsavedChanges = false;
                this.updateStatus('completed');
                
                // Update app state
                if (this.app.state.currentClip.groundTruth) {
                    this.app.state.currentClip.groundTruth.accidents = this.currentAccidents;
                }

                this.app.showSuccess('Accidents saved successfully');
            } else {
                throw new Error(response.error || 'Failed to save accidents');
            }

        } catch (error) {
            console.error('Failed to save accidents:', error);
            this.app.showError('Failed to save accidents: ' + error.message);
        } finally {
            this.setLoading(false);
        }
    }

    async reset() {
        if (!confirm('Are you sure you want to reset all accident scenarios? Any unsaved changes will be lost.')) {
            return;
        }

        // Reload from ground truth data
        if (this.app.state.currentClip?.groundTruth?.accidents) {
            this.currentAccidents = [...this.app.state.currentClip.groundTruth.accidents];
        } else {
            this.currentAccidents = [];
        }

        this.renderAccidents();
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    updateStatus(status) {
        if (this.elements.status) {
            this.elements.status.textContent = status;
            this.elements.status.className = `step-status ${status}`;
        }
    }

    setLoading(loading, message = 'Loading...') {
        // Disable/enable buttons
        const buttons = [
            this.elements.analyzeBtn,
            this.elements.saveBtn,
            this.elements.resetBtn
        ];

        buttons.forEach(btn => {
            if (btn) {
                btn.disabled = loading;
                if (loading) {
                    btn.dataset.originalText = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ' + message;
                } else {
                    btn.innerHTML = btn.dataset.originalText || btn.innerHTML;
                    
                    // Failsafe: Force reset to known good state for specific buttons
                    if (btn === this.elements.analyzeBtn) {
                        btn.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Analyze';
                    } else if (btn === this.elements.saveBtn) {
                        btn.innerHTML = '<i class="fas fa-save"></i> Save';
                    } else if (btn === this.elements.resetBtn) {
                        btn.innerHTML = '<i class="fas fa-undo"></i> Reset';
                    }
                }
            }
        });

        if (loading) {
            this.updateStatus('in-progress');
        }
    }

    updateUI() {
        // Update save button state
        if (this.elements.saveBtn) {
            this.elements.saveBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update reset button state
        if (this.elements.resetBtn) {
            this.elements.resetBtn.disabled = !this.hasUnsavedChanges;
        }
    }

    hasUnsavedChanges() {
        return this.hasUnsavedChanges;
    }

    getValidationErrors() {
        const errors = [];
        
        this.currentAccidents.forEach((accident, index) => {
            if (!accident.scene || accident.scene.trim().length < 5) {
                errors.push(`Accident ${index + 1}: Scene description must be at least 5 characters`);
            }
            
            if (!accident.accident || !['found', 'not_found'].includes(accident.accident)) {
                errors.push(`Accident ${index + 1}: Accident status must be 'found' or 'not_found'`);
            }
            
            if (!accident.consequence || accident.consequence.trim().length < 10) {
                errors.push(`Accident ${index + 1}: Consequence must be at least 10 characters`);
            }
        });

        return errors;
    }

    isValid() {
        return this.getValidationErrors().length === 0;
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.StepAccidentsComponent = StepAccidentsComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = StepAccidentsComponent;
}