/**
 * Step 1: Annotation Component
 * Handles annotation generation and optimization
 */

class StepAnnotationComponent {
    constructor(container, app) {
        this.container = container;
        this.app = app;
        this.currentAnnotation = '';
        this.hasUnsavedChanges = false;
        
        // DOM elements
        this.elements = {};
        
        // Request debouncing to prevent duplicate API calls
        this.pendingRequests = new Set();
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Step Annotation component initialized');
    }

    initElements() {
        this.elements = {
            generateBtn: this.container.querySelector('#generate-annotation-btn'),
            optimizeBtn: this.container.querySelector('#optimize-annotation-btn'),
            annotationText: this.container.querySelector('#annotation-text'),
            saveBtn: this.container.querySelector('#save-annotation-btn'),
            resetBtn: this.container.querySelector('#reset-annotation-btn'),
            status: this.container.querySelector('#step-1-status'),
            optimizationSection: this.container.querySelector('#optimization-section'),
            feedbackTextarea: this.container.querySelector('#optimization-feedback')
        };
    }

    setupEventListeners() {
        // Generation button
        if (this.elements.generateBtn) {
            this.elements.generateBtn.addEventListener('click', () => this.generateAnnotation());
        }

        // Optimization button
        if (this.elements.optimizeBtn) {
            this.elements.optimizeBtn.addEventListener('click', () => this.optimizeAnnotation());
        }

        // Text area changes
        if (this.elements.annotationText) {
            this.elements.annotationText.addEventListener('input', () => {
                this.hasUnsavedChanges = true;
                this.updateUI();
            });
        }

        // Save button
        if (this.elements.saveBtn) {
            this.elements.saveBtn.addEventListener('click', () => this.save());
        }

        // Reset button
        if (this.elements.resetBtn) {
            this.elements.resetBtn.addEventListener('click', () => this.reset());
        }
    }

    async loadData(groundTruthData) {
        if (groundTruthData && groundTruthData.annotation) {
            this.currentAnnotation = groundTruthData.annotation;
            
            if (this.elements.annotationText) {
                this.elements.annotationText.value = this.currentAnnotation;
            }
            
            // Show optimization section if annotation exists
            if (this.elements.optimizationSection) {
                this.elements.optimizationSection.style.display = 'block';
            }
            
            this.hasUnsavedChanges = false;
            this.updateUI();
        } else {
            // No annotation exists yet - clear the textarea and show placeholder
            this.currentAnnotation = '';
            if (this.elements.annotationText) {
                this.elements.annotationText.value = '';
                this.elements.annotationText.placeholder = 'Waiting to be generated with AI agent...';
            }
            
            // Hide optimization section when no annotation
            if (this.elements.optimizationSection) {
                this.elements.optimizationSection.style.display = 'none';
            }
            
            this.hasUnsavedChanges = false;
            this.updateUI();
        }
    }

    async generateAnnotation() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        const requestKey = `generate-${videoId}`;
        
        // Prevent duplicate requests
        if (this.pendingRequests.has(requestKey)) {
            console.log('Generate annotation request already pending, ignoring duplicate');
            return;
        }
        
        this.pendingRequests.add(requestKey);

        try {
            // Only show loading on the generate button
            this.setLoading(true, 'Generating annotation with AI agent...', this.elements.generateBtn);
            
            const response = await this.app.apiClient.generateAnnotation(videoId);

            if (response.success) {
                this.currentAnnotation = response.data.annotation;
                
                if (this.elements.annotationText) {
                    this.elements.annotationText.value = this.currentAnnotation;
                }

                this.hasUnsavedChanges = true;
                this.updateStatus('completed');
                
                // Show optimization section
                if (this.elements.optimizationSection) {
                    this.elements.optimizationSection.style.display = 'block';
                }

                this.app.showSuccess('Annotation generated successfully using DriveGuard VideoAnnotator');
            } else {
                throw new Error(response.error || 'Failed to generate annotation');
            }

        } catch (error) {
            console.error('Failed to generate annotation:', error);
            this.updateStatus('error');
            
            // Provide more helpful error messages
            let errorMessage = 'Failed to generate annotation';
            if (error.message.includes('404')) {
                errorMessage += ': Video clip not found. Please ensure the video file exists.';
            } else if (error.message.includes('timeout')) {
                errorMessage += ': Request timed out. Video processing can take several minutes for longer clips.';
            } else if (error.message.includes('network') || error.message.includes('fetch')) {
                errorMessage += ': Network error. Please check your connection and try again.';
            } else {
                errorMessage += ': ' + error.message;
            }
            
            this.app.showError(errorMessage);
        } finally {
            this.setLoading(false, '', this.elements.generateBtn);
            this.pendingRequests.delete(requestKey);
        }
    }

    async optimizeAnnotation() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.currentAnnotation) {
            this.app.showError('No annotation to optimize');
            return;
        }

        // Get feedback from inline textarea
        const userFeedback = this.elements.feedbackTextarea?.value?.trim();
        if (!userFeedback) {
            this.app.showError('Please enter optimization instructions');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        const requestKey = `optimize-${videoId}`;
        
        // Prevent duplicate requests
        if (this.pendingRequests.has(requestKey)) {
            console.log('Optimize annotation request already pending, ignoring duplicate');
            return;
        }
        
        this.pendingRequests.add(requestKey);

        try {
            // Only show loading on the optimize button
            this.setLoading(true, 'Optimizing annotation with AI...', this.elements.optimizeBtn);

            const response = await this.app.apiClient.optimizeAnnotation(videoId, userFeedback);

            if (response.success) {
                this.currentAnnotation = response.data.annotation;
                
                if (this.elements.annotationText) {
                    this.elements.annotationText.value = this.currentAnnotation;
                }

                this.hasUnsavedChanges = true;
                this.updateStatus('completed');
                
                // Clear feedback textarea after successful optimization
                if (this.elements.feedbackTextarea) {
                    this.elements.feedbackTextarea.value = '';
                }

                this.app.showSuccess('Annotation optimized successfully using LLM');
            } else {
                throw new Error(response.error || 'Failed to optimize annotation');
            }

        } catch (error) {
            console.error('Failed to optimize annotation:', error);
            
            // Provide more helpful error messages
            let errorMessage = 'Failed to optimize annotation';
            if (error.message.includes('timeout')) {
                errorMessage += ': Request timed out. LLM optimization can take time for complex instructions.';
            } else if (error.message.includes('network') || error.message.includes('fetch')) {
                errorMessage += ': Network error. Please check your connection and try again.';
            } else {
                errorMessage += ': ' + error.message;
            }
            
            this.app.showError(errorMessage);
        } finally {
            this.setLoading(false, '', this.elements.optimizeBtn);
            this.pendingRequests.delete(requestKey);
        }
    }


    async save() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.hasUnsavedChanges) {
            this.app.showSuccess('No changes to save');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        const annotation = this.elements.annotationText?.value || '';

        if (!annotation.trim()) {
            this.app.showError('Annotation cannot be empty');
            return;
        }

        try {
            // Only show loading on the save button
            this.setLoading(true, 'Saving annotation...', this.elements.saveBtn);

            const response = await this.app.apiClient.saveStepData(videoId, 1, {
                annotation: annotation
            });

            if (response.success) {
                this.currentAnnotation = annotation;
                this.hasUnsavedChanges = false;
                this.updateStatus('completed');
                
                // Advance to next step since this step is now completed
                this.app.advanceToNextStep();
                
                // Update app state
                if (this.app.state.currentClip.groundTruth) {
                    this.app.state.currentClip.groundTruth.annotation = annotation;
                }

                this.app.showSuccess('Annotation saved successfully');
                this.updateUI();
            } else {
                throw new Error(response.error || 'Failed to save annotation');
            }

        } catch (error) {
            console.error('Failed to save annotation:', error);
            this.app.showError('Failed to save annotation: ' + error.message);
        } finally {
            this.setLoading(false, '', this.elements.saveBtn);
        }
    }

    async reset() {
        if (!confirm('Are you sure you want to reset the annotation? Any unsaved changes will be lost.')) {
            return;
        }

        if (this.elements.annotationText) {
            this.elements.annotationText.value = this.currentAnnotation;
        }

        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    updateStatus(status) {
        if (this.elements.status) {
            this.elements.status.textContent = status;
            this.elements.status.className = `step-status ${status}`;
        }
    }

    setLoading(loading, message = 'Loading...', targetButton = null) {
        // If a specific button is targeted, only update that button
        if (targetButton) {
            if (loading) {
                targetButton.disabled = true;
                targetButton.dataset.originalText = targetButton.innerHTML;
                const spinner = message.includes('AI') ? 
                    '<i class="fas fa-brain"></i>' : 
                    '<i class="fas fa-spinner fa-spin"></i>';
                targetButton.innerHTML = `${spinner} ${message}`;
            } else {
                targetButton.disabled = false;
                // Fix for save button - explicitly restore the correct text
                if (targetButton.id === 'save-annotation-btn') {
                    targetButton.innerHTML = '<i class="fas fa-save"></i> Save';
                } else {
                    targetButton.innerHTML = targetButton.dataset.originalText || targetButton.innerHTML;
                }
            }
        } else {
            // Legacy behavior - update all buttons (only when no specific button is targeted)
            const buttons = [
                this.elements.generateBtn,
                this.elements.optimizeBtn,
                this.elements.saveBtn,
                this.elements.resetBtn
            ];

            buttons.forEach(btn => {
                if (btn) {
                    btn.disabled = loading;
                    if (loading) {
                        btn.dataset.originalText = btn.innerHTML;
                        const spinner = message.includes('AI') ? 
                            '<i class="fas fa-brain"></i>' : 
                            '<i class="fas fa-spinner fa-spin"></i>';
                        btn.innerHTML = `${spinner} ${message}`;
                    } else {
                        btn.innerHTML = btn.dataset.originalText || btn.innerHTML;
                    }
                }
            });
        }

        // Disable textarea during operations
        if (this.elements.annotationText) {
            this.elements.annotationText.disabled = loading;
            if (loading) {
                this.elements.annotationText.placeholder = 'Processing video with AI agents...';
            } else {
                this.elements.annotationText.placeholder = 'Waiting to be generated with AI agent...';
            }
        }

        if (loading) {
            this.updateStatus('in-progress');
        }

        // Show global loading if needed for longer operations
        if (loading && (message.includes('Generating') || message.includes('Optimizing'))) {
            this.app.showLoading(message);
        } else if (!loading) {
            this.app.hideLoading();
        }
    }

    updateUI() {
        // Save button stays enabled - users can save multiple times
        // Don't disable based on hasUnsavedChanges
        
        // Update reset button state
        if (this.elements.resetBtn) {
            this.elements.resetBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update character count if needed
        this.updateCharacterCount();
    }

    updateCharacterCount() {
        const textarea = this.elements.annotationText;
        if (!textarea) return;

        const text = textarea.value;
        const charCount = text.length;
        const wordCount = text.trim() ? text.trim().split(/\s+/).length : 0;

        // Find or create character count display
        let countDisplay = this.container.querySelector('.character-count');
        if (!countDisplay) {
            countDisplay = document.createElement('div');
            countDisplay.className = 'character-count';
            countDisplay.style.cssText = 'font-size: 0.75rem; color: #64748b; text-align: right; margin-top: 0.25rem;';
            textarea.parentNode.insertBefore(countDisplay, textarea.nextSibling);
        }

        countDisplay.textContent = `${charCount} characters, ${wordCount} words`;
    }

    hasUnsavedChanges() {
        return this.hasUnsavedChanges;
    }

    getValidationErrors() {
        const errors = [];
        
        const annotation = this.elements.annotationText?.value || '';
        
        if (!annotation.trim()) {
            errors.push('Annotation is required');
        } else if (annotation.length < 50) {
            errors.push('Annotation should be at least 50 characters');
        }

        return errors;
    }

    isValid() {
        return this.getValidationErrors().length === 0;
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.StepAnnotationComponent = StepAnnotationComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = StepAnnotationComponent;
}