/**
 * Step 5: Assessment Component
 * Handles final driving assessment generation and editing
 */

class StepAssessmentComponent {
    constructor(container, app) {
        this.container = container;
        this.app = app;
        this.currentAssessment = null;
        this.hasUnsavedChanges = false;
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Step Assessment component initialized');
    }

    initElements() {
        this.elements = {
            generateBtn: this.container.querySelector('#generate-assessment-btn'),
            assessmentForm: this.container.querySelector('#assessment-form'),
            assessmentPlaceholder: this.container.querySelector('#assessment-placeholder'),
            saveBtn: this.container.querySelector('#save-assessment-btn'),
            resetBtn: this.container.querySelector('#reset-assessment-btn'),
            status: this.container.querySelector('#step-5-status'),
            
            // Form elements for real assessment structure
            safetyScore: this.container.querySelector('#safety-score'),
            scoreDisplay: this.container.querySelector('#score-display'),
            riskLevel: this.container.querySelector('#risk-level'),
            overallEvaluation: this.container.querySelector('#overall-evaluation'),
            strengths: this.container.querySelector('#strengths'),
            weaknesses: this.container.querySelector('#weaknesses'),
            improvementAdvice: this.container.querySelector('#improvement-advice'),
            
            // Criteria toggle element (now links to standalone reference)
            criteriaToggle: this.container.querySelector('#score-criteria-toggle'),
            
            // Optimization elements
            optimizationSection: this.container.querySelector('#assessment-optimization-section'),
            optimizeFeedbackTextarea: this.container.querySelector('#assessment-optimization-feedback'),
            optimizeBtn: this.container.querySelector('#optimize-assessment-btn')
        };
    }

    setupEventListeners() {
        // Generate assessment button
        if (this.elements.generateBtn) {
            this.elements.generateBtn.addEventListener('click', () => this.generateAssessment());
        }

        // Optimization button
        if (this.elements.optimizeBtn) {
            this.elements.optimizeBtn.addEventListener('click', () => this.optimizeAssessment());
        }

        // Safety score slider
        if (this.elements.safetyScore) {
            this.elements.safetyScore.addEventListener('input', () => {
                this.updateScoreDisplay();
                this.autoUpdateRiskLevel();
                this.highlightActiveCriteria();
                this.hasUnsavedChanges = true;
                this.updateUI();
            });
        }
        
        // Criteria toggle (now handled by scoring reference component)
        // The button click is handled by ScoringReferenceComponent

        // Form change listeners for all assessment fields
        const formElements = [
            this.elements.riskLevel,
            this.elements.overallEvaluation,
            this.elements.strengths,
            this.elements.weaknesses,
            this.elements.improvementAdvice
        ];

        formElements.forEach(element => {
            if (element) {
                element.addEventListener('input', () => {
                    this.hasUnsavedChanges = true;
                    this.updateUI();
                });
            }
        });

        // Save button
        if (this.elements.saveBtn) {
            this.elements.saveBtn.addEventListener('click', () => this.save());
        }

        // Reset button
        if (this.elements.resetBtn) {
            this.elements.resetBtn.addEventListener('click', () => this.reset());
        }
    }

    async loadData(groundTruthData) {
        // Always reset state regardless of data presence
        if (groundTruthData && groundTruthData.assessment) {
            this.currentAssessment = groundTruthData.assessment;
            this.populateForm();
            
            // Show optimization section if assessment exists
            if (this.elements.optimizationSection) {
                this.elements.optimizationSection.style.display = 'block';
            }
        } else {
            this.currentAssessment = null; // Clear previous clip's data
            this.clearForm();
            
            // Hide optimization section when no assessment
            if (this.elements.optimizationSection) {
                this.elements.optimizationSection.style.display = 'none';
            }
        }
        
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    populateForm() {
        if (!this.currentAssessment) return;

        // Populate safety score (1-10 scale from DrivingMentor)
        if (this.elements.safetyScore && this.currentAssessment.safety_score !== undefined) {
            this.elements.safetyScore.value = this.currentAssessment.safety_score;
        }
        
        // Populate risk level
        if (this.elements.riskLevel && this.currentAssessment.risk_level) {
            this.elements.riskLevel.value = this.currentAssessment.risk_level;
        }
        
        // Populate overall evaluation
        if (this.elements.overallEvaluation && this.currentAssessment.overall_evaluation) {
            this.elements.overallEvaluation.value = this.currentAssessment.overall_evaluation;
        }
        
        // Populate strengths (convert array to text)
        if (this.elements.strengths && this.currentAssessment.strengths) {
            this.elements.strengths.value = this.formatArrayToText(this.currentAssessment.strengths);
        }
        
        // Populate weaknesses (convert array to text)
        if (this.elements.weaknesses && this.currentAssessment.weaknesses) {
            this.elements.weaknesses.value = this.formatArrayToText(this.currentAssessment.weaknesses);
        }
        
        // Populate improvement advice (convert array to text)
        if (this.elements.improvementAdvice && this.currentAssessment.improvement_advice) {
            this.elements.improvementAdvice.value = this.formatArrayToText(this.currentAssessment.improvement_advice);
        }

        // Update score display and related UI
        this.updateScoreDisplay();
        this.autoUpdateRiskLevel();
        this.highlightActiveCriteria();
    }

    updateScoreDisplay() {
        const score = this.elements.safetyScore?.value || 5;
        const scoreValue = parseInt(score);
        
        // Get risk level for this score
        const riskLevel = this.getRiskLevelForScore(scoreValue);
        
        // Update score display (1-10 scale)
        if (this.elements.scoreDisplay) {
            this.elements.scoreDisplay.textContent = `${scoreValue}/10`;
            
            // Update color based on risk level
            this.elements.scoreDisplay.className = 'score-display';
            this.elements.scoreDisplay.classList.add(`score-${riskLevel}`);
        }
    }
    
    getRiskLevelForScore(score) {
        if (score >= 1 && score <= 4) return 'critical';
        if (score >= 5 && score <= 7) return 'high';
        if (score === 8) return 'medium';
        if (score >= 9 && score <= 10) return 'low';
        return 'medium'; // fallback
    }
    
    autoUpdateRiskLevel() {
        if (!this.elements.riskLevel || !this.elements.safetyScore) return;
        
        const score = parseInt(this.elements.safetyScore.value);
        const riskLevel = this.getRiskLevelForScore(score);
        
        // Map to dropdown values
        const riskLevelMap = {
            'critical': 'critical',
            'high': 'high', 
            'medium': 'medium',
            'low': 'low'
        };
        
        this.elements.riskLevel.value = riskLevelMap[riskLevel];
    }
    
    
    highlightActiveCriteria() {
        if (!this.elements.safetyScore) return;
        
        const currentScore = parseInt(this.elements.safetyScore.value);
        
        // Highlight in the standalone scoring reference if available
        if (this.scoringReference) {
            this.scoringReference.highlightScore(currentScore);
        }
    }

    formatArrayToText(array) {
        // Convert array to newline-separated text for textarea
        if (!Array.isArray(array)) {
            return array || '';
        }
        return array.join('\n');
    }

    parseTextToArray(text) {
        // Convert textarea text to array, splitting by newlines
        if (!text || typeof text !== 'string') {
            return [];
        }
        return text.split('\n')
                  .map(line => line.trim())
                  .filter(line => line.length > 0);
    }

    async generateAssessment() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Generating assessment...');
            
            const response = await this.app.apiClient.generateAssessment(videoId);

            if (response.success) {
                this.currentAssessment = response.data.assessment;
                this.populateForm();
                this.hasUnsavedChanges = true;
                this.updateStatus('completed');
                this.app.showSuccess('Assessment generated successfully');
                
                // Show form
                this.showAssessmentForm();
                
            } else {
                throw new Error(response.error || 'Failed to generate assessment');
            }

        } catch (error) {
            console.error('Failed to generate assessment:', error);
            this.app.showError('Failed to generate assessment: ' + error.message);
            this.updateStatus('error');
        } finally {
            this.setLoading(false);
        }
    }

    async optimizeAssessment() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.currentAssessment) {
            this.app.showError('No assessment generated yet. Please generate an assessment first.');
            return;
        }

        const userFeedback = this.elements.optimizeFeedbackTextarea?.value?.trim();
        if (!userFeedback) {
            this.app.showError('Please provide optimization instructions before optimizing.');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        const requestKey = `optimize-${videoId}`;
        
        // Prevent duplicate requests
        if (this.pendingRequests && this.pendingRequests.has(requestKey)) {
            console.log('Optimize assessment request already pending, ignoring duplicate');
            return;
        }
        
        if (!this.pendingRequests) {
            this.pendingRequests = new Set();
        }
        this.pendingRequests.add(requestKey);

        try {
            // Only show loading on the optimize button
            this.setLoading(true, 'Optimizing assessment with AI...', this.elements.optimizeBtn);

            const response = await this.app.apiClient.optimizeAssessment(videoId, userFeedback);

            if (response.success) {
                this.currentAssessment = response.data.assessment;
                this.populateForm();
                this.hasUnsavedChanges = true;
                this.updateStatus('completed');
                
                // Clear feedback textarea after successful optimization
                if (this.elements.optimizeFeedbackTextarea) {
                    this.elements.optimizeFeedbackTextarea.value = '';
                }

                this.app.showSuccess('Assessment optimized successfully using AI');
            } else {
                throw new Error(response.error || 'Failed to optimize assessment');
            }

        } catch (error) {
            console.error('Failed to optimize assessment:', error);
            
            // Provide more helpful error messages
            let errorMessage = 'Failed to optimize assessment';
            if (error.message.includes('timeout')) {
                errorMessage += ': Request timed out. AI optimization can take time for complex instructions.';
            } else if (error.message.includes('network') || error.message.includes('fetch')) {
                errorMessage += ': Network error. Please check your connection and try again.';
            } else {
                errorMessage += ': ' + error.message;
            }
            
            this.app.showError(errorMessage);
        } finally {
            this.setLoading(false, '', this.elements.optimizeBtn);
            if (this.pendingRequests) {
                this.pendingRequests.delete(requestKey);
            }
        }
    }

    showAssessmentForm() {
        if (this.elements.assessmentForm) {
            this.elements.assessmentForm.style.display = 'block';
        }
        
        if (this.elements.assessmentPlaceholder) {
            this.elements.assessmentPlaceholder.style.display = 'none';
        }
        
        // Show optimization section when form is shown and assessment exists
        if (this.currentAssessment && this.elements.optimizationSection) {
            this.elements.optimizationSection.style.display = 'block';
        }
    }

    hideAssessmentForm() {
        if (this.elements.assessmentForm) {
            this.elements.assessmentForm.style.display = 'none';
        }
        
        if (this.elements.assessmentPlaceholder) {
            this.elements.assessmentPlaceholder.style.display = 'block';
        }
        
        // Hide optimization section when form is hidden
        if (this.elements.optimizationSection) {
            this.elements.optimizationSection.style.display = 'none';
        }
    }

    collectFormData() {
        return {
            safety_score: parseInt(this.elements.safetyScore?.value || 5),
            risk_level: this.elements.riskLevel?.value || 'medium',
            overall_evaluation: this.elements.overallEvaluation?.value || '',
            strengths: this.parseTextToArray(this.elements.strengths?.value || ''),
            weaknesses: this.parseTextToArray(this.elements.weaknesses?.value || ''),
            improvement_advice: this.parseTextToArray(this.elements.improvementAdvice?.value || '')
        };
    }

    async save() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.hasUnsavedChanges) {
            this.app.showSuccess('No changes to save');
            return;
        }

        // Validate assessment
        const errors = this.getValidationErrors();
        if (errors.length > 0) {
            this.app.showError('Please fix the following errors:\n' + errors.join('\n'));
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        const assessmentData = this.collectFormData();

        try {
            this.setLoading(true, 'Saving assessment...');

            const response = await this.app.apiClient.saveStepData(videoId, 5, {
                assessment: assessmentData
            });

            if (response.success) {
                this.currentAssessment = assessmentData;
                this.hasUnsavedChanges = false;
                this.updateStatus('completed');
                
                // Update app state
                if (this.app.state.currentClip.groundTruth) {
                    this.app.state.currentClip.groundTruth.assessment = assessmentData;
                }

                this.app.showSuccess('Assessment saved successfully');
            } else {
                throw new Error(response.error || 'Failed to save assessment');
            }

        } catch (error) {
            console.error('Failed to save assessment:', error);
            this.app.showError('Failed to save assessment: ' + error.message);
        } finally {
            this.setLoading(false);
        }
    }

    async exportGroundTruth() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Exporting ground truth...');
            
            const response = await this.app.apiClient.exportGroundTruth(videoId);

            if (response.success) {
                this.app.showSuccess('Ground truth exported successfully');
                
                // Optionally download the file
                if (response.data.download_url) {
                    const link = document.createElement('a');
                    link.href = response.data.download_url;
                    link.download = `${videoId}_ground_truth.json`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                }
            } else {
                throw new Error(response.error || 'Failed to export ground truth');
            }

        } catch (error) {
            console.error('Failed to export ground truth:', error);
            this.app.showError('Failed to export ground truth: ' + error.message);
        } finally {
            this.setLoading(false);
        }
    }

    async reset() {
        if (!confirm('Are you sure you want to reset the assessment? Any unsaved changes will be lost.')) {
            return;
        }

        // Reload from ground truth data
        if (this.app.state.currentClip?.groundTruth?.assessment) {
            this.currentAssessment = { ...this.app.state.currentClip.groundTruth.assessment };
            this.populateForm();
        } else {
            this.currentAssessment = null;
            this.clearForm();
            this.hideAssessmentForm();
        }

        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    clearForm() {
        if (this.elements.safetyScore) this.elements.safetyScore.value = 5;
        if (this.elements.riskLevel) this.elements.riskLevel.value = 'medium';
        if (this.elements.overallEvaluation) this.elements.overallEvaluation.value = '';
        if (this.elements.strengths) this.elements.strengths.value = '';
        if (this.elements.weaknesses) this.elements.weaknesses.value = '';
        if (this.elements.improvementAdvice) this.elements.improvementAdvice.value = '';
        
        this.updateScoreDisplay();
        this.autoUpdateRiskLevel();
        this.highlightActiveCriteria();
    }

    updateStatus(status) {
        if (this.elements.status) {
            this.elements.status.textContent = status;
            this.elements.status.className = `step-status ${status}`;
        }
    }

    setLoading(loading, message = 'Loading...', specificButton = null) {
        // Disable/enable buttons
        const buttons = [
            this.elements.generateBtn,
            this.elements.saveBtn,
            this.elements.resetBtn,
            this.elements.exportBtn,
            this.elements.optimizeBtn
        ];

        buttons.forEach(btn => {
            if (btn) {
                // If specific button is provided, only update that button, otherwise update all
                if (specificButton && btn !== specificButton) {
                    return;
                }
                
                btn.disabled = loading;
                if (loading) {
                    btn.dataset.originalText = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ' + message;
                } else {
                    btn.innerHTML = btn.dataset.originalText || btn.innerHTML;
                    
                    // Failsafe: Force reset to known good state for specific buttons
                    if (btn === this.elements.generateBtn) {
                        btn.innerHTML = '<i class="fas fa-chart-line"></i> Generate';
                    } else if (btn === this.elements.saveBtn) {
                        btn.innerHTML = '<i class="fas fa-save"></i> Save Assessment';
                    } else if (btn === this.elements.resetBtn) {
                        btn.innerHTML = '<i class="fas fa-undo"></i> Reset';
                    } else if (btn === this.elements.exportBtn) {
                        btn.innerHTML = '<i class="fas fa-download"></i> Export';
                    } else if (btn === this.elements.optimizeBtn) {
                        btn.innerHTML = '<i class="fas fa-magic"></i> Optimize with AI';
                    }
                }
            }
        });

        // Disable form elements
        const formElements = [
            this.elements.safetyScore,
            this.elements.riskLevel,
            this.elements.overallEvaluation,
            this.elements.strengths,
            this.elements.weaknesses,
            this.elements.improvementAdvice
        ];

        formElements.forEach(element => {
            if (element) {
                element.disabled = loading;
            }
        });

        if (loading) {
            this.updateStatus('in-progress');
        }
    }

    updateUI() {
        // Update save button state
        if (this.elements.saveBtn) {
            this.elements.saveBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update reset button state
        if (this.elements.resetBtn) {
            this.elements.resetBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update score display
        this.updateScoreDisplay();

        // Show/hide form based on whether we have an assessment
        if (this.currentAssessment) {
            this.showAssessmentForm();
        }
    }

    hasUnsavedChanges() {
        return this.hasUnsavedChanges;
    }

    getValidationErrors() {
        const errors = [];
        const data = this.collectFormData();
        
        if (data.safety_score < 1 || data.safety_score > 10) {
            errors.push('Safety score must be between 1 and 10');
        }
        
        if (!data.overall_evaluation || data.overall_evaluation.trim().length < 20) {
            errors.push('Overall evaluation must be at least 20 characters');
        }
        
        if (!data.strengths || data.strengths.length === 0) {
            errors.push('At least one strength must be identified');
        }
        
        if (!data.weaknesses || data.weaknesses.length === 0) {
            errors.push('At least one weakness must be identified');
        }
        
        if (!data.improvement_advice || data.improvement_advice.length === 0) {
            errors.push('At least one improvement recommendation must be provided');
        }
        
        if (!data.risk_level) {
            errors.push('Risk level is required');
        }

        return errors;
    }

    isValid() {
        return this.getValidationErrors().length === 0;
    }

    // Utility method to format rating display
    formatRating(rating) {
        const ratingMap = {
            'excellent': '⭐⭐⭐⭐⭐ Excellent',
            'good': '⭐⭐⭐⭐ Good',
            'fair': '⭐⭐⭐ Fair',
            'poor': '⭐⭐ Poor',
            'dangerous': '⭐ Dangerous'
        };
        return ratingMap[rating] || rating;
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.StepAssessmentComponent = StepAssessmentComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = StepAssessmentComponent;
}