/**
 * Step 2: Scenes Component
 * Handles scene extraction and manual editing
 */

class StepScenesComponent {
    constructor(container, app) {
        this.container = container;
        this.app = app;
        this.currentScenes = [];
        this.hasUnsavedChanges = false;
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Step Scenes component initialized');
    }

    initElements() {
        this.elements = {
            extractBtn: this.container.querySelector('#extract-scenes-btn'),
            scenesList: this.container.querySelector('#scenes-list'),
            addSceneBtn: this.container.querySelector('#add-scene-btn'),
            saveBtn: this.container.querySelector('#save-scenes-btn'),
            resetBtn: this.container.querySelector('#reset-scenes-btn'),
            status: this.container.querySelector('#step-2-status')
        };
    }

    setupEventListeners() {
        // Extract scenes button
        if (this.elements.extractBtn) {
            this.elements.extractBtn.addEventListener('click', () => this.extractScenes());
        }

        // Add scene button
        if (this.elements.addSceneBtn) {
            this.elements.addSceneBtn.addEventListener('click', () => this.addNewScene());
        }

        // Save button
        if (this.elements.saveBtn) {
            this.elements.saveBtn.addEventListener('click', () => this.save());
        }

        // Reset button
        if (this.elements.resetBtn) {
            this.elements.resetBtn.addEventListener('click', () => this.reset());
        }
    }

    async loadData(groundTruthData) {
        // Always reset state regardless of data presence
        if (groundTruthData && groundTruthData.scenes) {
            this.currentScenes = groundTruthData.scenes;
        } else {
            this.currentScenes = []; // Clear previous clip's data
        }
        
        this.renderScenes();
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    async extractScenes() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Extracting scenes...', this.elements.extractBtn);
            
            const response = await this.app.apiClient.extractScenes(videoId);

            if (response.success) {
                this.currentScenes = response.data.scenes;
                this.renderScenes();
                this.hasUnsavedChanges = true;
                this.updateStatus('completed');
                this.updateUI(); // Refresh button states including reset button
                
                // Notify other steps about new scenes (but don't auto-save to maintain user control)
                // This allows Steps 3&4 to preview the new scene structure
                this.app.syncScenesWithOtherSteps(this.currentScenes);
                
                this.app.showSuccess('Scenes extracted successfully');
            } else {
                throw new Error(response.error || 'Failed to extract scenes');
            }

        } catch (error) {
            console.error('Failed to extract scenes:', error);
            this.app.showError('Failed to extract scenes: ' + error.message);
            this.updateStatus('error');
        } finally {
            console.log('🔧 Extract Finally block executing - resetting extract button loading state');
            this.setLoading(false, '', this.elements.extractBtn);
            console.log('🔧 Extract button loading state reset completed');
            
            // Emergency failsafe: Reset button state after 500ms in case setLoading fails
            setTimeout(() => {
                if (this.elements.extractBtn && this.elements.extractBtn.disabled) {
                    console.log('🚨 Emergency failsafe triggered - force resetting extract button');
                    this.elements.extractBtn.disabled = false;
                    this.elements.extractBtn.innerHTML = '<i class="fas fa-magic"></i> Extract Scenes';
                }
            }, 500);
        }
    }

    renderScenes() {
        if (!this.elements.scenesList) return;

        if (this.currentScenes.length === 0) {
            this.elements.scenesList.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-film"></i>
                    <p>No scenes extracted yet</p>
                    <small>Use the "Extract Scenes" button to generate scenes from the annotation</small>
                </div>
            `;
            return;
        }

        const html = this.currentScenes.map((scene, index) => {
            const sceneText = typeof scene === 'string' ? scene : (scene.description || '');
            
            return `
                <div class="scene-item-compact" data-scene-index="${index}">
                    <span class="scene-number">Scene ${index + 1}</span>
                    <textarea 
                        class="scene-description-compact" 
                        rows="2"
                        placeholder="Describe what happens in this scene..."
                        onchange="app.stepComponents[2].updateScene(${index}, this.value)"
                    >${sceneText}</textarea>
                    <button class="btn-icon scene-delete-btn" onclick="app.stepComponents[2].deleteScene(${index})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            `;
        }).join('');

        this.elements.scenesList.innerHTML = html;
    }

    updateScene(index, value) {
        if (index >= 0 && index < this.currentScenes.length) {
            this.currentScenes[index] = value;
            this.hasUnsavedChanges = true;
            this.updateUI();
        }
    }

    addNewScene() {
        const newScene = '';

        this.currentScenes.push(newScene);
        this.renderScenes();
        this.hasUnsavedChanges = true;
        this.updateUI();

        // Scroll to the new scene
        setTimeout(() => {
            const newSceneElement = this.elements.scenesList.lastElementChild;
            if (newSceneElement) {
                newSceneElement.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                
                // Focus the description textarea
                const textarea = newSceneElement.querySelector('.scene-description');
                if (textarea) {
                    textarea.focus();
                }
            }
        }, 100);
    }

    deleteScene(index) {
        if (index >= 0 && index < this.currentScenes.length) {
            if (confirm('Are you sure you want to delete this scene?')) {
                this.currentScenes.splice(index, 1);
                this.renderScenes();
                this.hasUnsavedChanges = true;
                this.updateUI();
            }
        }
    }



    async save() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.hasUnsavedChanges) {
            this.app.showSuccess('No changes to save');
            return;
        }

        // Validate scenes
        const errors = this.getValidationErrors();
        if (errors.length > 0) {
            this.app.showError('Please fix the following errors:\n' + errors.join('\n'));
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Saving scenes...', this.elements.saveBtn);

            const response = await this.app.apiClient.saveStepData(videoId, 2, {
                scenes: this.currentScenes
            });

            if (response.success) {
                this.hasUnsavedChanges = false;
                this.updateStatus('completed');
                
                // Update app state
                if (this.app.state.currentClip.groundTruth) {
                    this.app.state.currentClip.groundTruth.scenes = this.currentScenes;
                }
                
                // Notify other steps about scene changes
                this.app.notifyStepDataChanged(2, { scenes: this.currentScenes });
                
                // Advance to next step since this step is now completed
                this.app.advanceToNextStep();

                this.app.showSuccess('Scenes saved successfully');
            } else {
                throw new Error(response.error || 'Failed to save scenes');
            }

        } catch (error) {
            console.error('Failed to save scenes:', error);
            this.app.showError('Failed to save scenes: ' + error.message);
        } finally {
            console.log('🔧 Finally block executing - resetting save button loading state');
            this.setLoading(false, '', this.elements.saveBtn);
            console.log('🔧 Save button loading state reset completed');
            
            // Emergency failsafe: Reset button state after 500ms in case setLoading fails
            setTimeout(() => {
                if (this.elements.saveBtn && this.elements.saveBtn.disabled) {
                    console.log('🚨 Emergency failsafe triggered - force resetting save button');
                    this.elements.saveBtn.disabled = false;
                    this.elements.saveBtn.innerHTML = '<i class="fas fa-save"></i> Save Scenes';
                }
            }, 500);
        }
    }

    async reset() {
        if (!confirm('Are you sure you want to delete ALL scenes?\n\nThis will permanently remove all current scenes from the interface. You will need to save to make this change permanent.')) {
            return;
        }

        // Clear all scenes completely
        this.currentScenes = [];

        this.renderScenes();
        this.hasUnsavedChanges = true; // Mark as changed since we're clearing scenes
        this.updateUI();
        
        // Show success message
        this.app.showSuccess('All scenes have been deleted. Save to make this change permanent.');
    }

    updateStatus(status) {
        if (this.elements.status) {
            this.elements.status.textContent = status;
            this.elements.status.className = `step-status ${status}`;
        }
    }

    setLoading(loading, message = 'Loading...', targetButton = null) {
        console.log(`🔧 setLoading called: loading=${loading}, message="${message}", targetButton=`, targetButton);
        
        if (targetButton) {
            console.log(`🔧 Target button found, current state: disabled=${targetButton.disabled}, innerHTML="${targetButton.innerHTML}"`);
            
            // Update only the specific target button
            targetButton.disabled = loading;
            
            if (loading) {
                targetButton.dataset.originalText = targetButton.innerHTML;
                targetButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ' + message;
                console.log(`🔧 Set to loading state: "${targetButton.innerHTML}"`);
            } else {
                const originalText = targetButton.dataset.originalText;
                console.log(`🔧 Resetting from loading, originalText="${originalText}"`);
                targetButton.innerHTML = originalText || targetButton.innerHTML;
                console.log(`🔧 Reset complete, new innerHTML="${targetButton.innerHTML}"`);
                
                // Failsafe: Force reset to known good state for specific buttons
                if (targetButton === this.elements.saveBtn) {
                    targetButton.innerHTML = '<i class="fas fa-save"></i> Save Scenes';
                    console.log('🔧 Applied failsafe reset for save button');
                } else if (targetButton === this.elements.extractBtn) {
                    targetButton.innerHTML = '<i class="fas fa-magic"></i> Extract Scenes';
                    console.log('🔧 Applied failsafe reset for extract button');
                }
            }
        } else {
            // Fallback: update all buttons (for backward compatibility)
            const buttons = [
                this.elements.extractBtn,
                this.elements.addSceneBtn,
                this.elements.saveBtn,
                this.elements.resetBtn
            ];

            buttons.forEach(btn => {
                if (btn) {
                    btn.disabled = loading;
                    if (loading) {
                        btn.dataset.originalText = btn.innerHTML;
                        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ' + message;
                    } else {
                        btn.innerHTML = btn.dataset.originalText || btn.innerHTML;
                    }
                }
            });
        }

        if (loading) {
            this.updateStatus('in-progress');
        }
    }

    updateUI() {
        // Update save button state
        if (this.elements.saveBtn) {
            this.elements.saveBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update reset button state - enable when there are scenes to delete
        if (this.elements.resetBtn) {
            this.elements.resetBtn.disabled = this.currentScenes.length === 0;
        }

        // Update add scene button
        if (this.elements.addSceneBtn) {
            this.elements.addSceneBtn.style.display = 'inline-flex';
        }
    }

    hasUnsavedChanges() {
        return this.hasUnsavedChanges;
    }

    getValidationErrors() {
        const errors = [];
        
        // Allow empty scenes array (for "Delete All Scenes" functionality)
        if (this.currentScenes.length === 0) {
            return errors; // No validation errors for empty scenes
        }

        this.currentScenes.forEach((scene, index) => {
            const sceneText = typeof scene === 'string' ? scene : (scene.description || '');
            
            if (!sceneText || sceneText.trim().length < 10) {
                errors.push(`Scene ${index + 1}: Description must be at least 10 characters`);
            }
        });

        return errors;
    }

    isValid() {
        return this.getValidationErrors().length === 0;
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.StepScenesComponent = StepScenesComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = StepScenesComponent;
}