/**
 * Step 3: Violations Component
 * Handles traffic violation analysis and editing
 */

class StepViolationsComponent {
    constructor(container, app) {
        this.container = container;
        this.app = app;
        this.currentViolations = [];
        this.hasUnsavedChanges = false;
        
        // DOM elements
        this.elements = {};
        
        this.init();
    }

    init() {
        this.initElements();
        this.setupEventListeners();
        console.log('✅ Step Violations component initialized');
    }

    initElements() {
        this.elements = {
            analyzeBtn: this.container.querySelector('#analyze-violations-btn'),
            violationsList: this.container.querySelector('#violations-list'),
            saveBtn: this.container.querySelector('#save-violations-btn'),
            resetBtn: this.container.querySelector('#reset-violations-btn'),
            status: this.container.querySelector('#step-3-status')
        };
    }

    setupEventListeners() {
        // Analyze violations button
        if (this.elements.analyzeBtn) {
            this.elements.analyzeBtn.addEventListener('click', () => this.analyzeViolations());
        }

        // Save button
        if (this.elements.saveBtn) {
            this.elements.saveBtn.addEventListener('click', () => this.save());
        }

        // Reset button
        if (this.elements.resetBtn) {
            this.elements.resetBtn.addEventListener('click', () => this.reset());
        }
    }

    async loadData(groundTruthData) {
        // Always reset state regardless of data presence
        if (groundTruthData && groundTruthData.violations) {
            this.currentViolations = groundTruthData.violations;
        } else {
            this.currentViolations = []; // Clear previous clip's data
        }
        
        this.renderViolations();
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    async analyzeViolations() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        const videoId = this.app.state.currentClip.video_id;
        console.log('Starting violations analysis for:', videoId);

        try {
            this.setLoading(true, 'Analyzing violations...');
            
            // Check if scenes are available before starting analysis
            const scenesComponent = this.app.stepComponents[2];
            if (!scenesComponent || !scenesComponent.currentScenes || scenesComponent.currentScenes.length === 0) {
                throw new Error('No scenes available. Please complete Step 2 (Extract Scenes) first.');
            }
            
            const response = await this.app.apiClient.analyzeViolations(videoId);
            console.log('API response received:', response);

            if (response.success) {
                // Validate response data structure
                if (!response.data || !Array.isArray(response.data.violations)) {
                    throw new Error('Invalid response format: violations data is missing or invalid');
                }

                console.log('Processing violations data:', response.data.violations);
                
                try {
                    // Process the response data
                    this.currentViolations = response.data.violations;
                    console.log('Set currentViolations:', this.currentViolations);
                    
                    this.renderViolations();
                    console.log('Rendered violations to UI');
                    
                    this.hasUnsavedChanges = true;
                    this.updateStatus('completed');
                    
                    // Update app state if available
                    if (this.app.state.currentClip.groundTruth) {
                        this.app.state.currentClip.groundTruth.violations = this.currentViolations;
                        console.log('Updated app state');
                    }
                    
                    this.app.showSuccess('Violations analyzed successfully');
                    console.log('✅ Violations analysis completed successfully');
                    
                } catch (processingError) {
                    console.error('Error processing successful response:', processingError);
                    throw new Error('Failed to process analysis results: ' + processingError.message);
                }
            } else {
                // Enhanced error handling for different error types
                let errorMessage = response.error || 'Failed to analyze violations';
                
                // Check for common error patterns and provide helpful messages
                if (errorMessage.includes('JSON') && errorMessage.includes('Extra data')) {
                    errorMessage = 'AI response parsing error. The violation analysis may have succeeded but the response format was invalid. Please try again.';
                } else if (errorMessage.includes('No scenes available')) {
                    errorMessage = 'No scenes found for analysis. Please complete Step 2 (Extract Scenes) first.';
                } else if (errorMessage.includes('Failed to analyze violations')) {
                    errorMessage = 'Analysis engine encountered an error. This may be due to complex scene content or temporary service issues. Please try again.';
                } else if (errorMessage.includes('timeout') || errorMessage.includes('network')) {
                    errorMessage = 'Network or timeout error. Please check your connection and try again.';
                }
                
                throw new Error(errorMessage);
            }

        } catch (error) {
            console.error('Failed to analyze violations:', error);
            
            // Provide contextual error messages
            let userMessage = error.message;
            if (error.message.includes('NetworkError') || error.message.includes('fetch')) {
                userMessage = 'Network error. Please check your connection and try again.';
            } else if (error.message.includes('No clip selected')) {
                userMessage = 'Please select a video clip first.';
            }
            
            this.app.showError('Failed to analyze violations: ' + userMessage);
            this.updateStatus('error');
        } finally {
            console.log('Resetting loading state');
            this.setLoading(false);
        }
    }

    renderViolations() {
        if (!this.elements.violationsList) return;

        if (this.currentViolations.length === 0) {
            this.elements.violationsList.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-shield-alt"></i>
                    <p>No traffic violations detected</p>
                    <small>Use the "Analyze Violations" button to check for traffic rule violations, or ensure scenes are extracted in Step 2</small>
                </div>
            `;
            return;
        }

        const html = this.currentViolations.map((violation, index) => {
            const violationStatus = violation.violation || 'not_found';
            const isFound = violationStatus === 'found';
            
            return `
                <div class="violation-item-compact ${isFound ? 'found' : 'not-found'}" data-violation-index="${index}">
                    <div class="violation-header-compact">
                        <span class="violation-number">Scene ${index + 1}</span>
                        <span class="violation-status-indicator ${isFound ? 'found' : 'not-found'}">
                            <i class="fas ${isFound ? 'fa-exclamation-triangle' : 'fa-check-circle'}"></i>
                            ${isFound ? 'Violation Found' : 'No Violation'}
                        </span>
                    </div>
                    <div class="violation-content-compact">
                        <div class="form-row-inline">
                            <label>Scene:</label>
                            <span class="scene-text-compact">${violation.scene || 'Scene description not available'}</span>
                        </div>
                        <div class="form-row-inline">
                            <label>Status:</label>
                            <select 
                                class="form-control-compact violation-status-select" 
                                onchange="app.stepComponents[3].updateViolation(${index}, 'violation', this.value)"
                            >
                                <option value="not_found" ${violationStatus === 'not_found' ? 'selected' : ''}>No Violation</option>
                                <option value="found" ${violationStatus === 'found' ? 'selected' : ''}>Violation Found</option>
                            </select>
                        </div>
                        <div class="form-row-block">
                            <label>Reason:</label>
                            <textarea 
                                class="form-control-compact violation-reason" 
                                rows="2"
                                placeholder="${isFound ? 'Describe the traffic violation...' : 'Explain why no violation was found...'}"
                                onchange="app.stepComponents[3].updateViolation(${index}, 'reason', this.value)"
                            >${violation.reason || ''}</textarea>
                        </div>
                    </div>
                </div>
            `;
        }).join('');

        this.elements.violationsList.innerHTML = html;
    }


    updateViolation(index, field, value) {
        if (index >= 0 && index < this.currentViolations.length) {
            this.currentViolations[index][field] = value;
            this.hasUnsavedChanges = true;
            this.updateUI();
            
            // Re-render if severity changed (affects styling)
            if (field === 'severity' || field === 'violation_type') {
                this.renderViolations();
            }
        }
    }

    // Sync with scenes from Step 2
    syncWithScenes(scenes) {
        console.log('Step 3: Syncing with scenes from Step 2:', scenes);
        
        if (!Array.isArray(scenes)) {
            console.warn('Step 3: Invalid scenes data received');
            return;
        }
        
        // Preserve existing violation data where possible
        const existingViolations = [...this.currentViolations];
        const newViolations = [];
        
        scenes.forEach((scene, index) => {
            const sceneText = typeof scene === 'string' ? scene : (scene.description || scene);
            
            // Try to find existing violation for this scene
            let existingViolation = null;
            
            // First try to match by scene text
            existingViolation = existingViolations.find(v => v.scene === sceneText);
            
            // If not found, try to match by index (if within range)
            if (!existingViolation && existingViolations[index]) {
                existingViolation = existingViolations[index];
            }
            
            if (existingViolation) {
                // Update scene text but preserve violation status and reason
                newViolations.push({
                    scene: sceneText,
                    violation: existingViolation.violation || 'not_found',
                    reason: existingViolation.reason || ''
                });
            } else {
                // Create new violation entry
                newViolations.push({
                    scene: sceneText,
                    violation: 'not_found',
                    reason: ''
                });
            }
        });
        
        this.currentViolations = newViolations;
        this.renderViolations();
        this.hasUnsavedChanges = true;
        this.updateUI();
        
        console.log('Step 3: Synced violations:', this.currentViolations);
    }

    // Removed: addNewViolation, deleteViolation, duplicateViolation
    // These methods are no longer needed as scenes are managed by Step 2


    async save() {
        if (!this.app.state.currentClip) {
            this.app.showError('No clip selected');
            return;
        }

        if (!this.hasUnsavedChanges) {
            this.app.showSuccess('No changes to save');
            return;
        }

        // Validate violations
        const errors = this.getValidationErrors();
        if (errors.length > 0) {
            this.app.showError('Please fix the following errors:\n' + errors.join('\n'));
            return;
        }

        const videoId = this.app.state.currentClip.video_id;

        try {
            this.setLoading(true, 'Saving violations...');

            const response = await this.app.apiClient.saveStepData(videoId, 3, {
                violations: this.currentViolations
            });

            if (response.success) {
                this.hasUnsavedChanges = false;
                this.updateStatus('completed');
                
                // Update app state
                if (this.app.state.currentClip.groundTruth) {
                    this.app.state.currentClip.groundTruth.violations = this.currentViolations;
                }

                this.app.showSuccess('Violations saved successfully');
            } else {
                throw new Error(response.error || 'Failed to save violations');
            }

        } catch (error) {
            console.error('Failed to save violations:', error);
            this.app.showError('Failed to save violations: ' + error.message);
        } finally {
            this.setLoading(false);
        }
    }

    async reset() {
        if (!confirm('Are you sure you want to reset all violations? Any unsaved changes will be lost.')) {
            return;
        }

        // Reload from ground truth data
        if (this.app.state.currentClip?.groundTruth?.violations) {
            this.currentViolations = [...this.app.state.currentClip.groundTruth.violations];
        } else {
            this.currentViolations = [];
        }

        this.renderViolations();
        this.hasUnsavedChanges = false;
        this.updateUI();
    }

    updateStatus(status) {
        if (this.elements.status) {
            this.elements.status.textContent = status;
            this.elements.status.className = `step-status ${status}`;
        }
    }

    setLoading(loading, message = 'Loading...') {
        // Disable/enable buttons
        const buttons = [
            this.elements.analyzeBtn,
            this.elements.saveBtn,
            this.elements.resetBtn
        ];

        buttons.forEach(btn => {
            if (btn) {
                btn.disabled = loading;
                if (loading) {
                    btn.dataset.originalText = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ' + message;
                } else {
                    btn.innerHTML = btn.dataset.originalText || btn.innerHTML;
                    
                    // Failsafe: Force reset to known good state for specific buttons
                    if (btn === this.elements.analyzeBtn) {
                        btn.innerHTML = '<i class="fas fa-search"></i> Analyze';
                    } else if (btn === this.elements.saveBtn) {
                        btn.innerHTML = '<i class="fas fa-save"></i> Save';
                    } else if (btn === this.elements.resetBtn) {
                        btn.innerHTML = '<i class="fas fa-undo"></i> Reset';
                    }
                }
            }
        });

        if (loading) {
            this.updateStatus('in-progress');
        }
    }

    updateUI() {
        // Update save button state
        if (this.elements.saveBtn) {
            this.elements.saveBtn.disabled = !this.hasUnsavedChanges;
        }

        // Update reset button state
        if (this.elements.resetBtn) {
            this.elements.resetBtn.disabled = !this.hasUnsavedChanges;
        }
    }

    hasUnsavedChanges() {
        return this.hasUnsavedChanges;
    }

    getValidationErrors() {
        const errors = [];
        
        this.currentViolations.forEach((violation, index) => {
            if (!violation.scene || violation.scene.trim().length < 5) {
                errors.push(`Violation ${index + 1}: Scene description must be at least 5 characters`);
            }
            
            if (!violation.violation || !['found', 'not_found'].includes(violation.violation)) {
                errors.push(`Violation ${index + 1}: Violation status must be 'found' or 'not_found'`);
            }
            
            if (!violation.reason || violation.reason.trim().length < 10) {
                errors.push(`Violation ${index + 1}: Reason must be at least 10 characters`);
            }
        });

        return errors;
    }

    isValid() {
        return this.getValidationErrors().length === 0;
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.StepViolationsComponent = StepViolationsComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = StepViolationsComponent;
}