/**
 * Video Player Component
 * Handles video playback using Video.js with custom controls
 */

class VideoPlayerComponent {
    constructor(videoElement) {
        this.videoElement = videoElement;
        this.player = null;
        this.currentVideoPath = null;
        
        // Player options
        this.playerOptions = {
            controls: true,
            responsive: true,
            fluid: true,
            playbackRates: [0.25, 0.5, 0.75, 1, 1.25, 1.5, 2],
            plugins: {
                // Add any Video.js plugins here
            }
        };
        
        // Event callbacks
        this.onLoadedCallback = null;
        this.onErrorCallback = null;
        this.onPlayCallback = null;
        this.onPauseCallback = null;
    }

    async init() {
        try {
            // Initialize Video.js player
            this.player = videojs(this.videoElement, this.playerOptions);
            
            // Set up event listeners
            this.setupEventListeners();
            
            // Hide video initially
            this.hide();
            
            console.log('✅ Video player initialized');
            
        } catch (error) {
            console.error('❌ Failed to initialize video player:', error);
            throw error;
        }
    }

    setupEventListeners() {
        if (!this.player) return;

        // Video loading events
        this.player.on('loadstart', () => {
            console.log('Video loading started');
        });

        this.player.on('loadeddata', () => {
            console.log('Video data loaded');
            this.show();
            if (this.onLoadedCallback) {
                this.onLoadedCallback();
            }
        });

        this.player.on('canplaythrough', () => {
            console.log('Video can play through');
        });

        // Playback events
        this.player.on('play', () => {
            console.log('Video playing');
            if (this.onPlayCallback) {
                this.onPlayCallback();
            }
        });

        this.player.on('pause', () => {
            console.log('Video paused');
            if (this.onPauseCallback) {
                this.onPauseCallback();
            }
        });

        this.player.on('ended', () => {
            console.log('Video ended');
        });

        this.player.on('timeupdate', () => {
            // Handle time updates if needed
            this.handleTimeUpdate();
        });

        // Error handling
        this.player.on('error', (event) => {
            const error = this.player.error();
            console.error('Video player error:', error);
            this.handleError(error);
        });

        // Volume/mute events
        this.player.on('volumechange', () => {
            const volume = this.player.volume();
            const muted = this.player.muted();
            this.handleVolumeChange(volume, muted);
        });

        // Fullscreen events
        this.player.on('fullscreenchange', () => {
            const isFullscreen = this.player.isFullscreen();
            this.handleFullscreenChange(isFullscreen);
        });
    }

    /**
     * Load a video file
     * @param {string} videoPath - Path to the video file
     * @returns {Promise<void>}
     */
    async loadVideo(videoPath) {
        if (!this.player) {
            throw new Error('Video player not initialized');
        }

        if (this.currentVideoPath === videoPath) {
            console.log('Video already loaded:', videoPath);
            return;
        }

        try {
            console.log('Loading video:', videoPath);
            
            // Reset player state
            this.player.pause();
            this.player.currentTime(0);
            
            // Set video source
            this.player.src({
                type: 'video/mp4',
                src: videoPath
            });

            // Wait for video to load
            await new Promise((resolve, reject) => {
                const onLoaded = () => {
                    this.player.off('loadeddata', onLoaded);
                    this.player.off('error', onError);
                    resolve();
                };

                const onError = () => {
                    this.player.off('loadeddata', onLoaded);
                    this.player.off('error', onError);
                    reject(new Error('Failed to load video'));
                };

                this.player.on('loadeddata', onLoaded);
                this.player.on('error', onError);
            });

            this.currentVideoPath = videoPath;
            console.log('✅ Video loaded successfully');

        } catch (error) {
            console.error('❌ Failed to load video:', error);
            this.handleError(error);
            throw error;
        }
    }

    /**
     * Play the video
     */
    play() {
        if (this.player && !this.player.paused()) {
            return this.player.play();
        }
    }

    /**
     * Pause the video
     */
    pause() {
        if (this.player && !this.player.paused()) {
            this.player.pause();
        }
    }

    /**
     * Stop the video (pause and reset to beginning)
     */
    stop() {
        if (this.player) {
            this.player.pause();
            this.player.currentTime(0);
        }
    }

    /**
     * Seek to specific time
     * @param {number} time - Time in seconds
     */
    seekTo(time) {
        if (this.player && time >= 0) {
            this.player.currentTime(time);
        }
    }

    /**
     * Get current playback time
     * @returns {number} Current time in seconds
     */
    getCurrentTime() {
        return this.player ? this.player.currentTime() : 0;
    }

    /**
     * Get video duration
     * @returns {number} Duration in seconds
     */
    getDuration() {
        return this.player ? this.player.duration() : 0;
    }

    /**
     * Set playback rate
     * @param {number} rate - Playback rate (0.25, 0.5, 1, 1.5, 2, etc.)
     */
    setPlaybackRate(rate) {
        if (this.player && rate > 0) {
            this.player.playbackRate(rate);
        }
    }

    /**
     * Get current playback rate
     * @returns {number} Current playback rate
     */
    getPlaybackRate() {
        return this.player ? this.player.playbackRate() : 1;
    }

    /**
     * Set volume
     * @param {number} volume - Volume level (0-1)
     */
    setVolume(volume) {
        if (this.player && volume >= 0 && volume <= 1) {
            this.player.volume(volume);
        }
    }

    /**
     * Get current volume
     * @returns {number} Volume level (0-1)
     */
    getVolume() {
        return this.player ? this.player.volume() : 1;
    }

    /**
     * Mute/unmute video
     * @param {boolean} muted - Whether to mute
     */
    setMuted(muted) {
        if (this.player) {
            this.player.muted(muted);
        }
    }

    /**
     * Check if video is muted
     * @returns {boolean} Whether video is muted
     */
    isMuted() {
        return this.player ? this.player.muted() : false;
    }

    /**
     * Check if video is paused
     * @returns {boolean} Whether video is paused
     */
    isPaused() {
        return this.player ? this.player.paused() : true;
    }

    /**
     * Enter/exit fullscreen
     * @param {boolean} fullscreen - Whether to enter fullscreen
     */
    setFullscreen(fullscreen) {
        if (this.player) {
            if (fullscreen) {
                this.player.requestFullscreen();
            } else {
                this.player.exitFullscreen();
            }
        }
    }

    /**
     * Check if video is in fullscreen
     * @returns {boolean} Whether video is in fullscreen
     */
    isFullscreen() {
        return this.player ? this.player.isFullscreen() : false;
    }

    /**
     * Show video player
     */
    show() {
        if (this.videoElement) {
            this.videoElement.style.display = 'block';
            
            // Hide placeholder
            const placeholder = document.getElementById('video-placeholder');
            if (placeholder) {
                placeholder.style.display = 'none';
            }
        }
    }

    /**
     * Hide video player
     */
    hide() {
        if (this.videoElement) {
            this.videoElement.style.display = 'none';
            
            // Show placeholder
            const placeholder = document.getElementById('video-placeholder');
            if (placeholder) {
                placeholder.style.display = 'flex';
            }
        }
    }

    /**
     * Handle time updates
     */
    handleTimeUpdate() {
        const currentTime = this.getCurrentTime();
        const duration = this.getDuration();
        
        // Update any time-based UI elements
        this.updateTimeDisplay(currentTime, duration);
        
        // Handle any time-based annotations or markers
        this.updateVideoMarkers(currentTime);
    }

    /**
     * Update time display (if custom controls are needed)
     */
    updateTimeDisplay(currentTime, duration) {
        // Format time as MM:SS
        const formatTime = (time) => {
            const minutes = Math.floor(time / 60);
            const seconds = Math.floor(time % 60);
            return `${minutes}:${seconds.toString().padStart(2, '0')}`;
        };

        // Update custom time displays if they exist
        const timeDisplay = document.querySelector('.custom-time-display');
        if (timeDisplay) {
            timeDisplay.textContent = `${formatTime(currentTime)} / ${formatTime(duration)}`;
        }
    }

    /**
     * Update video markers based on current time
     */
    updateVideoMarkers(currentTime) {
        // This could be used to highlight current scene or annotation
        const markers = document.querySelectorAll('.video-marker');
        markers.forEach(marker => {
            const startTime = parseFloat(marker.dataset.startTime) || 0;
            const endTime = parseFloat(marker.dataset.endTime) || 0;
            const isActive = currentTime >= startTime && currentTime <= endTime;
            marker.classList.toggle('active', isActive);
        });
    }

    /**
     * Handle volume change
     */
    handleVolumeChange(volume, muted) {
        console.log(`Volume changed: ${volume} (muted: ${muted})`);
        
        // Update any custom volume controls
        const volumeDisplay = document.querySelector('.custom-volume-display');
        if (volumeDisplay) {
            volumeDisplay.textContent = muted ? 'Muted' : `${Math.round(volume * 100)}%`;
        }
    }

    /**
     * Handle fullscreen change
     */
    handleFullscreenChange(isFullscreen) {
        console.log(`Fullscreen: ${isFullscreen}`);
        
        // Update UI for fullscreen mode
        document.body.classList.toggle('video-fullscreen', isFullscreen);
    }

    /**
     * Handle video errors
     */
    handleError(error) {
        console.error('Video player error:', error);
        
        // Show error state
        this.showError(error);
        
        if (this.onErrorCallback) {
            this.onErrorCallback(error);
        }
    }

    /**
     * Show error message
     */
    showError(error) {
        const placeholder = document.getElementById('video-placeholder');
        if (placeholder) {
            placeholder.innerHTML = `
                <i class="fas fa-exclamation-triangle"></i>
                <p>Failed to load video</p>
                <small>${error.message || 'Unknown error occurred'}</small>
            `;
            placeholder.style.display = 'flex';
        }
        
        this.hide();
    }

    /**
     * Handle responsive resize
     */
    handleResize() {
        if (this.player) {
            // Video.js handles responsive behavior automatically
            // but we can add custom logic here if needed
            console.log('Video player resized');
        }
    }

    /**
     * Set event callbacks
     */
    onLoaded(callback) {
        this.onLoadedCallback = callback;
    }

    onError(callback) {
        this.onErrorCallback = callback;
    }

    onPlay(callback) {
        this.onPlayCallback = callback;
    }

    onPause(callback) {
        this.onPauseCallback = callback;
    }

    /**
     * Take screenshot of current video frame
     * @returns {string} Base64 encoded image data
     */
    takeScreenshot() {
        if (!this.player) return null;

        try {
            const canvas = document.createElement('canvas');
            const video = this.player.el().querySelector('video');
            
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            
            const ctx = canvas.getContext('2d');
            ctx.drawImage(video, 0, 0);
            
            return canvas.toDataURL('image/png');
        } catch (error) {
            console.error('Failed to take screenshot:', error);
            return null;
        }
    }

    /**
     * Dispose of video player
     */
    dispose() {
        if (this.player) {
            this.player.dispose();
            this.player = null;
            this.currentVideoPath = null;
            console.log('Video player disposed');
        }
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.VideoPlayerComponent = VideoPlayerComponent;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = VideoPlayerComponent;
}