/**
 * API Client for backend communication
 * Handles all HTTP requests to the FastAPI backend
 */

class APIClient {
    constructor(baseURL = '') {
        this.baseURL = baseURL;
        this.defaultHeaders = {
            'Content-Type': 'application/json',
        };
    }

    /**
     * Make a GET request
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Query parameters
     * @returns {Promise<Object>} Response data
     */
    async get(endpoint, params = {}) {
        const url = this.buildURL(endpoint, params);
        
        try {
            const response = await fetch(url, {
                method: 'GET',
                headers: this.defaultHeaders,
            });

            return await this.handleResponse(response);
        } catch (error) {
            console.error('GET request failed:', error);
            throw new Error(`GET ${endpoint}: ${error.message}`);
        }
    }

    /**
     * Make a POST request
     * @param {string} endpoint - API endpoint
     * @param {Object} data - Request body data
     * @returns {Promise<Object>} Response data
     */
    async post(endpoint, data = {}) {
        try {
            const response = await fetch(this.baseURL + endpoint, {
                method: 'POST',
                headers: this.defaultHeaders,
                body: JSON.stringify(data),
            });

            return await this.handleResponse(response);
        } catch (error) {
            console.error('POST request failed:', error);
            throw new Error(`POST ${endpoint}: ${error.message}`);
        }
    }

    /**
     * Make a PUT request
     * @param {string} endpoint - API endpoint
     * @param {Object} data - Request body data
     * @returns {Promise<Object>} Response data
     */
    async put(endpoint, data = {}) {
        try {
            const response = await fetch(this.baseURL + endpoint, {
                method: 'PUT',
                headers: this.defaultHeaders,
                body: JSON.stringify(data),
            });

            return await this.handleResponse(response);
        } catch (error) {
            console.error('PUT request failed:', error);
            throw new Error(`PUT ${endpoint}: ${error.message}`);
        }
    }

    /**
     * Make a DELETE request
     * @param {string} endpoint - API endpoint
     * @returns {Promise<Object>} Response data
     */
    async delete(endpoint) {
        try {
            const response = await fetch(this.baseURL + endpoint, {
                method: 'DELETE',
                headers: this.defaultHeaders,
            });

            return await this.handleResponse(response);
        } catch (error) {
            console.error('DELETE request failed:', error);
            throw new Error(`DELETE ${endpoint}: ${error.message}`);
        }
    }

    /**
     * Build URL with query parameters
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Query parameters
     * @returns {string} Complete URL
     */
    buildURL(endpoint, params = {}) {
        const url = new URL(this.baseURL + endpoint, window.location.origin);
        
        Object.keys(params).forEach(key => {
            if (params[key] !== null && params[key] !== undefined) {
                url.searchParams.append(key, params[key]);
            }
        });

        return url.toString();
    }

    /**
     * Handle fetch response
     * @param {Response} response - Fetch response object
     * @returns {Promise<Object>} Parsed response data
     */
    async handleResponse(response) {
        // Handle different content types
        const contentType = response.headers.get('content-type');
        let data;

        if (contentType && contentType.includes('application/json')) {
            data = await response.json();
        } else {
            data = await response.text();
        }

        // Handle HTTP errors
        if (!response.ok) {
            const error = new Error(`HTTP ${response.status}: ${response.statusText}`);
            error.status = response.status;
            error.data = data;
            throw error;
        }

        return data;
    }

    /**
     * Session API methods
     */
    async getCurrentSession() {
        return await this.get('/api/session/current');
    }

    async navigateToPosition(videoId, step) {
        return await this.post('/api/session/navigate', {
            video_id: videoId,
            step: step
        });
    }

    async getSessionStatistics() {
        return await this.get('/api/session/statistics');
    }

    /**
     * Clips API methods
     */
    async getClips() {
        return await this.get('/api/clips');
    }

    async getClip(videoId) {
        return await this.get(`/api/clips/${videoId}`);
    }

    async getClipProgress(videoId) {
        return await this.get(`/api/clips/${videoId}/progress`);
    }

    async getGroundTruth(videoId) {
        return await this.get(`/api/clips/${videoId}/ground-truth`);
    }

    async streamVideo(videoId) {
        return `${this.baseURL}/api/clips/${videoId}/stream`;
    }

    /**
     * Pipeline execution methods
     */
    async generateAnnotation(videoId, options = {}) {
        return await this.post(`/api/clips/${videoId}/step/1/generate`, {
            optimize: options.optimize || false,
            user_feedback: options.user_feedback || null
        });
    }

    async optimizeAnnotation(videoId, userInstructions) {
        return await this.post(`/api/clips/${videoId}/step/1/optimize`, {
            user_instructions: userInstructions
        });
    }

    async extractScenes(videoId) {
        return await this.post(`/api/clips/${videoId}/step/2/extract`);
    }

    async analyzeViolations(videoId) {
        return await this.post(`/api/clips/${videoId}/step/3/analyze`);
    }

    async analyzeAccidents(videoId) {
        return await this.post(`/api/clips/${videoId}/step/4/analyze`);
    }

    async generateAssessment(videoId) {
        return await this.post(`/api/clips/${videoId}/step/5/generate`);
    }

    async optimizeAssessment(videoId, userInstructions) {
        return await this.post(`/api/clips/${videoId}/step/5/optimize`, {
            user_instructions: userInstructions
        });
    }

    /**
     * Data persistence methods
     */
    async saveStepData(videoId, step, data) {
        return await this.put(`/api/clips/${videoId}/step/${step}`, {
            data: data
        });
    }

    async exportGroundTruth(videoId) {
        return await this.post(`/api/clips/${videoId}/export`);
    }

    /**
     * Utility methods
     */
    async createBackup() {
        return await this.post('/api/backup');
    }

    async getRecentLogs() {
        return await this.get('/api/logs');
    }

    async healthCheck() {
        return await this.get('/api/health');
    }

    /**
     * Request with loading state
     * @param {Function} requestFn - Function that returns a Promise
     * @param {string} loadingMessage - Message to show while loading
     * @returns {Promise<Object>} Request result
     */
    async withLoading(requestFn, loadingMessage = 'Processing...') {
        // Show loading if app is available
        if (window.app && typeof window.app.showLoading === 'function') {
            window.app.showLoading(loadingMessage);
        }

        try {
            const result = await requestFn();
            return result;
        } finally {
            // Hide loading if app is available
            if (window.app && typeof window.app.hideLoading === 'function') {
                window.app.hideLoading();
            }
        }
    }

    /**
     * Batch requests
     * @param {Array<Function>} requests - Array of request functions
     * @returns {Promise<Array>} Array of results
     */
    async batch(requests) {
        try {
            const results = await Promise.allSettled(
                requests.map(requestFn => requestFn())
            );

            return results.map(result => {
                if (result.status === 'fulfilled') {
                    return { success: true, data: result.value };
                } else {
                    return { success: false, error: result.reason.message };
                }
            });
        } catch (error) {
            console.error('Batch request failed:', error);
            throw error;
        }
    }

    /**
     * Retry failed requests
     * @param {Function} requestFn - Request function to retry
     * @param {number} maxRetries - Maximum number of retries
     * @param {number} delay - Delay between retries in ms
     * @returns {Promise<Object>} Request result
     */
    async retry(requestFn, maxRetries = 3, delay = 1000) {
        let lastError;

        for (let attempt = 0; attempt <= maxRetries; attempt++) {
            try {
                return await requestFn();
            } catch (error) {
                lastError = error;
                
                if (attempt < maxRetries) {
                    console.warn(`Request attempt ${attempt + 1} failed, retrying in ${delay}ms...`);
                    await new Promise(resolve => setTimeout(resolve, delay));
                    delay *= 2; // Exponential backoff
                } else {
                    console.error(`Request failed after ${maxRetries + 1} attempts`);
                }
            }
        }

        throw lastError;
    }

    /**
     * Cancel ongoing requests (if needed)
     */
    cancelRequests() {
        // This would require implementing AbortController for each request
        // For now, just log that cancellation was requested
        console.log('Request cancellation requested');
    }
}

// Create global API client instance
if (typeof window !== 'undefined') {
    window.APIClient = APIClient;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = APIClient;
}