/**
 * Progress Tracker Utility
 * Tracks completion status across all steps and clips
 */

class ProgressTracker {
    constructor() {
        this.clipProgress = new Map(); // videoId -> progress data
        this.sessionStats = {
            totalClips: 0,
            completedClips: 0,
            inProgressClips: 0,
            startTime: Date.now()
        };
    }

    /**
     * Update progress for a specific clip
     * @param {string} videoId - Video identifier
     * @param {Object} progressData - Progress data for the clip
     */
    updateClipProgress(videoId, progressData) {
        this.clipProgress.set(videoId, {
            ...progressData,
            lastUpdated: Date.now()
        });
        
        this.recalculateStats();
    }

    /**
     * Get progress for a specific clip
     * @param {string} videoId - Video identifier
     * @returns {Object} Progress data or null
     */
    getClipProgress(videoId) {
        return this.clipProgress.get(videoId) || null;
    }

    /**
     * Mark a step as completed for a clip
     * @param {string} videoId - Video identifier
     * @param {number} step - Step number (1-5)
     */
    markStepCompleted(videoId, step) {
        let progress = this.clipProgress.get(videoId) || {
            stepsCompleted: {},
            totalSteps: 5,
            completedSteps: 0,
            isComplete: false
        };

        if (!progress.stepsCompleted[step]) {
            progress.stepsCompleted[step] = {
                completed: true,
                timestamp: Date.now()
            };
            
            progress.completedSteps = Object.keys(progress.stepsCompleted).length;
            progress.isComplete = progress.completedSteps === progress.totalSteps;
            progress.lastUpdated = Date.now();
            
            this.clipProgress.set(videoId, progress);
            this.recalculateStats();
        }
    }

    /**
     * Mark a step as in progress for a clip
     * @param {string} videoId - Video identifier
     * @param {number} step - Step number (1-5)
     */
    markStepInProgress(videoId, step) {
        let progress = this.clipProgress.get(videoId) || {
            stepsCompleted: {},
            totalSteps: 5,
            completedSteps: 0,
            isComplete: false,
            currentStep: step
        };

        progress.currentStep = step;
        progress.lastUpdated = Date.now();
        
        this.clipProgress.set(videoId, progress);
    }

    /**
     * Get completion percentage for a clip
     * @param {string} videoId - Video identifier
     * @returns {number} Percentage (0-100)
     */
    getClipCompletionPercentage(videoId) {
        const progress = this.clipProgress.get(videoId);
        if (!progress) return 0;
        
        return Math.round((progress.completedSteps / progress.totalSteps) * 100);
    }

    /**
     * Get overall session statistics
     * @returns {Object} Session statistics
     */
    getSessionStats() {
        return {
            ...this.sessionStats,
            duration: Date.now() - this.sessionStats.startTime,
            completionRate: this.sessionStats.totalClips > 0 
                ? (this.sessionStats.completedClips / this.sessionStats.totalClips) * 100 
                : 0
        };
    }

    /**
     * Set total number of clips in session
     * @param {number} total - Total number of clips
     */
    setTotalClips(total) {
        this.sessionStats.totalClips = total;
        this.recalculateStats();
    }

    /**
     * Recalculate session statistics
     */
    recalculateStats() {
        let completed = 0;
        let inProgress = 0;

        this.clipProgress.forEach(progress => {
            if (progress.isComplete) {
                completed++;
            } else if (progress.completedSteps > 0) {
                inProgress++;
            }
        });

        this.sessionStats.completedClips = completed;
        this.sessionStats.inProgressClips = inProgress;
    }

    /**
     * Get progress summary for all clips
     * @returns {Array} Array of clip progress summaries
     */
    getAllClipProgress() {
        const progressList = [];
        
        this.clipProgress.forEach((progress, videoId) => {
            progressList.push({
                videoId,
                completionPercentage: this.getClipCompletionPercentage(videoId),
                completedSteps: progress.completedSteps,
                totalSteps: progress.totalSteps,
                isComplete: progress.isComplete,
                currentStep: progress.currentStep,
                lastUpdated: progress.lastUpdated
            });
        });

        return progressList.sort((a, b) => b.lastUpdated - a.lastUpdated);
    }

    /**
     * Get next recommended clip to work on
     * @param {Array} availableClips - Available clips to choose from
     * @returns {string|null} Video ID of recommended clip
     */
    getNextRecommendedClip(availableClips) {
        if (!availableClips || availableClips.length === 0) {
            return null;
        }

        // Find incomplete clips
        const incompleteClips = availableClips.filter(clip => {
            const progress = this.clipProgress.get(clip.video_id);
            return !progress || !progress.isComplete;
        });

        if (incompleteClips.length === 0) {
            return null; // All clips completed
        }

        // Sort by progress (least progress first)
        incompleteClips.sort((a, b) => {
            const progressA = this.clipProgress.get(a.video_id);
            const progressB = this.clipProgress.get(b.video_id);
            
            const stepsA = progressA ? progressA.completedSteps : 0;
            const stepsB = progressB ? progressB.completedSteps : 0;
            
            return stepsA - stepsB;
        });

        return incompleteClips[0].video_id;
    }

    /**
     * Export progress data for backup
     * @returns {Object} Progress data
     */
    exportProgress() {
        return {
            clipProgress: Object.fromEntries(this.clipProgress),
            sessionStats: this.sessionStats,
            exportedAt: Date.now()
        };
    }

    /**
     * Import progress data from backup
     * @param {Object} progressData - Previously exported progress data
     */
    importProgress(progressData) {
        if (progressData.clipProgress) {
            this.clipProgress = new Map(Object.entries(progressData.clipProgress));
        }
        
        if (progressData.sessionStats) {
            this.sessionStats = {
                ...this.sessionStats,
                ...progressData.sessionStats
            };
        }
        
        this.recalculateStats();
    }

    /**
     * Reset all progress data
     */
    reset() {
        this.clipProgress.clear();
        this.sessionStats = {
            totalClips: 0,
            completedClips: 0,
            inProgressClips: 0,
            startTime: Date.now()
        };
    }

    /**
     * Get progress statistics for display
     * @returns {Object} Display-friendly statistics
     */
    getProgressStats() {
        const stats = this.getSessionStats();
        const totalTime = stats.duration;
        const hours = Math.floor(totalTime / (1000 * 60 * 60));
        const minutes = Math.floor((totalTime % (1000 * 60 * 60)) / (1000 * 60));

        return {
            totalClips: stats.totalClips,
            completedClips: stats.completedClips,
            inProgressClips: stats.inProgressClips,
            completionRate: Math.round(stats.completionRate),
            sessionDuration: `${hours}:${minutes.toString().padStart(2, '0')}`,
            averageTimePerClip: stats.completedClips > 0 
                ? Math.round(totalTime / (stats.completedClips * 1000 * 60)) + ' min'
                : 'N/A'
        };
    }
}

// Export for global use
if (typeof window !== 'undefined') {
    window.ProgressTracker = ProgressTracker;
}

// Export for Node.js modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ProgressTracker;
}