"""
Configuration management for system output generation.
"""

import sys
from pathlib import Path
from typing import Dict, List, Optional
from dataclasses import dataclass

# Add project root to path for imports
project_root = Path(__file__).parent.parent.parent.parent
sys.path.append(str(project_root))

from src.utils.settings import settings


def get_model_dir_name(model_id: str) -> str:
    """Extract model name for use as directory name from full model ID.
    
    Args:
        model_id: Full model ID (e.g., "openai:gpt-4o", "gateway:routeway-discount/gpt-5")
        
    Returns:
        Model name for directory (e.g., "gpt-4o", "gpt-5")
    
    Examples:
        "openai:gpt-4o" → "gpt-4o"
        "groq:llama-3.3-70b-versatile" → "llama-3.3-70b-versatile"  
        "gateway:routeway-discount/gpt-5" → "gpt-5"
        "gateway:xai/grok-4" → "grok-4"
    """
    if ":" not in model_id:
        return model_id
    
    platform, model_part = model_id.split(":", 1)
    
    if platform == "gateway":
        # For gateway models, extract just the model name after the last slash
        # e.g., "routeway-discount/gpt-5" → "gpt-5"
        if "/" in model_part:
            return model_part.split("/")[-1]
        return model_part
    else:
        # For other platforms, return the model name directly
        # e.g., "gpt-4o" → "gpt-4o"
        return model_part


@dataclass
class GenerationConfig:
    """Configuration for system output generation."""
    
    # Base paths
    project_root: Path
    dashcam_videos_dir: Path
    output_base_dir: Path
    
    # Component settings
    component: str
    model_id: Optional[str] = None  # API model ID (e.g., "gpt-4o")
    driveguard_model_id: Optional[str] = None  # DriveGuard model ID (e.g., "openai:gpt-4o")
    
    # Processing settings
    fps: int = 2
    batch_size: int = 10
    max_retries: int = 3
    process_all_videos: bool = False  # If True, process all videos; if False, only those with ground truth
    
    # Output settings
    overwrite_existing: bool = False
    save_metadata: bool = True
    
    def __post_init__(self):
        """Validate configuration after initialization."""
        if not self.dashcam_videos_dir.exists():
            raise ValueError(f"Dashcam videos directory not found: {self.dashcam_videos_dir}")
        
        # Create output directory if it doesn't exist
        self.output_base_dir.mkdir(parents=True, exist_ok=True)
    
    @property
    def ground_truth_dir(self) -> Path:
        """Get the ground truth directory."""
        return self.project_root / settings.app.path["data"] / "evaluation" / "ground_truth"
    
    @property
    def component_output_dir(self) -> Path:
        """Get the output directory for the current component."""
        return self.output_base_dir / self.component
    
    @property
    def model_output_dir(self) -> Path:
        """Get the output directory for the current model."""
        full_model_id = self.driveguard_model_id or self.model_id
        if not full_model_id:
            raise ValueError("Either model_id or driveguard_model_id must be set to get model output directory")
        
        model_dir_name = get_model_dir_name(full_model_id)
        return self.component_output_dir / model_dir_name
    
    def get_video_list(self, video_filter: Optional[List[str]] = None) -> List[Path]:
        """Get list of videos to process."""
        all_videos = list(self.dashcam_videos_dir.glob("*.mp4"))
        all_videos.sort()  # Ensure consistent ordering
        
        # Filter to only ground truth videos unless process_all_videos is True
        if not self.process_all_videos:
            ground_truth_files = set(f.stem for f in self.ground_truth_dir.glob("*.json"))
            all_videos = [v for v in all_videos if v.stem in ground_truth_files]
        
        if video_filter:
            # Filter by video IDs (e.g., ["0001", "0002"])
            filtered_videos = []
            for video in all_videos:
                video_id = video.stem.split('_')[0]  # Extract ID from filename
                if video_id in video_filter:
                    filtered_videos.append(video)
            return filtered_videos
        
        return all_videos
    
    def get_output_path(self, video_path: Path) -> Path:
        """Get output path for a specific video."""
        full_model_id = self.driveguard_model_id or self.model_id
        if not full_model_id:
            raise ValueError("Either model_id or driveguard_model_id must be set to get output path")
        
        # Ensure model output directory exists (uses the updated model_output_dir property)
        self.model_output_dir.mkdir(parents=True, exist_ok=True)
        
        # Generate output filename
        output_filename = video_path.stem + ".json"
        return self.model_output_dir / output_filename
    
    def should_process_video(self, video_path: Path) -> bool:
        """Check if a video should be processed."""
        output_path = self.get_output_path(video_path)
        
        if not output_path.exists():
            return True
        
        if self.overwrite_existing:
            return True
        
        return False


def create_default_config(
    component: str,
    driveguard_model_id: Optional[str] = None,
    project_root: Optional[Path] = None,
    process_all_videos: bool = False
) -> GenerationConfig:
    """Create a default configuration."""
    
    if project_root is None:
        # Use the project root we already determined
        project_root = Path(__file__).parent.parent.parent.parent
    
    dashcam_videos_dir = project_root / settings.app.path["data"] / "dashcam"
    output_base_dir = project_root / settings.app.path["data"] / "evaluation" / "system_outputs"
    
    # Convert DriveGuard model ID to API model ID if provided
    api_model_id = None
    if driveguard_model_id:
        api_model_id = get_api_model_id(driveguard_model_id)
    
    return GenerationConfig(
        project_root=project_root,
        dashcam_videos_dir=dashcam_videos_dir,
        output_base_dir=output_base_dir,
        component=component,
        model_id=api_model_id,
        driveguard_model_id=driveguard_model_id,
        process_all_videos=process_all_videos
    )


# Available components
AVAILABLE_COMPONENTS = [
    "annotation",
    "scene", 
    "violation",
    "accident",
    "assessment"
]

# Get supported model IDs from DriveGuard settings
SUPPORTED_MODEL_IDS = settings.supported_models.llm

def get_api_model_id(driveguard_model_id: str) -> str:
    """Convert DriveGuard model ID to API model ID."""
    if driveguard_model_id not in settings.supported_models.llm:
        raise ValueError(f"Unsupported model ID: {driveguard_model_id}. Available: {SUPPORTED_MODEL_IDS}")
    # With the new format, the driveguard_model_id is already in "platform:model" format
    # Extract just the model part for API usage
    platform, model_name = driveguard_model_id.split(":", 1)
    return model_name