"""
AssessmentGenerator system output generator.
"""

import json
import sys
import time
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple

# Add project root to path for imports
project_root = Path(__file__).parent.parent.parent.parent.parent
sys.path.append(str(project_root))

from src.llm.agent.driving_suggestion import driving_assessor
from src.llm.llms import get_llm
from .base_generator import BaseSystemGenerator


class AssessmentGenerator(BaseSystemGenerator):
    """Generator for comprehensive driving assessments."""
    
    def __init__(self, config):
        """Initialize the AssessmentGenerator."""
        super().__init__(config)
        
        # Initialize the driving assessor (already configured with gateway model support)
        self.driving_assessor = driving_assessor
        self.model_id = config.driveguard_model_id or config.model_id
        
    def get_component_name(self) -> str:
        """Return the component name."""
        return "assessment"
    
    def get_ground_truth_list(self, video_filter: Optional[List[str]] = None) -> List[Path]:
        """Get list of ground truth files to process instead of video files."""
        ground_truth_dir = self.config.ground_truth_dir
        all_ground_truth = list(ground_truth_dir.glob("*.json"))
        all_ground_truth.sort()  # Ensure consistent ordering
        
        if video_filter:
            # Filter by video IDs (e.g., ["0000", "0001"])
            filtered_files = []
            for gt_file in all_ground_truth:
                video_id = gt_file.stem.split('_')[0]  # Extract ID from filename
                if video_id in video_filter:
                    filtered_files.append(gt_file)
            return filtered_files
        
        return all_ground_truth
    
    def get_ground_truth_path(self, video_path: Path) -> Path:
        """Get ground truth annotation file path for a video."""
        video_id = video_path.stem
        ground_truth_dir = self.config.project_root / "data" / "evaluation" / "ground_truth"
        ground_truth_path = ground_truth_dir / f"{video_id}.json"
        return ground_truth_path
    
    def format_accident_results(self, accidents: List[Dict]) -> str:
        """Format accident results for assessment prompt."""
        return "\\n".join([
            f"Scene: {accident['scene']}\\n"
            f"Accident Risk: {accident['accident']} - {accident['consequence']}\\n"
            for accident in accidents
        ])
    
    def format_violation_results(self, violations: List[Dict]) -> str:
        """Format violation results for assessment prompt."""
        return "\\n".join([
            f"Scene: {violation['scene']}\\n"
            f"Rule Violation: {violation['violation']} - {violation['reason']}\\n"
            for violation in violations
        ])
    
    def generate_output_from_ground_truth(self, ground_truth_path: Path) -> Tuple[Dict, float]:
        """
        Generate comprehensive driving assessment from ground truth data.
        
        Args:
            ground_truth_path: Path to the ground truth JSON file
            
        Returns:
            Tuple of (assessment result dict, total processing time)
        """
        start_time = time.time()
        
        try:
            with open(ground_truth_path, 'r') as f:
                ground_truth_data = json.load(f)
            
            # Extract required data from ground truth
            ground_truth_content = ground_truth_data['ground_truth']
            annotation = ground_truth_content['annotation']
            violations = ground_truth_content['violations']
            accidents = ground_truth_content['accidents']
            
            # Format results for the driving assessor
            accident_summary = self.format_accident_results(accidents)
            rule_summary = self.format_violation_results(violations)
            
            # Generate comprehensive assessment
            assessment = self.driving_assessor.invoke({
                'annotation': annotation,
                'accident_results': accident_summary,
                'rule_results': rule_summary
            })
            
            total_time = time.time() - start_time
            return assessment, total_time
            
        except Exception as e:
            raise Exception(f"Failed to generate driving assessment: {e}")
    
    def generate_output(self, video_path: Path) -> Dict:
        """
        Generate driving assessment output from ground truth annotation.
        
        Args:
            video_path: Path to the video file (used to find corresponding ground truth)
            
        Returns:
            Driving assessment result
        """
        # Find corresponding ground truth file
        ground_truth_path = self.get_ground_truth_path(video_path)
        assessment, _ = self.generate_output_from_ground_truth(ground_truth_path)
        return assessment
    
    def create_output_metadata(
        self, 
        video_path: Path, 
        content: Any,
        generation_time: float,
        additional_metadata: Dict = None
    ) -> Dict[str, Any]:
        """Create output metadata with AssessmentGenerator-specific info."""
        
        # Extract assessment statistics
        safety_score = content.get('safety_score', 0) if isinstance(content, dict) else 0
        risk_level = content.get('risk_level', 'unknown') if isinstance(content, dict) else 'unknown'
        
        # Count statistics
        strengths_count = len(content.get('strengths', [])) if isinstance(content, dict) else 0
        weaknesses_count = len(content.get('weaknesses', [])) if isinstance(content, dict) else 0
        advice_count = len(content.get('improvement_advice', [])) if isinstance(content, dict) else 0
        
        # Prepare additional metadata
        assessment_metadata = {
            "safety_score": safety_score,
            "risk_level": risk_level,
            "strengths_count": strengths_count,
            "weaknesses_count": weaknesses_count,
            "advice_count": advice_count,
            "model_type": "text",
            "prompt_type": "comprehensive_driving_assessment",
            "source": "ground_truth_violations_and_accidents"
        }
        
        if additional_metadata:
            assessment_metadata.update(additional_metadata)
        
        return super().create_output_metadata(
            video_path, 
            content, 
            generation_time, 
            assessment_metadata
        )
    
    def process_videos(
        self, 
        video_filter: Optional[List[str]] = None,
        progress_callback: Optional[callable] = None
    ) -> Dict[str, Any]:
        """
        Process multiple ground truth files for comprehensive assessments.
        Override the base method to work with ground truth files instead of video files.
        """
        # Get list of ground truth files to process
        ground_truth_files = self.get_ground_truth_list(video_filter)
        
        # For each ground truth file, we need to create a corresponding "video path"
        # for the output file naming system to work correctly
        videos_to_process = []
        for gt_file in ground_truth_files:
            # Create a fake video path based on the ground truth filename
            video_name = gt_file.stem + ".mp4"  # Convert 0000_something.json -> 0000_something.mp4
            fake_video_path = self.config.dashcam_videos_dir / video_name
            
            # Check if we should process this file
            if self.config.should_process_video(fake_video_path):
                videos_to_process.append((fake_video_path, gt_file))
        
        self.stats["total_videos"] = len(ground_truth_files)
        self.stats["skipped"] = len(ground_truth_files) - len(videos_to_process)
        self.stats["start_time"] = self._get_current_time()
        
        print(f"Found {len(ground_truth_files)} total ground truth files")
        print(f"Processing {len(videos_to_process)} files")
        print(f"Skipping {self.stats['skipped']} existing files")
        print(f"Component: {self.get_component_name()}")
        print(f"Model: {self.config.model_id}")
        print()
        
        # Process each ground truth file
        for i, (video_path, gt_file) in enumerate(videos_to_process, 1):
            print(f"[{i}/{len(videos_to_process)}] Processing {gt_file.name}...")
            
            success = self.process_ground_truth_file(video_path, gt_file)
            
            if success:
                self.stats["processed"] += 1
            else:
                self.stats["failed"] += 1
            
            # Call progress callback if provided
            if progress_callback:
                progress_callback(i, len(videos_to_process), success)
        
        self.stats["end_time"] = self._get_current_time()
        
        # Print final statistics
        self.print_summary()
        
        return self.stats
    
    def process_ground_truth_file(self, video_path: Path, ground_truth_path: Path) -> bool:
        """
        Process a single ground truth file and save assessment output.
        
        Args:
            video_path: Fake video path for output file naming
            ground_truth_path: Path to the ground truth file
            
        Returns:
            True if successful, False otherwise
        """
        try:
            # Generate the assessment from ground truth file (with timing)
            content, generation_time = self.generate_output_from_ground_truth(ground_truth_path)
            
            # Create standardized output using the fake video path for file naming
            output_data = self.create_output_metadata(
                video_path, 
                content, 
                generation_time
            )
            
            # Save to file
            if self.save_output(video_path, output_data):
                print(f"✓ Completed {ground_truth_path.name} in {generation_time:.2f}s")
                return True
            else:
                print(f"✗ Failed to save output for {ground_truth_path.name}")
                return False
                
        except Exception as e:
            print(f"✗ Error processing {ground_truth_path.name}: {e}")
            return False
    
    def _get_current_time(self) -> str:
        """Get current time as ISO string."""
        from datetime import datetime
        return datetime.now().isoformat()