"""
Abstract base class for system output generators.
"""

import json
import time
from abc import ABC, abstractmethod
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

from ..config import GenerationConfig


class BaseSystemGenerator(ABC):
    """Base class for generating system outputs from DriveGuard components."""
    
    def __init__(self, config: GenerationConfig):
        """Initialize the generator with configuration."""
        self.config = config
        self.stats = {
            "total_videos": 0,
            "processed": 0,
            "skipped": 0,
            "failed": 0,
            "start_time": None,
            "end_time": None
        }
    
    @abstractmethod
    def generate_output(self, video_path: Path) -> Dict[str, Any]:
        """
        Generate system output for a single video.
        
        Args:
            video_path: Path to the video file
            
        Returns:
            Dictionary containing the system output
        """
        pass
    
    @abstractmethod
    def get_component_name(self) -> str:
        """Return the name of the component this generator handles."""
        pass
    
    def create_output_metadata(
        self, 
        video_path: Path, 
        content: Any,
        generation_time: float,
        additional_metadata: Optional[Dict] = None
    ) -> Dict[str, Any]:
        """
        Create standardized output metadata.
        
        Args:
            video_path: Path to the input video
            content: The generated content
            generation_time: Time taken to generate output
            additional_metadata: Additional metadata to include
            
        Returns:
            Standardized output dictionary
        """
        video_id = video_path.stem
        
        output = {
            "video_id": video_id,
            "video_path": str(video_path),
            "model_id": self.config.driveguard_model_id or self.config.model_id,
            "component": self.get_component_name(),
            "generated_at": datetime.now().isoformat(),
            "generation_time": generation_time,
            "content": content,
            "metadata": {
                "fps": self.config.fps,
                "generator_version": "1.0.0"
            }
        }
        
        if additional_metadata:
            output["metadata"].update(additional_metadata)
            
        return output
    
    def save_output(self, video_path: Path, output_data: Dict[str, Any]) -> bool:
        """
        Save output data to file.
        
        Args:
            video_path: Path to the input video
            output_data: Data to save
            
        Returns:
            True if successful, False otherwise
        """
        try:
            output_path = self.config.get_output_path(video_path)
            
            with open(output_path, 'w', encoding='utf-8') as f:
                json.dump(output_data, f, indent=2, ensure_ascii=False)
            
            return True
            
        except Exception as e:
            print(f"Error saving output for {video_path.name}: {e}")
            return False
    
    def process_video(self, video_path: Path) -> bool:
        """
        Process a single video and save output.
        
        Args:
            video_path: Path to the video file
            
        Returns:
            True if successful, False otherwise
        """
        try:
            print(f"Processing {video_path.name}...")
            
            start_time = time.time()
            
            # Generate the output
            content = self.generate_output(video_path)
            
            generation_time = time.time() - start_time
            
            # Create standardized output
            output_data = self.create_output_metadata(
                video_path, 
                content, 
                generation_time
            )
            
            # Save to file
            if self.save_output(video_path, output_data):
                print(f"✓ Completed {video_path.name} in {generation_time:.2f}s")
                return True
            else:
                print(f"✗ Failed to save output for {video_path.name}")
                return False
                
        except Exception as e:
            print(f"✗ Error processing {video_path.name}: {e}")
            return False
    
    def process_videos(
        self, 
        video_filter: Optional[List[str]] = None,
        progress_callback: Optional[callable] = None
    ) -> Dict[str, Any]:
        """
        Process multiple videos.
        
        Args:
            video_filter: Optional list of video IDs to process
            progress_callback: Optional callback for progress updates
            
        Returns:
            Processing statistics
        """
        # Get list of videos to process
        videos = self.config.get_video_list(video_filter)
        
        # Filter videos that need processing
        videos_to_process = [
            video for video in videos 
            if self.config.should_process_video(video)
        ]
        
        self.stats["total_videos"] = len(videos)
        self.stats["skipped"] = len(videos) - len(videos_to_process)
        self.stats["start_time"] = datetime.now().isoformat()
        
        print(f"Found {len(videos)} total videos")
        print(f"Processing {len(videos_to_process)} videos")
        print(f"Skipping {self.stats['skipped']} existing videos")
        print(f"Component: {self.get_component_name()}")
        print(f"Model: {self.config.model_id}")
        print()
        
        # Process each video
        for i, video_path in enumerate(videos_to_process, 1):
            print(f"[{i}/{len(videos_to_process)}]", end=" ")
            
            success = self.process_video(video_path)
            
            if success:
                self.stats["processed"] += 1
            else:
                self.stats["failed"] += 1
            
            # Call progress callback if provided
            if progress_callback:
                progress_callback(i, len(videos_to_process), success)
        
        self.stats["end_time"] = datetime.now().isoformat()
        
        # Print final statistics
        self.print_summary()
        
        return self.stats
    
    def print_summary(self):
        """Print processing summary."""
        print()
        print("=" * 50)
        print("PROCESSING SUMMARY")
        print("=" * 50)
        print(f"Total videos found: {self.stats['total_videos']}")
        print(f"Successfully processed: {self.stats['processed']}")
        print(f"Skipped (existing): {self.stats['skipped']}")
        print(f"Failed: {self.stats['failed']}")
        
        if self.stats["start_time"] and self.stats["end_time"]:
            start = datetime.fromisoformat(self.stats["start_time"])
            end = datetime.fromisoformat(self.stats["end_time"])
            duration = end - start
            print(f"Total time: {duration}")
        
        print(f"Output saved to: {self.config.model_output_dir}")
    
    def get_status(self) -> Dict[str, Any]:
        """Get current processing status."""
        videos = self.config.get_video_list()
        existing_outputs = list(self.config.model_output_dir.glob("*.json")) if self.config.model_output_dir.exists() else []
        
        return {
            "component": self.get_component_name(),
            "model_id": self.config.model_id,
            "total_videos": len(videos),
            "completed_videos": len(existing_outputs),
            "remaining_videos": len(videos) - len(existing_outputs),
            "output_directory": str(self.config.model_output_dir),
            "completion_rate": len(existing_outputs) / len(videos) * 100 if videos else 0
        }