"""
VideoAnnotator system output generator.
"""

import sys
from pathlib import Path
from typing import Any, Dict

# Add project root to path for imports
project_root = Path(__file__).parent.parent.parent.parent.parent
sys.path.append(str(project_root))

from src.llm.agent.dashcam_annotation import VideoAnnotator
from .base_generator import BaseSystemGenerator


class VideoAnnotatorGenerator(BaseSystemGenerator):
    """Generator for VideoAnnotator system outputs."""
    
    def __init__(self, config):
        """Initialize the VideoAnnotator generator."""
        super().__init__(config)
        
        # Initialize VideoAnnotator with DriveGuard model ID
        # VideoAnnotator expects DriveGuard model IDs, not API model IDs
        model_id_to_use = config.driveguard_model_id if config.driveguard_model_id else config.model_id
        self.annotator = VideoAnnotator(model_id=model_id_to_use)
        
    def get_component_name(self) -> str:
        """Return the component name."""
        return "annotation"
    
    def generate_output(self, video_path: Path) -> str:
        """
        Generate annotation output for a video.
        
        Args:
            video_path: Path to the video file
            
        Returns:
            Annotation string from VideoAnnotator
        """
        try:
            # Use VideoAnnotator to annotate the video
            annotation = self.annotator.annotate_video(
                str(video_path), 
                fps=self.config.fps
            )
            
            return annotation
            
        except Exception as e:
            raise Exception(f"Failed to generate annotation: {e}")
    
    def create_output_metadata(
        self, 
        video_path: Path, 
        content: str,
        generation_time: float,
        additional_metadata: Dict = None
    ) -> Dict[str, Any]:
        """Create output metadata with VideoAnnotator-specific info."""
        
        # Get frame count for metadata
        try:
            base64_frames = VideoAnnotator.video2frames(str(video_path), self.config.fps)
            frame_count = len(base64_frames)
        except Exception:
            frame_count = None
        
        # Prepare additional metadata
        video_metadata = {
            "frame_count": frame_count,
            "model_type": "multimodal",
            "prompt_type": "dashcam_annotation"
        }
        
        if additional_metadata:
            video_metadata.update(additional_metadata)
        
        return super().create_output_metadata(
            video_path, 
            content, 
            generation_time, 
            video_metadata
        )