"""
File management utilities for system output generation.
"""

import json
import shutil
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any


class FileManager:
    """Handles file operations for system output generation."""
    
    def __init__(self, base_output_dir: Path):
        """Initialize file manager with base output directory."""
        self.base_output_dir = Path(base_output_dir)
        self.base_output_dir.mkdir(parents=True, exist_ok=True)
    
    def get_component_dir(self, component: str) -> Path:
        """Get directory for a specific component."""
        component_dir = self.base_output_dir / component
        component_dir.mkdir(parents=True, exist_ok=True)
        return component_dir
    
    def get_model_dir(self, component: str, model_id: str) -> Path:
        """Get directory for a specific component and model."""
        model_dir = self.get_component_dir(component) / model_id
        model_dir.mkdir(parents=True, exist_ok=True)
        return model_dir
    
    def list_components(self) -> List[str]:
        """List all available components."""
        if not self.base_output_dir.exists():
            return []
        
        components = []
        for item in self.base_output_dir.iterdir():
            if item.is_dir() and not item.name.startswith('.'):
                components.append(item.name)
        
        return sorted(components)
    
    def list_models(self, component: str) -> List[str]:
        """List all available models for a component."""
        component_dir = self.get_component_dir(component)
        
        if not component_dir.exists():
            return []
        
        models = []
        for item in component_dir.iterdir():
            if item.is_dir() and not item.name.startswith('.'):
                models.append(item.name)
        
        return sorted(models)
    
    def count_outputs(self, component: str, model_id: str) -> int:
        """Count number of output files for a component and model."""
        model_dir = self.get_model_dir(component, model_id)
        return len(list(model_dir.glob("*.json")))
    
    def list_output_files(self, component: str, model_id: str) -> List[Path]:
        """List all output files for a component and model."""
        model_dir = self.get_model_dir(component, model_id)
        return sorted(list(model_dir.glob("*.json")))
    
    def output_exists(self, component: str, model_id: str, video_id: str) -> bool:
        """Check if output exists for a specific video."""
        model_dir = self.get_model_dir(component, model_id)
        output_file = model_dir / f"{video_id}.json"
        return output_file.exists()
    
    def load_output(self, component: str, model_id: str, video_id: str) -> Optional[Dict[str, Any]]:
        """Load output data for a specific video."""
        model_dir = self.get_model_dir(component, model_id)
        output_file = model_dir / f"{video_id}.json"
        
        if not output_file.exists():
            return None
        
        try:
            with open(output_file, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception as e:
            print(f"Error loading output {output_file}: {e}")
            return None
    
    def save_output(
        self, 
        component: str, 
        model_id: str, 
        video_id: str, 
        data: Dict[str, Any]
    ) -> bool:
        """Save output data for a specific video."""
        model_dir = self.get_model_dir(component, model_id)
        output_file = model_dir / f"{video_id}.json"
        
        try:
            with open(output_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=2, ensure_ascii=False)
            return True
        except Exception as e:
            print(f"Error saving output {output_file}: {e}")
            return False
    
    def delete_output(self, component: str, model_id: str, video_id: str) -> bool:
        """Delete output for a specific video."""
        model_dir = self.get_model_dir(component, model_id)
        output_file = model_dir / f"{video_id}.json"
        
        if output_file.exists():
            try:
                output_file.unlink()
                return True
            except Exception as e:
                print(f"Error deleting output {output_file}: {e}")
                return False
        
        return False
    
    def clear_model_outputs(self, component: str, model_id: str) -> bool:
        """Clear all outputs for a specific component and model."""
        model_dir = self.get_model_dir(component, model_id)
        
        if not model_dir.exists():
            return True
        
        try:
            for output_file in model_dir.glob("*.json"):
                output_file.unlink()
            return True
        except Exception as e:
            print(f"Error clearing outputs from {model_dir}: {e}")
            return False
    
    def backup_model_outputs(self, component: str, model_id: str, backup_suffix: Optional[str] = None) -> Path:
        """Create a backup of model outputs."""
        model_dir = self.get_model_dir(component, model_id)
        
        if backup_suffix is None:
            backup_suffix = datetime.now().strftime("%Y%m%d_%H%M%S")
        
        backup_dir = model_dir.parent / f"{model_id}_backup_{backup_suffix}"
        
        if model_dir.exists():
            shutil.copytree(model_dir, backup_dir)
        
        return backup_dir
    
    def get_status_summary(self) -> Dict[str, Any]:
        """Get overall status summary."""
        summary = {
            "total_components": 0,
            "components": {}
        }
        
        components = self.list_components()
        summary["total_components"] = len(components)
        
        for component in components:
            models = self.list_models(component)
            component_info = {
                "total_models": len(models),
                "models": {}
            }
            
            for model in models:
                output_count = self.count_outputs(component, model)
                component_info["models"][model] = {
                    "output_count": output_count,
                    "output_dir": str(self.get_model_dir(component, model))
                }
            
            summary["components"][component] = component_info
        
        return summary
    
    def validate_outputs(self, component: str, model_id: str) -> Dict[str, Any]:
        """Validate outputs for a component and model."""
        validation_result = {
            "valid_files": 0,
            "invalid_files": 0,
            "missing_fields": [],
            "errors": []
        }
        
        output_files = self.list_output_files(component, model_id)
        
        required_fields = [
            "video_id", "video_path", "model_id", "component", 
            "generated_at", "generation_time", "content", "metadata"
        ]
        
        for output_file in output_files:
            try:
                with open(output_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                
                # Check required fields
                missing = [field for field in required_fields if field not in data]
                
                if missing:
                    validation_result["invalid_files"] += 1
                    validation_result["missing_fields"].extend(
                        [(output_file.name, field) for field in missing]
                    )
                else:
                    validation_result["valid_files"] += 1
                    
            except Exception as e:
                validation_result["invalid_files"] += 1
                validation_result["errors"].append((output_file.name, str(e)))
        
        return validation_result