"""
Model tracking utilities for system output generation.
"""

import json
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any


class ModelTracker:
    """Tracks model usage and performance across system output generation."""
    
    def __init__(self, tracking_dir: Path):
        """Initialize model tracker."""
        self.tracking_dir = Path(tracking_dir)
        self.tracking_dir.mkdir(parents=True, exist_ok=True)
        self.tracking_file = self.tracking_dir / "model_tracking.json"
        
        # Load existing tracking data
        self.data = self._load_tracking_data()
    
    def _load_tracking_data(self) -> Dict[str, Any]:
        """Load existing tracking data."""
        if self.tracking_file.exists():
            try:
                with open(self.tracking_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except Exception as e:
                print(f"Warning: Could not load tracking data: {e}")
                return self._create_empty_tracking_data()
        
        return self._create_empty_tracking_data()
    
    def _create_empty_tracking_data(self) -> Dict[str, Any]:
        """Create empty tracking data structure."""
        return {
            "created_at": datetime.now().isoformat(),
            "last_updated": datetime.now().isoformat(),
            "models": {},
            "components": {},
            "runs": []
        }
    
    def _save_tracking_data(self):
        """Save tracking data to file."""
        self.data["last_updated"] = datetime.now().isoformat()
        
        try:
            with open(self.tracking_file, 'w', encoding='utf-8') as f:
                json.dump(self.data, f, indent=2, ensure_ascii=False)
        except Exception as e:
            print(f"Warning: Could not save tracking data: {e}")
    
    def register_model(
        self, 
        model_id: str, 
        model_info: Optional[Dict[str, Any]] = None
    ):
        """Register a model for tracking."""
        if model_id not in self.data["models"]:
            self.data["models"][model_id] = {
                "first_seen": datetime.now().isoformat(),
                "total_runs": 0,
                "total_videos_processed": 0,
                "total_generation_time": 0.0,
                "components_used": [],
                "info": model_info or {}
            }
        
        self._save_tracking_data()
    
    def register_component(self, component: str, component_info: Optional[Dict[str, Any]] = None):
        """Register a component for tracking."""
        if component not in self.data["components"]:
            self.data["components"][component] = {
                "first_seen": datetime.now().isoformat(),
                "total_runs": 0,
                "total_videos_processed": 0,
                "models_used": [],
                "info": component_info or {}
            }
        
        self._save_tracking_data()
    
    def start_run(
        self, 
        model_id: str, 
        component: str, 
        video_count: int,
        run_config: Optional[Dict[str, Any]] = None
    ) -> str:
        """Start tracking a new run."""
        run_id = f"{component}_{model_id}_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        
        run_data = {
            "run_id": run_id,
            "model_id": model_id,
            "component": component,
            "start_time": datetime.now().isoformat(),
            "end_time": None,
            "planned_video_count": video_count,
            "processed_videos": 0,
            "successful_videos": 0,
            "failed_videos": 0,
            "total_generation_time": 0.0,
            "average_generation_time": 0.0,
            "status": "running",
            "config": run_config or {},
            "errors": []
        }
        
        self.data["runs"].append(run_data)
        
        # Register model and component
        self.register_model(model_id)
        self.register_component(component)
        
        # Update component and model counters
        self.data["models"][model_id]["total_runs"] += 1
        self.data["components"][component]["total_runs"] += 1
        
        # Track component usage in model
        if component not in self.data["models"][model_id]["components_used"]:
            self.data["models"][model_id]["components_used"].append(component)
        
        # Track model usage in component
        if model_id not in self.data["components"][component]["models_used"]:
            self.data["components"][component]["models_used"].append(model_id)
        
        self._save_tracking_data()
        return run_id
    
    def update_run_progress(
        self, 
        run_id: str, 
        processed_count: int, 
        successful_count: int,
        failed_count: int,
        total_generation_time: float
    ):
        """Update run progress."""
        run_data = self._find_run(run_id)
        if run_data:
            run_data["processed_videos"] = processed_count
            run_data["successful_videos"] = successful_count
            run_data["failed_videos"] = failed_count
            run_data["total_generation_time"] = total_generation_time
            
            if processed_count > 0:
                run_data["average_generation_time"] = total_generation_time / processed_count
            
            self._save_tracking_data()
    
    def finish_run(self, run_id: str, final_stats: Dict[str, Any]):
        """Finish tracking a run."""
        run_data = self._find_run(run_id)
        if run_data:
            run_data["end_time"] = datetime.now().isoformat()
            run_data["status"] = "completed"
            run_data.update(final_stats)
            
            # Update model and component totals
            model_id = run_data["model_id"]
            component = run_data["component"]
            
            self.data["models"][model_id]["total_videos_processed"] += run_data["successful_videos"]
            self.data["models"][model_id]["total_generation_time"] += run_data["total_generation_time"]
            
            self.data["components"][component]["total_videos_processed"] += run_data["successful_videos"]
            
            self._save_tracking_data()
    
    def add_run_error(self, run_id: str, error_info: Dict[str, Any]):
        """Add error information to a run."""
        run_data = self._find_run(run_id)
        if run_data:
            run_data["errors"].append({
                "timestamp": datetime.now().isoformat(),
                **error_info
            })
            self._save_tracking_data()
    
    def _find_run(self, run_id: str) -> Optional[Dict[str, Any]]:
        """Find run data by ID."""
        for run_data in self.data["runs"]:
            if run_data["run_id"] == run_id:
                return run_data
        return None
    
    def get_model_stats(self, model_id: str) -> Optional[Dict[str, Any]]:
        """Get statistics for a model."""
        if model_id not in self.data["models"]:
            return None
        
        model_data = self.data["models"][model_id].copy()
        
        # Calculate additional stats
        if model_data["total_videos_processed"] > 0:
            model_data["average_time_per_video"] = (
                model_data["total_generation_time"] / model_data["total_videos_processed"]
            )
        else:
            model_data["average_time_per_video"] = 0.0
        
        # Get recent runs
        recent_runs = [
            run for run in self.data["runs"]
            if run["model_id"] == model_id
        ][-5:]  # Last 5 runs
        
        model_data["recent_runs"] = recent_runs
        
        return model_data
    
    def get_component_stats(self, component: str) -> Optional[Dict[str, Any]]:
        """Get statistics for a component."""
        if component not in self.data["components"]:
            return None
        
        component_data = self.data["components"][component].copy()
        
        # Get recent runs
        recent_runs = [
            run for run in self.data["runs"]
            if run["component"] == component
        ][-5:]  # Last 5 runs
        
        component_data["recent_runs"] = recent_runs
        
        return component_data
    
    def get_run_history(self, limit: Optional[int] = None) -> List[Dict[str, Any]]:
        """Get run history."""
        runs = sorted(
            self.data["runs"],
            key=lambda x: x["start_time"],
            reverse=True
        )
        
        if limit:
            runs = runs[:limit]
        
        return runs
    
    def get_summary(self) -> Dict[str, Any]:
        """Get overall tracking summary."""
        total_runs = len(self.data["runs"])
        completed_runs = len([r for r in self.data["runs"] if r["status"] == "completed"])
        running_runs = len([r for r in self.data["runs"] if r["status"] == "running"])
        
        total_videos = sum(
            model_data["total_videos_processed"] 
            for model_data in self.data["models"].values()
        )
        
        total_time = sum(
            model_data["total_generation_time"] 
            for model_data in self.data["models"].values()
        )
        
        return {
            "total_models": len(self.data["models"]),
            "total_components": len(self.data["components"]),
            "total_runs": total_runs,
            "completed_runs": completed_runs,
            "running_runs": running_runs,
            "total_videos_processed": total_videos,
            "total_generation_time": total_time,
            "average_time_per_video": total_time / total_videos if total_videos > 0 else 0.0,
            "tracking_since": self.data["created_at"],
            "last_updated": self.data["last_updated"]
        }