"""Example usage of the complete DriveGuard video evaluation workflow."""

import sys
from pathlib import Path

# Add the project root to the path
root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.workflow.evaluate_driving_video import (
    DrivingVideoEvaluator,
    evaluate_driving_video,
    get_driving_safety_summary
)
from src.utils.log import logger


def example_basic_evaluation():
    """Example: Basic video evaluation using convenience function."""
    print("="*60)
    print("EXAMPLE 1: Basic Video Evaluation")
    print("="*60)
    
    # Example video path (replace with actual video file)
    video_path = root / "data" / "dashcam" / "cut_off_accident.mp4"
    
    try:
        # Simple one-line evaluation
        assessment = evaluate_driving_video(video_path, model_id='openai_gpt_4o')
        
        print(f"🏆 Safety Score: {assessment['safety_score']}/10")
        print(f"⚠️  Risk Level: {assessment['risk_level'].upper()}")
        print(f"\\n📝 Overall Evaluation:")
        print(f"   {assessment['overall_evaluation']}")
        
        print(f"\\n✅ Strengths:")
        for strength in assessment['strengths']:
            print(f"   + {strength}")
        
        print(f"\\n❌ Areas for Improvement:")
        for weakness in assessment['weaknesses']:
            print(f"   - {weakness}")
        
        print(f"\\n💡 Recommendations:")
        for advice in assessment['improvement_advice']:
            print(f"   → {advice}")
            
    except FileNotFoundError:
        print(f"❌ Video file not found: {video_path}")
        print("   Please ensure the video file exists or update the path")
    except Exception as e:
        print(f"❌ Evaluation failed: {e}")


def example_quick_safety_summary():
    """Example: Quick safety summary for multiple videos."""
    print("\\n" + "="*60)
    print("EXAMPLE 2: Quick Safety Summary")
    print("="*60)
    
    # Example video path
    video_path = root / "data" / "dashcam" / "cut_off_accident.mp4"
    
    try:
        # Get concise summary
        summary = get_driving_safety_summary(video_path)
        
        print(f"📁 Video: {summary['video_file']}")
        print(f"🏆 Safety Score: {summary['safety_score']}/10")
        print(f"⚠️  Risk Level: {summary['risk_level'].upper()}")
        print(f"\\n📊 Analysis Summary:")
        print(f"   • {summary['strengths_count']} strengths identified")
        print(f"   • {summary['weaknesses_count']} areas for improvement")
        print(f"   • {summary['advice_count']} recommendations provided")
        
        print(f"\\n🔍 Key Issues:")
        for issue in summary['key_issues']:
            print(f"   ⚠️  {issue}")
        
        print(f"\\n💡 Top Recommendations:")
        for rec in summary['top_recommendations']:
            print(f"   → {rec}")
            
    except FileNotFoundError:
        print(f"❌ Video file not found: {video_path}")
    except Exception as e:
        print(f"❌ Summary failed: {e}")


def example_detailed_evaluation():
    """Example: Detailed evaluation with all intermediate results."""
    print("\\n" + "="*60)
    print("EXAMPLE 3: Detailed Evaluation")
    print("="*60)
    
    video_path = root / "data" / "dashcam" / "cut_off_accident.mp4"
    video_path = root / "data" / "dashcam" / "left_turn_cut_off.mp4"  # Update with your actual video path
    
    try:
        # Create evaluator for detailed analysis
        evaluator = DrivingVideoEvaluator(model_id='openai_gpt_4o')
        detailed_results = evaluator.evaluate_with_details(video_path)
        
        print(f"📁 Video: {Path(detailed_results['video_path']).name}")
        print(f"⏱️  Analysis Time: {detailed_results['evaluation_time']:.2f} seconds")
        print(f"\\n📊 Analysis Summary:")
        print(f"   • {detailed_results['summary']['total_scenes']} scenes analyzed")
        print(f"   • {detailed_results['summary']['violations_found']} rule violations found")
        print(f"   • {detailed_results['summary']['accident_risks']} accident risks identified")
        print(f"   • Safety Score: {detailed_results['summary']['safety_score']}/10")
        
        print(f"\\n🎯 Extracted Scenes:")
        for i, scene in enumerate(detailed_results['scenes'], 1):
            print(f"   {i}. {scene}")
        
        print(f"\\n🚨 Rule Violations:")
        for result in detailed_results['rule_analysis']:
            if result['analysis']['violation'] == 'found':
                print(f"   ❌ {result['scene'][:60]}...")
                print(f"      → {result['analysis']['reason']}")
        
        print(f"\\n⚠️  Accident Risks:")
        for result in detailed_results['accident_analysis']:
            if result['analysis']['accident'] == 'found':
                print(f"   🚨 {result['scene'][:60]}...")
                print(f"      → {result['analysis']['consequence']}")
        
        # Add complete safety assessment output
        assessment = detailed_results['assessment']
        print(f"\\n" + "="*40)
        print("COMPLETE SAFETY ASSESSMENT")
        print("="*40)
        
        print(f"🏆 SAFETY SCORE: {assessment['safety_score']}/10")
        print(f"⚠️  RISK LEVEL: {assessment['risk_level'].upper()}")
        
        print(f"\\n📝 OVERALL EVALUATION:")
        print(f"   {assessment['overall_evaluation']}")
        
        print(f"\\n✅ STRENGTHS ({len(assessment['strengths'])}):")
        for strength in assessment['strengths']:
            print(f"   + {strength}")
        
        print(f"\\n❌ WEAKNESSES ({len(assessment['weaknesses'])}):")
        for weakness in assessment['weaknesses']:
            print(f"   - {weakness}")
        
        print(f"\\n💡 IMPROVEMENT ADVICE ({len(assessment['improvement_advice'])}):")
        for advice in assessment['improvement_advice']:
            print(f"   → {advice}")
                
    except FileNotFoundError:
        print(f"❌ Video file not found: {video_path}")
    except Exception as e:
        print(f"❌ Detailed evaluation failed: {e}")


def example_multiple_videos():
    """Example: Evaluate multiple videos for comparison."""
    print("\\n" + "="*60)
    print("EXAMPLE 4: Multiple Video Evaluation")
    print("="*60)
    
    # Example video paths (add more as needed)
    video_paths = [
        root / "data" / "dashcam" / "cut_off_accident.mp4",
        # Add more video paths here when available
    ]
    
    try:
        evaluator = DrivingVideoEvaluator()
        results = evaluator.evaluate_multiple_videos(video_paths)
        
        print(f"📊 Evaluated {len(results)} videos:")
        print("-" * 40)
        
        for i, result in enumerate(results, 1):
            if 'error' in result:
                print(f"{i}. ❌ {result['video_file']}: {result['error']}")
            else:
                print(f"{i}. {result['video_file']}")
                print(f"   🏆 Score: {result['safety_score']}/10 | Risk: {result['risk_level']}")
                print(f"   📝 {result['overall_evaluation'][:80]}...")
        
        # Calculate average score
        valid_results = [r for r in results if 'error' not in r]
        if valid_results:
            avg_score = sum(r['safety_score'] for r in valid_results) / len(valid_results)
            print(f"\\n📈 Average Safety Score: {avg_score:.1f}/10")
            
    except Exception as e:
        print(f"❌ Multiple video evaluation failed: {e}")


if __name__ == "__main__":
    print("DriveGuard Video Evaluation Examples")
    print("=" * 60)
    print("NOTE: These examples require actual video files in data/dashcam/")
    print("Replace the video paths with your actual dashcam video files.")
    print("=" * 60)
    
    # Run all examples
    # example_basic_evaluation()
    # example_quick_safety_summary()
    example_detailed_evaluation()
    # example_multiple_videos()
    
    print("\\n" + "🎉" * 20)
    print("Examples completed! Check the output above for results.")
    print("🎉" * 20)