"""Example demonstrating the new model tracking and component timing features.

This script shows how to:
1. Run evaluations with different model configurations
2. Track which models are used for each component
3. Measure component-level performance timing
4. Compare different model configurations
"""

import sys
from pathlib import Path

# Add project root to path
root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.workflow.evaluate_driving_video import DrivingVideoEvaluator
from src.utils.log import logger


def example_basic_model_tracking():
    """Example 1: Basic model tracking with default configuration."""
    print("=" * 60)
    print("EXAMPLE 1: Basic Model Tracking")
    print("=" * 60)
    
    # Use default models
    evaluator = DrivingVideoEvaluator()
    
    # Run evaluation on sample video
    video_path = root / "data" / "dashcam" / "000_cut_off_accident.mp4"
    
    if video_path.exists():
        results = evaluator.evaluate_with_details(video_path)
        
        print("\n📊 Models Used:")
        for component, model in results['models_used'].items():
            print(f"  {component}: {model}")
        
        print("\n⏱️  Component Timings:")
        for component, timing in results['component_timings'].items():
            print(f"  {component}: {timing:.2f}s")
        
        print(f"\n🎯 Safety Score: {results['assessment']['safety_score']}/10")
        print(f"🚨 Risk Level: {results['assessment']['risk_level']}")
    else:
        print(f"❌ Video not found: {video_path}")


def example_model_overrides():
    """Example 2: Using model overrides for specific components."""
    print("\n" + "=" * 60)
    print("EXAMPLE 2: Model Overrides")
    print("=" * 60)
    
    # Override specific components to use different models
    model_overrides = {
        "scene_extraction": "openai_gpt_4_1",  # Use GPT-4.1 instead of fast model
        "driving_assessment": "openai_gpt_4_1"  # Use GPT-4.1 for assessment
    }
    
    evaluator = DrivingVideoEvaluator(model_overrides=model_overrides)
    
    video_path = root / "data" / "dashcam" / "000_cut_off_accident.mp4"
    
    if video_path.exists():
        results = evaluator.evaluate_with_details(video_path)
        
        print("\n📊 Models Used (with overrides):")
        for component, model in results['models_used'].items():
            custom = "✓" if component in model_overrides else " "
            print(f"  {component}: {model} {custom}")
        
        print("\n⏱️  Component Timings:")
        for component, timing in results['component_timings'].items():
            print(f"  {component}: {timing:.2f}s")
        
        print(f"\n🎯 Safety Score: {results['assessment']['safety_score']}/10")
        
        # Show performance comparison
        fastest = results['performance_metrics']['summary'].get('fastest_component')
        slowest = results['performance_metrics']['summary'].get('slowest_component')
        if fastest and slowest:
            print(f"\n🏃 Fastest Component: {fastest['component']} ({fastest['duration']:.2f}s)")
            print(f"🐌 Slowest Component: {slowest['component']} ({slowest['duration']:.2f}s)")
    else:
        print(f"❌ Video not found: {video_path}")


def example_video_annotation_model():
    """Example 3: Changing the video annotation model."""
    print("\n" + "=" * 60)
    print("EXAMPLE 3: Video Annotation Model Override")
    print("=" * 60)
    
    # Use GPT-4.1 for video annotation instead of default GPT-4o
    evaluator = DrivingVideoEvaluator(model_id="openai_gpt_4_1")
    
    video_path = root / "data" / "dashcam" / "000_cut_off_accident.mp4"
    
    if video_path.exists():
        results = evaluator.evaluate_with_details(video_path)
        
        print("\n📊 Models Used (GPT-4.1 for video annotation):")
        for component, model in results['models_used'].items():
            highlight = "🎯" if component == "video_annotation" else "  "
            print(f"{highlight}{component}: {model}")
        
        print(f"\n⏱️  Video Annotation Time: {results['component_timings']['video_annotation']:.2f}s")
        print(f"⏱️  Total Evaluation Time: {results['evaluation_time']:.2f}s")
        
        print(f"\n🎯 Safety Score: {results['assessment']['safety_score']}/10")
    else:
        print(f"❌ Video not found: {video_path}")


def example_performance_comparison():
    """Example 4: Compare performance between different configurations."""
    print("\n" + "=" * 60)
    print("EXAMPLE 4: Performance Comparison")
    print("=" * 60)
    
    video_path = root / "data" / "dashcam" / "000_cut_off_accident.mp4"
    
    if not video_path.exists():
        print(f"❌ Video not found: {video_path}")
        return
    
    configs = [
        ("Default (GPT-4o + Fast models)", {}, None),
        ("All GPT-4.1", {}, "openai_gpt_4_1"),
        ("Mixed (Fast extraction + GPT-4.1 assessment)", {"driving_assessment": "openai_gpt_4_1"}, None)
    ]
    
    results = []
    
    for config_name, model_overrides, video_model in configs:
        print(f"\n🧪 Testing: {config_name}")
        
        evaluator = DrivingVideoEvaluator(
            model_id=video_model,
            model_overrides=model_overrides if model_overrides else None
        )
        
        eval_results = evaluator.evaluate_with_details(video_path)
        
        results.append({
            "name": config_name,
            "total_time": eval_results['evaluation_time'],
            "video_annotation_time": eval_results['component_timings']['video_annotation'],
            "safety_score": eval_results['assessment']['safety_score'],
            "models": eval_results['models_used']
        })
        
        print(f"  ⏱️  Total Time: {eval_results['evaluation_time']:.2f}s")
        print(f"  🎯 Safety Score: {eval_results['assessment']['safety_score']}/10")
    
    print("\n" + "=" * 60)
    print("PERFORMANCE COMPARISON SUMMARY")
    print("=" * 60)
    
    print(f"{'Configuration':<30} {'Total Time':<12} {'Video Ann.':<12} {'Safety':<8}")
    print("-" * 70)
    
    for result in results:
        print(f"{result['name']:<30} {result['total_time']:<12.2f} "
              f"{result['video_annotation_time']:<12.2f} {result['safety_score']:<8.1f}")


def main():
    """Run all examples."""
    try:
        example_basic_model_tracking()
        example_model_overrides()
        example_video_annotation_model()
        example_performance_comparison()
        
        print("\n" + "=" * 60)
        print("✅ ALL EXAMPLES COMPLETED SUCCESSFULLY")
        print("=" * 60)
        
    except Exception as e:
        logger.error(f"Example failed: {e}")
        print(f"\n❌ Example failed: {e}")


if __name__ == "__main__":
    main()