from langchain_core.documents import Document
from langchain_core.retrievers import BaseRetriever
from .milvus_db import MilvusDB

class MilvusRetriever(BaseRetriever):
    """A retriever that uses Milvus to retrieve documents."""
    milvus: MilvusDB
    text_key: str = "text"
    meta: list[str] = []
    top_k: int = 5

    def _get_relevant_documents(self, query: str) -> list[Document]:
        """Retrieve relevant documents from Milvus based on the query."""
        # Implement the logic to query Milvus and return a list of Document objects
        content = self.milvus.emb_func(query)
        result = self.milvus.client.search(
            collection_name=self.milvus.name,
            data=[content],
            output_fields=[self.text_key]+self.meta,
            limit=self.top_k,
        )
        documents = []
        for item in result[0]:
            metadata = {k: item['entity'][k] for k in self.meta if k in item['entity']}
            metadata['similarity'] = item.get('distance', 0)
            doc = Document(
                page_content=item['entity'][self.text_key],
                metadata=metadata
            )
            documents.append(doc)
        return documents
    

