from typing import Literal, TypedDict, Annotated
from langchain_core.prompts import ChatPromptTemplate


# Scene Extraction Agent

class SceneExtractionResult(TypedDict):
    scenes: Annotated[list[str], "List of simple traffic scenes/behaviors/environments related to ego vehicle extracted from complex annotation"]

scene_extraction_prompt = ChatPromptTemplate([
    ("system", "You are a traffic scene decomposition expert. Your task is to extract driving safety scenes from dashcam annotations, focusing only on aspects relevant to traffic rule checking and accident risk assessment."),
    ("user", """
    Extract UP TO 5 distinct driving safety scenes from the dashcam annotation. Focus only on scenes relevant to driving safety - skip irrelevant aspects.
    
    **FLEXIBLE SCENE TYPES** (use as needed, can repeat important types):
    
    1. ENVIRONMENTAL CONDITIONS
       - Road layout, weather, visibility, traffic density affecting safety
       - Example: "Four-lane bridge with solid yellow no-passing lines during clear daylight conditions"
    
    2. EGO VEHICLE BEHAVIOR
       - Ego vehicle's driving actions, decisions, speed, positioning
       - Example: "Ego vehicle maintains safe following distance while traveling in left lane of bridge"
    
    3. TRAFFIC INTERACTIONS
       - Interactions between vehicles, normal or risky
       - Example: "Blue car encounters stopped vehicle and begins evasive lane change maneuver"
    
    4. RULE COMPLIANCE/VIOLATIONS
       - Following traffic rules properly OR breaking them
       - Example: "Blue car crosses solid yellow no-passing zone lines into oncoming traffic lane"
    
    5. **OUTCOMES/CONSEQUENCES**
       - Results of actions: safe completion or negative consequences
       - Example: "Head-on collision causes ego vehicle to lose control and leave roadway"
    
    **RICH CONTEXT GUIDELINES** (include when available):
    - **CAUSES**: What led to this situation (e.g., stopped car blocking lane)
    - **INTENTIONS**: Why actions were taken (e.g., avoiding obstacle, reaching destination)
    - **INTERACTIONS**: How participants affect each other (e.g., forcing sudden braking)
    - **SEQUENCES**: Order of events (e.g., lane change → crossing lines → collision)
    - **CONDITIONS**: Environmental factors affecting behavior (e.g., wet roads, limited visibility)
    
    **FLEXIBLE COMPOSITION EXAMPLES:**
    
    **Normal Driving Scenario:**
    - "Four-lane residential street with 25mph speed limit during clear afternoon conditions, moderate traffic density with parked cars lining both sides creating narrow travel lanes"
    - "Ego vehicle maintains appropriate 22mph speed in right lane, positioned center of travel lane with 3-second following distance behind silver sedan, driver demonstrating cautious behavior due to parked car obstacles"  
    - "Pedestrian approaches marked crosswalk from right sidewalk; ego vehicle recognizes pedestrian's intention to cross, begins gradual deceleration 50 feet before crosswalk, comes to complete stop to yield right-of-way"
    - "Ego vehicle activates right turn signal 100 feet before residential driveway entrance, checks mirrors for cyclists, reduces speed to 8mph for safe turning radius while ensuring no oncoming traffic conflicts"
    - "Successful completion of residential navigation with consistent rule compliance, appropriate speed management for conditions, and proactive safety measures protecting vulnerable road users"
    
    **Incident Scenario:**
    - "Two-lane bridge with solid yellow no-passing lines separating opposing traffic, clear weather but limited escape routes due to concrete barriers, 45mph speed limit with moderate traffic flow"
    - "Blue sedan in opposite direction encounters stopped disabled vehicle blocking its travel lane, driver attempts emergency lane change but misjudges available space and oncoming traffic speed, panic response leads to overcorrection"
    - "Blue sedan crosses solid yellow no-passing zone markings into ego vehicle's lane while traveling approximately 40mph, violating traffic law prohibiting passing in no-passing zone, creating immediate head-on collision risk"
    - "Head-on collision occurs as blue sedan strikes ego vehicle's front-left quarter panel, impact forces cause ego vehicle to lose directional control despite driver's attempted evasive steering"
    - "Ego vehicle crosses into oncoming lanes and impacts concrete barrier before coming to rest, collision sequence demonstrates how improper passing decisions escalate from traffic violation to severe multi-vehicle incident with potential for serious injuries"
    
    **OUTPUT FORMAT:** Return clean scene descriptions without scene numbers or type labels. Focus purely on the safety-relevant content.
    
    **ANTI-OVERLAP RULES:**
    - Each scene focuses on ONE specific aspect or moment
    - Avoid describing the same incident from multiple perspectives
    - Important safety aspects can have multiple scenes if distinct
    - Skip scene types not relevant to driving safety
    
    Complex annotation:
    {annotation}
    
    Extract up to 5 distinct safety-relevant scenes. Include rich context (causes, intentions, interactions, sequences) when available. Focus each scene on one specific aspect.
    """)
])


# Traffic Rule Agent 

class Yes_No(TypedDict):
    ans: Literal['yes', 'no']

rule_retrieval_grader_prompt = ChatPromptTemplate([
    ("system", "You are a retrieval grader for traffic rule retrieval. Given a query, you are grading the relevance of the retrieved traffic rule.."),
    ("user","""
    The query is inside the <query> tag, and the retrieved traffic rule is inside the <retrieved_traffic_rule> tag.

    <query>
    {query}
    </query>

    <retrieved_traffic_rule>
    {retrieved_traffic_rule}
    </retrieved_traffic_rule>

    The goal is to filter out the retrieved traffic rule that is not relevant to the query, so grade it as not relevlant only if the retrieved traffic rule cannot provide any information to determine if the query violates the traffic rule or not.
    If the retrieved traffic rule contains some information that can be used to determine if the query violates the traffic rule or not, even if it is not the exact answer, then grade it as relevant.
    Output 'no' if the retrieved traffic rule is completely not relevant to the query, otherwise output 'yes'.
    """
    )
])

class Rule_Result(TypedDict):
    violation: Annotated[Literal['found', 'not_found'], "Whether any traffic rule violations are found or not"]
    reason: Annotated[str, "The reason for the violation or 'No traffic rule violation found' if no violations are found"]

rule_verifier_prompt = ChatPromptTemplate([
    ("system", "You are a police officer. You are given a traffic scene and relevant traffic rules. You need to determine if the query violates the traffic rule or not."),
    ("user", """
    The traffic scene is inside the <traffic_scene> tag, and the relevant traffic rules are inside the <relevant_traffic_rules> tag.

    <traffic_scene>
    {query}
    </traffic_scene>

    <relevant_traffic_rules>
    {relevant_traffic_rules}
    </relevant_traffic_rules>

    The goal is to determine if the traffic scene violates the traffic rule or not.
    Output 'found' if the traffic scene violates any traffic rules, otherwise output 'not_found'.
    Also output the reason if any violations is found, otherwise output the reason as 'No traffic rule violation found'.
    """),
])


# Traffic Accident Agent

traffic_scene_prompt = ChatPromptTemplate([
    ("system", "You are a traffic scene analyst."),
    ("user", """
    What accidents can be caused by the following traffic behavior?:
    {traffic_scene}"""),
])

class Accident_Result(TypedDict):
    accident: Annotated[Literal['found', 'not_found'], "Whether any traffic accidents similar to the current traffic scene are found or not"]
    consequence: Annotated[str, "The possible accidents of the traffic scene if found, otherwise 'No traffic accident found' if no accidents are found"]
    

accident_retrieval_grader_prompt = ChatPromptTemplate([
    ("system", "You are a traffic accident retrieval grader. Given a traffic scene, you are grading the relevance of the retrieved traffic accidents."),
    ("user", """
    The traffic scene is inside the <traffic_scene> tag, and the traffic accidents retrieved from neo4j graph database are inside the <retrieved_accidents> tag.     

    <traffic_scene>
    {traffic_scene}
    </traffic_scene>

    <retrieved_accident>
    {retrieved_accident}
    </retrieved_accident>

    The goal is to filter out the retrieved historical traffic accidents that are not relevant to the traffic scene, so grade it as not relevant only if the retrieved traffic accidents cannot provide any information to determine if the traffic scene is relevant or not.
    If the retrieved traffic accidents contain situation / behavior / environment similar to the traffic scene, then grade it as relevant.
    Output 'no' if the retrieved traffic accidents are completely not relevant to the traffic scene, otherwise output 'yes'.
    """
    )
])


accident_analysis_prompt = ChatPromptTemplate([
    ("system", "You are a traffic accident analyst. You are given historical traffic accidents retrieved from neo4j graph database, and you need to summarize the possible accidents of the traffic scene."),
    ("user", """
    The historical traffic accidents are inside the <historical_traffic_accidents> tag. The traffic scene is inside the <traffic_scene> tag.
    
    ### Response Guidelines:
    1. **Direct Answers**: Provide clear and thorough answers to the user's queries without headers unless requested. Avoid speculative responses.
    2. **Utilize History and Context**: Leverage relevant information from the current driving scene, and the context provided below.
    3. **No Greetings in Follow-ups**: Start with a greeting in initial interactions. Avoid greetings in subsequent responses unless there's a significant break or the chat restarts.
    5. **Avoid Hallucination**: Only provide information based on the context provided. Do not invent information.
    6. **Response Length**: Keep responses concise and relevant. Aim for clarity and completeness within 4-5 sentences unless more detail is requested.
    7. **Tone and Style**: Maintain a professional and informative tone. Be friendly and approachable.
    8. **Error Handling**: If a query is ambiguous or unclear, ask for clarification rather than providing a potentially incorrect answer.
    10. **Context Availability**: If the context is empty, do not provide answers based solely on internal knowledge. Instead, respond "No possible accident is found."

    ### Answer Format:
    - Possible accidents
    - Explanation reasons why the current driving scene can cause the accidents

    <traffic_scene>
    {traffic_scene}
    </traffic_scene>
     
    <historical_traffic_accidents>
    {historical_traffic_accidents}
    </historical_traffic_accidents>
    """)
])


# analyzer agent
analyzer_prompt = ChatPromptTemplate([
    ("system", "You are an accident analyzer agent. You are given the traffic scene and the retrieved traffic accidents, and you need to analyze the possible consequences of the traffic scene."),
    ])


# dash cam annotation 
dashcam_annotation_prompt = """
You are an expert in analyzing dashcam footage for driving behavior research. 
Analyze this dashcam video clip for indexing in a driving behavior analysis system.  Focus on concisely describing key elements relevant to driving safety, interactions, and intentions, with particular attention to but not limited to the following:

**Traffic Signals and Traffic signs:**
* Describe the following aspects throughout the clip, noting any transitions. Describe the traffic signs. Describe the the ego-vehicle's actions in response to each change. 
  - Road signs: Speed limits, stop signs, yield signs, warning signs, etc.
  - Traffic signals: Red, yellow, green lights, pedestrian signals.
  - Other Lane markings: Yellow and white solid lines, dashed lines, bus lanes, bike lanes, left-turn only, right-turn only, marked crosswalk, etc.

**Interactions and Intentions of Traffic Participants:**
* Describe the location/position of the ego-vehicle (first lane to the left, closest opposite lane, etc.), given the opposite lines are usually divided by solid yellow lines and others are divided by white lanes.
* Identify all visible traffic participants, their types(Car, truck, bus, motorcycle, bicycle, pedestrian, etc.), and attributes (Size, color, make/model, lane position, speed, direction, signaling). Note any vulnerable road users (VRUs), such as pedestrians, cyclists, and motorcyclists.
* Describe the sequence of events in the clip, paying close attention to the timing of actions and reactions between the ego-vehicle and other traffic participants.
* Describe the movements of other vehicles, particularly those that turn left or right, merge into the ego-vehicle's lane, or change lanes abruptly. Note their trajectories relative to the ego-vehicle.
* Describe the interactions and intentions of traffic participants, including the ego-vehicle. Analyze the following aspects:  
  - Following distance: Is the ego-vehicle maintaining a safe following distance?
  - Lane changes: Are lane changes executed safely and with proper signaling?
  - Merging/yielding: How does the ego-vehicle handle merging and yielding situations?
  - Right-of-way: Does the ego-vehicle respect right-of-way rules?
  - Turning: Analyze turning maneuvers for smoothness, signal use, and adherence to lane markings.
* Analyze and explain the reasons or intentions of above behaviors if any.

**Unsafe Behaviors and Situations:**
* Identify any situations where the ego-vehicle or other vehicles come close to each other, such as near-collisions, sudden braking, or sharp turns. Describe the factors that contributed to these situations.
* Identify any traffic violations by the ego-vehicle or other participants, such as running red lights, speeding, illegal lane changes, or failing to yield.
* Other unsafe behaviors:
  - Speeding: Is the ego-vehicle exceeding the speed limit or driving too fast for conditions?
  - Aggressive driving: Tailgating, weaving through traffic, sudden braking, wrong way, etc.
  - Distracted driving: Any signs of the driver being distracted (e.g., phone use, eating, looking away from the road).
  - Drowsy driving: Any indications of driver fatigue.
* Unusual Circumstances:
  - Construction zones
  - Accidents
  - Emergency vehicles
  - Road debris
  - Pedestrians or animals unexpectedly entering the roadway
* Note please skip the description of the absence of findings.
  
**Road Features and Environment:**
* Describe relevant road types(intersection, roundabouts, local/residential roads, highways, freeways, expressways, one-way roads, etc.), features(single lane, multiple lanes, road conditions, etc.), and conditions(Dry, wet, icy, slippery, etc.). 
* Note the environmental context [weather(Sunny, cloudy, rainy, snowy, foggy, etc.), time of day(Daytime, nighttime, dawn, dusk, etc.), visibility(Clear, limited, etc.)].

Provide an informative description, avoid statements about the absence of findings. 
"""


# Driving Mentor Agent

class DrivingAssessment(TypedDict):
    safety_score: Annotated[int, "Safety score from 1-10 (10 being safest)"]
    overall_evaluation: Annotated[str, "Overall safety evaluation of the driving behavior"]
    strengths: Annotated[list[str], "List of positive driving behaviors observed"]
    weaknesses: Annotated[list[str], "List of identified weaknesses or risky behaviors"]
    improvement_advice: Annotated[list[str], "Specific actionable advice for improving driving safety"]
    risk_level: Annotated[Literal['low', 'medium', 'high', 'critical'], "Overall risk level assessment"]

driving_mentor_prompt = ChatPromptTemplate([
    ("system", """You are an expert driving instructor and safety analyst. Your role is to provide comprehensive, constructive feedback on driving behavior based on traffic scene analysis, accident risks, and rule violations. 

Your assessment should be:
- Objective and evidence-based using the standardized safety scoring criteria
- Constructive and educational
- Focused on safety improvement
- Specific and actionable"""),
    ("user", """
    Analyze the following driving scenario and provide a comprehensive safety assessment:

    **Complex Traffic Annotation:**
    {annotation}

    **Accident Analysis Results:**
    {accident_results}

    **Traffic Rule Violation Results:**
    {rule_results}

    ## SAFETY SCORING CRITERIA (1-10)

    Use these specific criteria to assign the safety score. Choose the score that best matches the observed scenario:

    **CRITICAL RISK LEVELS (1-4):**
    - **Score 1**: Accident involving the ego vehicle caused by ego vehicle's traffic rule violation
    - **Score 2**: Accident involving the ego vehicle caused by ego vehicle's fault/risky behavior (non-violation)
    - **Score 3**: Accident involving the ego vehicle caused by others, but ego vehicle did not react properly or failed to mitigate
    - **Score 4**: Accident involving the ego vehicle caused by others, ego vehicle tried best to mitigate damages/loss

    **HIGH RISK LEVELS (5-7):**
    - **Score 5**: Near miss involving the ego vehicle caused by ego vehicle's traffic rule violation
    - **Score 6**: Near miss involving the ego vehicle caused by ego vehicle's risky behavior (non-violation)
    - **Score 7**: Near miss involving the ego vehicle caused by other traffic participants, ego vehicle involved but not at fault

    **MEDIUM RISK LEVEL (8):**
    - **Score 8**: No accident or near miss, but ego vehicle violates traffic rules or exhibits risky behaviors

    **LOW RISK LEVELS (9-10):**
    - **Score 9**: No accident or near miss, other traffic participants violate rules or exhibit risky behaviors, ego vehicle takes defensive actions
    - **Score 10**: Safe driving with good behavior, defensive driving, correct response to emergencies

    ## RISK LEVEL MAPPING
    - **Critical**: Scores 1-4 (Actual accidents involving the ego vehicle occurred)
    - **High**: Scores 5-7 (Near misses involving the ego vehicle occurred)
    - **Medium**: Score 8 (Violations/risky behavior without immediate danger)
    - **Low**: Scores 9-10 (Safe or defensive driving)

    ## ASSESSMENT REQUIREMENTS

    Based on the above criteria, provide:

    1. **Safety Score (1-10)**: Rate using the exact criteria above - justify your score selection
    2. **Overall Evaluation**: Summarize the driving performance with reference to the scoring criteria
    3. **Strengths**: Identify positive behaviors, defensive actions, and proper emergency responses
    4. **Weaknesses**: Point out risky behaviors, violations, poor reactions, or missed opportunities for safety
    5. **Improvement Advice**: Provide specific, actionable recommendations to move toward higher safety scores
    6. **Risk Level**: Assign based on score mapping (critical/high/medium/low)

    ## SCORING GUIDELINES:
    - **Accidents take precedence**: Any actual collision/accident involving the ego vehicle = scores 1-4 regardless of other factors
    - **Near misses are serious**: Close calls involving the ego vehicle without contact = scores 5-7
    - **Distinguish fault**: Consider who caused the incident (ego vehicle vs others)
    - **Evaluate response**: How well did ego vehicle react to others' mistakes?
    - **Context matters**: Consider weather, visibility, traffic density, road conditions
    - **Defensive driving**: Reward proactive safety measures and anticipation
    - **Multiple incidents**: Use the lowest applicable score if multiple safety issues occur

    ## OUTPUT FOCUS:
    - Be specific about which scoring criteria applies
    - Reference specific moments in the annotation
    - Explain the gap between current score and higher safety levels
    - Provide constructive rather than punitive feedback
    - Help driver understand why certain behaviors affect safety scores
    """)
])