"""Scene extraction agent for breaking down complex dashcam annotations into simple scenes."""

from typing import TypedDict

from langgraph.graph import StateGraph, START, END

from .agent_prompt import scene_extraction_prompt, SceneExtractionResult
from ..llms import get_llm
from ...utils.settings import settings
from ...utils.log import logger


# Initialize LLM
llm = get_llm(settings.app.llm['fast'])
scene_extractor = scene_extraction_prompt | llm.with_structured_output(SceneExtractionResult).with_retry()


# Graph State
class GraphState(TypedDict):
    """State for the scene extraction graph."""
    annotation: str
    scenes: list[str]


def extract_scenes(state: GraphState) -> dict:
    """Extract simple scenes from complex annotation."""
    logger.debug("-------Extracting Scenes-------")
    annotation = state['annotation']
    
    result = scene_extractor.invoke({'annotation': annotation})
    scenes = result['scenes']
    
    logger.debug(f"Extracted {len(scenes)} scenes from annotation")
    for i, scene in enumerate(scenes, 1):
        logger.debug(f"  Scene {i}: {scene[:100]}...")
    
    return {'scenes': scenes}


# Build graph
graph = StateGraph(GraphState)
graph.add_node("extract_scenes", extract_scenes)

graph.add_edge(START, "extract_scenes")
graph.add_edge("extract_scenes", END)

# Build agent
scene_extraction_agent = graph.compile()


class SceneExtractor:
    """Scene extraction agent for breaking down complex annotations into simple scenes."""
    
    def __init__(self):
        """Initialize the scene extractor."""
        self.agent = scene_extraction_agent
    
    def extract(self, annotation: str) -> list[str]:
        """Extract simple scenes from complex annotation.
        
        Args:
            annotation (str): Complex dashcam annotation from VideoAnnotator.
            
        Returns:
            list[str]: List of simple traffic scenes/behaviors/environments.
        """
        result = self.agent.invoke({'annotation': annotation})
        return result['scenes']
    
    def extract_for_analysis(self, annotation: str) -> dict:
        """Extract scenes formatted for downstream analysis.
        
        Args:
            annotation (str): Complex dashcam annotation.
            
        Returns:
            dict: Dictionary with scenes formatted for traffic rule and accident analysis.
        """
        scenes = self.extract(annotation)
        
        return {
            'scenes': scenes,
            'rule_queries': [{'query': scene} for scene in scenes],
            'accident_scenes': [{'scene': scene} for scene in scenes]
        }
