"""Complete workflow for evaluating driving behavior from dashcam videos.

This module implements the full DriveGuard workflow as shown in doc/workflow.png:
1. Video Input -> Data Annotator (VideoAnnotator)
2. Data Annotator -> Traffic Scene Extractor (SceneExtractor) 
3. Scene Extractor -> Collaborated Agents (Accident Retriever + Rule Checker)
4. All Analysis -> Driving Mentor -> Safety Assessment
"""

import time
from pathlib import Path
from typing import Dict, List, Optional, Union

from ..agent.dashcam_annotation import VideoAnnotator
from ..agent.scene_extraction import SceneExtractor
from ..agent.driving_suggestion import DrivingMentor
from ..agent.agent_prompt import DrivingAssessment
from ...utils.log import logger
from ...utils.settings import settings
from ...utils.model_tracker import ModelTracker, create_default_tracker
from ...utils.performance_timer import PerformanceTimer, ComponentPerformanceContext


class DrivingVideoEvaluator:
    """Complete workflow for evaluating driving behavior from dashcam videos."""
    
    def __init__(self, model_id: Optional[str] = None, model_overrides: Optional[Dict[str, str]] = None):
        """Initialize the driving video evaluator.
        
        Args:
            model_id (str, optional): Model ID for video annotation. 
                                    Defaults to multimodal setting.
            model_overrides (dict, optional): Override models for specific components.
                                            Format: {"component_name": "model_key"}
                                            Example: {"scene_extraction": "openai_gpt_4_1"}
        """
        # Initialize tracking systems
        self.model_tracker = create_default_tracker()
        self.performance_timer = PerformanceTimer()
        
        # Store the effective model ID for video annotation
        self.video_annotation_model = model_id or settings.app.llm['multimodal']
        
        # Apply any model overrides
        if model_overrides:
            for component, override_model in model_overrides.items():
                self.model_tracker.record_component_model(
                    component_name=component,
                    model_key=override_model,
                    model_type="llm",
                    custom_override=True
                )
        
        # Record video annotation model (may be overridden)
        video_model = model_overrides.get("video_annotation", self.video_annotation_model) if model_overrides else self.video_annotation_model
        self.model_tracker.record_component_model(
            component_name="video_annotation",
            model_key=video_model,
            model_type="llm",
            custom_override=bool(model_id)  # True if explicitly provided
        )
        
        # Initialize components
        self.video_annotator = VideoAnnotator(video_model)
        self.scene_extractor = SceneExtractor()
        self.driving_mentor = DrivingMentor()
        
        logger.info("DriveGuard Video Evaluator initialized")
        logger.info(f"Video annotation model: {video_model}")
        if model_overrides:
            logger.info(f"Component overrides: {model_overrides}")
            
        # Log all models being used
        models_summary = self.model_tracker.get_models_summary()
        logger.info(f"Component models: {models_summary}")
    
    def evaluate_video(self, video_path: Union[str, Path], fps: int = 2) -> DrivingAssessment:
        """Complete evaluation workflow from video to safety assessment.
        
        Args:
            video_path (Union[str, Path]): Path to the dashcam video file.
            fps (int): Frames per second to sample. Defaults to 2.
            
        Returns:
            DrivingAssessment: Comprehensive safety assessment with scores and advice.
        """
        start_time = time.time()
        video_path = Path(video_path)
        
        logger.info(f"Starting evaluation of video: {video_path.name}")
        
        if not video_path.exists():
            raise FileNotFoundError(f"Video file not found: {video_path}")
        
        try:
            # Step 1: Video Annotation (Data Annotator)
            logger.info("Step 1: Analyzing dashcam video...")
            annotation = self.video_annotator.annotate_video(str(video_path), fps=fps)
            logger.info(f" Video annotation completed ({len(annotation)} characters)")
            
            # Step 2-4: Scene Extraction + Collaborative Analysis + Driving Mentor
            logger.info("Step 2-4: Extracting scenes and generating safety assessment...")
            assessment = self.driving_mentor.assess_driving(annotation)
            
            evaluation_time = time.time() - start_time
            logger.info(f" Evaluation completed in {evaluation_time:.2f} seconds")
            logger.info(f"Safety Score: {assessment['safety_score']}/10 | Risk: {assessment['risk_level']}")
            
            return assessment
            
        except Exception as e:
            logger.error(f"Evaluation failed: {str(e)}")
            raise
    
    def evaluate_with_details(self, video_path: Union[str, Path], fps: int = 2) -> Dict:
        """Complete evaluation with detailed intermediate results.
        
        Args:
            video_path (Union[str, Path]): Path to the dashcam video file.
            fps (int): Frames per second to sample. Defaults to 2.
            
        Returns:
            Dict: Complete analysis including all intermediate steps.
        """
        start_time = time.time()
        video_path = Path(video_path)
        
        # Reset performance timer for this evaluation
        self.performance_timer.reset()
        
        logger.info(f"Starting detailed evaluation of video: {video_path.name}")
        
        if not video_path.exists():
            raise FileNotFoundError(f"Video file not found: {video_path}")
        
        try:
            # Step 1: Video Annotation
            logger.info("Step 1: Analyzing dashcam video...")
            with ComponentPerformanceContext(self.performance_timer, "video_annotation", 
                                           {"fps": fps, "video_file": video_path.name}):
                annotation = self.video_annotator.annotate_video(str(video_path), fps=fps)
            
            # Step 2: Scene Extraction
            logger.info("Step 2: Extracting traffic scenes...")
            with ComponentPerformanceContext(self.performance_timer, "scene_extraction",
                                           {"annotation_length": len(annotation)}):
                scenes = self.scene_extractor.extract(annotation)
            
            # Step 3-4: Complete Analysis with Driving Mentor
            logger.info("Step 3-4: Analyzing scenes and generating assessment...")
            with ComponentPerformanceContext(self.performance_timer, "driving_mentor",
                                           {"total_scenes": len(scenes)}):
                detailed_results = self.driving_mentor.assess_with_details(annotation)
            
            evaluation_time = time.time() - start_time
            
            # Get performance metrics
            performance_metrics = self.performance_timer.get_all_timings()
            component_timings = self.performance_timer.get_simple_timings()
            
            complete_results = {
                'video_path': str(video_path),
                'timestamp': self.performance_timer.session_timestamp,
                'models_used': self.model_tracker.get_models_summary(),
                'component_timings': component_timings,
                'performance_metrics': performance_metrics,
                'evaluation_time': evaluation_time,
                'annotation': annotation,
                'scenes': scenes,
                'accident_analysis': detailed_results['accident_analysis'],
                'rule_analysis': detailed_results['rule_analysis'], 
                'assessment': detailed_results['assessment'],
                'summary': {
                    'safety_score': detailed_results['assessment']['safety_score'],
                    'risk_level': detailed_results['assessment']['risk_level'],
                    'total_scenes': len(scenes),
                    'violations_found': len([r for r in detailed_results['rule_analysis'] if r['analysis']['violation'] == 'found']),
                    'accident_risks': len([r for r in detailed_results['accident_analysis'] if r['analysis']['accident'] == 'found'])
                }
            }
            
            logger.info(f" Detailed evaluation completed in {evaluation_time:.2f} seconds")
            logger.info(f"Analyzed {len(scenes)} scenes | Score: {complete_results['summary']['safety_score']}/10")
            
            # Log component timing summary
            fastest = performance_metrics.get('summary', {}).get('fastest_component')
            slowest = performance_metrics.get('summary', {}).get('slowest_component')
            if fastest and slowest:
                logger.info(f"Performance: Fastest={fastest['component']}({fastest['duration']:.2f}s), "
                          f"Slowest={slowest['component']}({slowest['duration']:.2f}s)")
            
            return complete_results
            
        except Exception as e:
            logger.error(f"Detailed evaluation failed: {str(e)}")
            raise
    
    def get_safety_summary(self, video_path: Union[str, Path], fps: int = 2) -> Dict:
        """Get a concise safety summary for quick assessment.
        
        Args:
            video_path (Union[str, Path]): Path to the dashcam video file.
            fps (int): Frames per second to sample. Defaults to 2.
            
        Returns:
            Dict: Concise safety summary with key metrics.
        """
        video_path = Path(video_path)
        logger.info(f"Getting safety summary for: {video_path.name}")
        
        try:
            # Get basic assessment
            assessment = self.evaluate_video(video_path, fps)
            
            # Create concise summary
            summary = {
                'video_file': video_path.name,
                'safety_score': assessment['safety_score'],
                'risk_level': assessment['risk_level'],
                'overall_evaluation': assessment['overall_evaluation'],
                'key_issues': assessment['weaknesses'][:3],  # Top 3 issues
                'top_recommendations': assessment['improvement_advice'][:3],  # Top 3 recommendations
                'strengths_count': len(assessment['strengths']),
                'weaknesses_count': len(assessment['weaknesses']),
                'advice_count': len(assessment['improvement_advice'])
            }
            
            logger.info(f"Safety summary: {summary['safety_score']}/10 ({summary['risk_level']} risk)")
            return summary
            
        except Exception as e:
            logger.error(f"Safety summary failed: {str(e)}")
            raise
    
    def evaluate_multiple_videos(self, video_paths: List[Union[str, Path]], fps: int = 2) -> List[Dict]:
        """Evaluate multiple videos and return comparative results.
        
        Args:
            video_paths (List[Union[str, Path]]): List of video file paths.
            fps (int): Frames per second to sample. Defaults to 2.
            
        Returns:
            List[Dict]: List of safety summaries for each video.
        """
        logger.info(f"Evaluating {len(video_paths)} videos...")
        
        results = []
        for i, video_path in enumerate(video_paths, 1):
            try:
                logger.info(f"Processing video {i}/{len(video_paths)}: {Path(video_path).name}")
                summary = self.get_safety_summary(video_path, fps)
                results.append(summary)
            except Exception as e:
                logger.error(f"Failed to process {video_path}: {str(e)}")
                results.append({
                    'video_file': Path(video_path).name,
                    'error': str(e),
                    'safety_score': 0,
                    'risk_level': 'unknown'
                })
        
        # Add comparative analysis
        valid_results = [r for r in results if 'error' not in r]
        if valid_results:
            avg_score = sum(r['safety_score'] for r in valid_results) / len(valid_results)
            logger.info(f" Completed evaluation of {len(video_paths)} videos | Average score: {avg_score:.1f}/10")
        
        return results


# Convenience functions for direct usage
def evaluate_driving_video(video_path: Union[str, Path], model_id: Optional[str] = None, fps: int = 2) -> DrivingAssessment:
    """Evaluate a single driving video.
    
    Args:
        video_path (Union[str, Path]): Path to the dashcam video file.
        model_id (str, optional): Model ID for analysis.
        fps (int): Frames per second to sample.
        
    Returns:
        DrivingAssessment: Complete safety assessment.
    """
    evaluator = DrivingVideoEvaluator(model_id)
    return evaluator.evaluate_video(video_path, fps)


def get_driving_safety_summary(video_path: Union[str, Path], model_id: Optional[str] = None, fps: int = 2) -> Dict:
    """Get a quick safety summary for a driving video.
    
    Args:
        video_path (Union[str, Path]): Path to the dashcam video file.
        model_id (str, optional): Model ID for analysis.
        fps (int): Frames per second to sample.
        
    Returns:
        Dict: Concise safety summary.
    """
    evaluator = DrivingVideoEvaluator(model_id)
    return evaluator.get_safety_summary(video_path, fps)
