"""Tests for dashcam video annotation agent."""

import sys
from pathlib import Path

root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.agent.dashcam_annotation import VideoAnnotator


def test_media_type_detection():
    """Test that media types are correctly detected for different models."""
    
    def get_media_type(model_id):
        """Helper function to get media type for a model."""
        platform = model_id.split(':', 1)[0] if ':' in model_id else 'openai'
        if platform == 'gateway':
            model_name = model_id.split(':', 1)[1]
            if any(provider in model_name.lower() for provider in ['claude', 'gemini']):
                return 'image/jpeg'
            else:
                return 'image/jpg'
        else:
            return 'image/jpg'
    
    # Test OpenAI model uses image/jpg
    annotator_openai = VideoAnnotator('openai:gpt-4o')
    media_type = get_media_type('openai:gpt-4o')
    assert media_type == 'image/jpg', f"OpenAI models should use image/jpg, got {media_type}"
    
    # Test Anthropic model via gateway uses image/jpeg
    media_type = get_media_type('gateway:routeway-discount/claude-opus-4-1-20250805')
    assert media_type == 'image/jpeg', f"Anthropic models should use image/jpeg, got {media_type}"
    
    # Test Google Gemini models via gateway use image/jpeg
    media_type = get_media_type('gateway:routeway-discount/gemini-2.5-pro')
    assert media_type == 'image/jpeg', f"Google Gemini models should use image/jpeg, got {media_type}"
    
    media_type = get_media_type('gateway:google-ai-studio/gemini-2.5-pro')
    assert media_type == 'image/jpeg', f"Google Gemini models should use image/jpeg, got {media_type}"
    
    # Test other gateway models use image/jpg
    media_type = get_media_type('gateway:routeway-discount/gpt-5')
    assert media_type == 'image/jpg', f"Non-Anthropic/Google gateway models should use image/jpg, got {media_type}"
    
    print("✓ Media type detection test passed for all model types")


def test_frame_sampling_limits():
    """Test that models with image limits get proper frame sampling."""
    
    # Test Gemma model has 16 frame limit
    annotator_gemma = VideoAnnotator('gateway:nebius/gemma-3-27b')
    max_images = annotator_gemma.get_max_images_for_model('gateway:nebius/gemma-3-27b')
    assert max_images == 16, f"Gemma models should have 16 frame limit, got {max_images}"
    
    # Test Qwen2.5-VL model has 16 frame limit
    annotator_qwen = VideoAnnotator('gateway:nebius/qwen2-5-vl-72b-instruct')
    max_images = annotator_qwen.get_max_images_for_model('gateway:nebius/qwen2-5-vl-72b-instruct')
    assert max_images == 16, f"Qwen2.5-VL models should have 16 frame limit, got {max_images}"
    
    # Test general Qwen models have no limit
    max_images = annotator_qwen.get_max_images_for_model('gateway:alibaba/qwen-max')
    assert max_images is None, f"General Qwen models should have no limit, got {max_images}"
    
    # Test frame sampling with 20 frames -> 16 frames
    test_frames = list(range(20))
    sampled = annotator_gemma.sample_frames_evenly(test_frames, 16)
    assert len(sampled) == 16, f"Should sample to 16 frames, got {len(sampled)}"
    assert sampled == [0, 1, 2, 3, 5, 6, 7, 8, 10, 11, 12, 13, 15, 16, 17, 18], f"Unexpected sampling pattern: {sampled}"
    
    # Test models without limits
    annotator_openai = VideoAnnotator('openai:gpt-4o')
    max_images = annotator_openai.get_max_images_for_model('openai:gpt-4o')
    assert max_images is None, f"OpenAI models should have no limit, got {max_images}"
    
    annotator_claude = VideoAnnotator('gateway:routeway-discount/claude-opus-4-1-20250805')
    max_images = annotator_claude.get_max_images_for_model('gateway:routeway-discount/claude-opus-4-1-20250805')
    assert max_images is None, f"Claude models should have no limit, got {max_images}"
    
    print("✓ Frame sampling limits test passed")


def test_annotate_video_method():
    """Test the annotate_video method for direct video processing."""
    # Initialize the video annotator
    annotator = VideoAnnotator(model_id='openai:gpt-4.1')

    # Process video directly with new method
    video_path = root / 'data' / 'dashcam' / 'cut_off_accident.mp4'
    video_path = root / 'data' / 'dashcam' / 'left_turn_cut_off.mp4'  # Update with your actual video path
    annotations = annotator.annotate_video(video_path, fps=2)

    # Print the annotations
    print("Direct video processing annotations:")
    print(annotations)


if __name__ == "__main__":
    print("Running dashcam annotation agent tests...\n")
    
    # Test media type detection
    test_media_type_detection()
    
    # Test frame sampling limits
    test_frame_sampling_limits()
    
    # Test direct video processing method
    test_annotate_video_method()
    
    print("\nAll tests completed!")
