"""Tests for driving mentor agent."""

import sys
from pathlib import Path

root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.agent.driving_suggestion import DrivingMentor
from src.utils.log import logger


def test_driving_mentor_basic():
    """Test basic driving mentor functionality."""
    mentor = DrivingMentor()
    
    # Simple test annotation
    annotation = """
    The ego vehicle is traveling in the right lane of a two-lane highway at 70 mph in a 65 mph zone.
    The vehicle approaches a construction zone with signs indicating a speed limit reduction to 45 mph.
    The ego vehicle fails to reduce speed and continues at 70 mph through the construction zone.
    A construction worker is visible near the roadway, creating a potential safety hazard.
    """
    
    try:
        # Test basic assessment
        assessment = mentor.assess_driving(annotation)
        
        logger.info("=== DRIVING ASSESSMENT ===")
        logger.info(f"Safety Score: {assessment['safety_score']}/10")
        logger.info(f"Risk Level: {assessment['risk_level']}")
        logger.info(f"Overall Evaluation: {assessment['overall_evaluation']}")
        
        logger.info("\\nStrengths:")
        for strength in assessment['strengths']:
            logger.info(f"  + {strength}")
        
        logger.info("\\nWeaknesses:")
        for weakness in assessment['weaknesses']:
            logger.info(f"  - {weakness}")
        
        logger.info("\\nImprovement Advice:")
        for advice in assessment['improvement_advice']:
            logger.info(f"  → {advice}")
        
        return assessment
        
    except Exception as e:
        logger.error(f"Test failed: {e}")
        raise


def test_safety_summary():
    """Test safety summary functionality."""
    mentor = DrivingMentor()
    
    annotation = """
    The ego vehicle approaches a four-way intersection with a stop sign.
    The vehicle rolls through the stop sign without coming to a complete stop.
    A pedestrian is crossing in the crosswalk as the vehicle proceeds through the intersection.
    The ego vehicle is traveling at 35 mph in a 25 mph residential zone.
    """
    
    try:
        summary = mentor.get_safety_summary(annotation)
        
        logger.info("\\n=== SAFETY SUMMARY ===")
        logger.info(f"Safety Score: {summary['safety_score']}/10")
        logger.info(f"Risk Level: {summary['risk_level']}")
        logger.info(f"Overall: {summary['overall_evaluation']}")
        
        logger.info("\\nKey Issues:")
        for issue in summary['key_issues']:
            logger.info(f"  ⚠ {issue}")
        
        logger.info("\\nTop Advice:")
        for advice in summary['top_advice']:
            logger.info(f"  💡 {advice}")
            
        return summary
        
    except Exception as e:
        logger.error(f"Safety summary test failed: {e}")
        raise


if __name__ == "__main__":
    print("Running driving mentor tests...\\n")
    
    # Test basic functionality
    test_driving_mentor_basic()
    
    print("\\n" + "="*50 + "\\n")
    
    # Test safety summary 
    test_safety_summary()
    
    print("\\nAll tests completed!")