"""Simple tests for driving mentor agent that don't require database connections."""

import sys
from pathlib import Path

root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.agent.driving_suggestion import DrivingMentor
from src.llm.agent.scene_extraction import SceneExtractor
from src.utils.log import logger


def test_scene_extraction_only():
    """Test just the scene extraction part of the workflow."""
    logger.info("=== TESTING SCENE EXTRACTION ===")
    
    extractor = SceneExtractor()
    
    annotation = """
    The ego vehicle is traveling in a residential area at 35 mph in a 25 mph zone.
    The vehicle approaches a four-way intersection with a stop sign.
    The ego vehicle performs a rolling stop instead of coming to a complete stop.
    A child on a bicycle is visible on the sidewalk near the intersection.
    The vehicle proceeds through the intersection without yielding to a pedestrian in the crosswalk.
    """
    
    scenes = extractor.extract(annotation)
    
    logger.info(f"Extracted {len(scenes)} scenes:")
    for i, scene in enumerate(scenes, 1):
        logger.info(f"  {i}. {scene}")
    
    return scenes


def test_driving_mentor_initialization():
    """Test that DrivingMentor can be initialized without database connections."""
    logger.info("=== TESTING DRIVING MENTOR INITIALIZATION ===")
    
    try:
        mentor = DrivingMentor()
        logger.info("✅ DrivingMentor initialized successfully")
        
        # Test that the agent graph is compiled
        logger.info(f"✅ Agent graph compiled: {mentor.agent is not None}")
        
        return mentor
        
    except Exception as e:
        logger.error(f"❌ Failed to initialize DrivingMentor: {e}")
        raise


def test_mock_assessment():
    """Test assessment with mock data (without external agents)."""
    logger.info("=== TESTING MOCK ASSESSMENT ===")
    
    # This would normally come from the full workflow
    mock_assessment = {
        'safety_score': 4,
        'risk_level': 'high',
        'overall_evaluation': 'The driving behavior shows multiple safety violations including speeding and failure to stop properly at intersections.',
        'strengths': ['Maintained lane position', 'Used turn signals appropriately'],
        'weaknesses': ['Exceeded speed limit', 'Failed to come to complete stop', 'Did not yield to pedestrian'],
        'improvement_advice': ['Always come to a complete stop at stop signs', 'Observe posted speed limits', 'Check for pedestrians before proceeding through intersections']
    }
    
    logger.info(f"Mock Safety Score: {mock_assessment['safety_score']}/10")
    logger.info(f"Risk Level: {mock_assessment['risk_level']}")
    logger.info(f"Evaluation: {mock_assessment['overall_evaluation']}")
    
    logger.info("\\nStrengths:")
    for strength in mock_assessment['strengths']:
        logger.info(f"  + {strength}")
    
    logger.info("\\nWeaknesses:")
    for weakness in mock_assessment['weaknesses']:
        logger.info(f"  - {weakness}")
    
    logger.info("\\nAdvice:")
    for advice in mock_assessment['improvement_advice']:
        logger.info(f"  → {advice}")
    
    return mock_assessment


if __name__ == "__main__":
    print("Running simple driving mentor tests...\\n")
    
    # Test scene extraction
    test_scene_extraction_only()
    
    print("\\n" + "="*50 + "\\n")
    
    # Test mentor initialization
    test_driving_mentor_initialization()
    
    print("\\n" + "="*50 + "\\n")
    
    # Test mock assessment
    test_mock_assessment()
    
    print("\\nSimple tests completed!")
    print("\\nNote: Full workflow tests with database connections")
    print("should be run separately when databases are available.")