"""Tests for scene extraction agent."""

import sys
from pathlib import Path

root = Path(__file__).parent.parent
sys.path.append(str(root))

from src.llm.agent.scene_extraction import SceneExtractor
from src.utils.log import logger


def test_scene_extraction():
    """Test the scene extraction agent with a complex annotation."""
    # Initialize the scene extractor
    extractor = SceneExtractor()
    
    # Complex annotation example (similar to what VideoAnnotator would produce)
    complex_annotation = """
    The ego vehicle is traveling in the right lane of a busy four-lane highway during daytime with clear weather conditions. 
    The vehicle is maintaining a following distance of approximately 2 seconds behind a blue sedan. 
    As the video progresses, a red pickup truck in the left lane begins to merge into the ego vehicle's lane without signaling, 
    forcing the ego vehicle to brake suddenly and change lanes to avoid a collision. 
    The ego vehicle then speeds up to 75 mph in a 65 mph zone while merging back into the right lane. 
    Traffic signals are green throughout the sequence, but construction signs indicate a lane closure ahead. 
    The ego vehicle fails to reduce speed in the construction zone and continues at highway speeds.
    """
    complex_annotation = "The ego vehicle was traveling in the left-most through lane on a multi-lane divided highway, approaching an intersection where two left-turn-only lanes appeared to its left. The ego vehicle maintained its lane, which was clear ahead, while vehicles in the adjacent left (left-turn) and right (through) lanes were slowing or braking. Suddenly, a silver sedan from the right adjacent lane cut sharply into the ego vehicle’s lane directly in front of it. To avoid a collision, the ego vehicle steered slightly left on its lane, but still made contact with the silver sedan on its right side. The ego vehicle came to rest in its original lane before the intersection. The silver sedan subsequently moved over to the right-most lane without stopping."
    
    # Extract simple scenes
    scenes = extractor.extract(complex_annotation)
    
    logger.info(f"Extracted {len(scenes)} scenes:")
    for i, scene in enumerate(scenes, 1):
        logger.info(f"  {i}. {scene}")
    
    # Test formatted output for downstream analysis
    analysis_data = extractor.extract_for_analysis(complex_annotation)
    logger.info(f"Formatted for analysis: {len(analysis_data['rule_queries'])} rule queries, {len(analysis_data['accident_scenes'])} accident scenes")
    
    return scenes


def test_scene_extraction_with_video_annotation():
    """Test integration with VideoAnnotator output format."""
    extractor = SceneExtractor()
    
    # Example of complex annotation from VideoAnnotator
    video_annotation = """
    ### Driving Behavior Analysis – Dashcam Sequence

    #### **Traffic Signals, Signs, and Lane Markings**
    - **Lane Markings:** The ego-vehicle begins in the leftmost lane, adjacent to a solid yellow line (median divider), then transitions to a lane with a left-turn arrow marking. The lane markings are clear: solid yellow for the median, dashed and solid white lines for lane separation, and a left-turn arrow indicating a dedicated turn lane.
    - **Road Signs:** A blue highway/exit sign is visible on the right side of the road, indicating an upcoming exit or highway information.
    - **Traffic Signals:** Overhead traffic lights are visible in the distance, currently green, indicating through traffic is permitted.

    #### **Ego-Vehicle Position and Actions**
    - **Lane Position:** The ego-vehicle starts in the leftmost through lane, then transitions into a dedicated left-turn lane as indicated by the arrow marking.
    - **Speed and Spacing:** The ego-vehicle maintains a moderate speed, matching the flow of traffic, and keeps a safe following distance from the black SUV directly ahead.
    - **Lane Change:** The ego-vehicle executes a smooth lane change from the through lane to the left-turn lane, following the curvature of the lane markings and the lead vehicle.

    #### **Traffic Participants and Interactions**
    - **Vehicles:**
    - **Ahead:** A black SUV is directly in front of the ego-vehicle, also transitioning into the left-turn lane.
    - **Adjacent Lanes:** Multiple vehicles (SUVs, sedans, pickup trucks) are present in the adjacent through lanes, moving in the same direction. A silver sedan in the right lane passes the ego-vehicle on the right, maintaining its lane.
    - **Opposite Direction:** No vehicles are visible in the opposite direction due to the median.
    - **Attributes:** Vehicles are of various sizes and types, including SUVs, sedans, and a pickup truck. No vulnerable road users (pedestrians, cyclists, motorcyclists) are visible.
    - **Intentions:** The black SUV ahead and the ego-vehicle both intend to turn left, as indicated by their lane choice and the left-turn arrow. Vehicles in the adjacent lanes continue straight.

    #### **Sequence of Events and Interactions**
    1. **Initial State:** Ego-vehicle follows the black SUV in the leftmost lane, approaching an intersection.
    2. **Lane Transition:** Both the ego-vehicle and the black SUV ahead move into the left-turn lane, following the lane markings.
    3. **Adjacent Traffic:** A silver sedan in the right lane overtakes the ego-vehicle, maintaining its lane and speed.
    4. **Approaching Intersection:** The ego-vehicle and the black SUV continue toward the intersection, preparing for a left turn as indicated by the lane markings and traffic signals.

    #### **Unsafe Behaviors and Situations**
    - **Close Proximity:** The lane change and merging into the left-turn lane are executed smoothly, with no abrupt maneuvers or close calls.
    - **Traffic Violations:** No visible violations such as running red lights, illegal lane changes, or failure to yield.
    - **Aggressive or Distracted Driving:** No evidence of aggressive or distracted driving behaviors.

    #### **Road Features and Environment**
    - **Road Type:** Multi-lane divided arterial road with a dedicated left-turn lane at an intersection.
    - **Road Condition:** Dry pavement, clear lane markings, and no visible debris or obstructions.
    - **Environment:** Overcast weather, daytime, with good visibility. No adverse weather conditions affecting driving.

    ---

    ### **Summary Table**

    | Element                  | Description                                                                 |
    |--------------------------|-----------------------------------------------------------------------------|
    | **Traffic Signals**      | Overhead green lights at intersection, left-turn lane arrow                 |
    | **Road Signs**           | Blue highway/exit sign on right                                             |
    | **Lane Markings**        | Solid yellow (median), dashed/solid white (lanes), left-turn arrow          |
    | **Ego-Vehicle Position** | Leftmost lane → left-turn lane                                              |
    | **Traffic Participants** | Black SUV ahead, silver sedan right, various vehicles in adjacent lanes      |
    | **Interactions**         | Smooth lane change, safe following distance, no conflicts                   |
    | **Unsafe Behaviors**     | None observed                                                               |
    | **Road Features**        | Multi-lane arterial, intersection, dry, clear, overcast, daytime            |

    ---

    **Key Takeaway:**  
    The ego-vehicle demonstrates safe and compliant driving behavior, smoothly transitioning into a left-turn lane at an intersection, maintaining appropriate following distance, and responding correctly to lane markings and traffic signals. No unsafe or unusual behaviors are observed in this sequence.
    """
    
    scenes = extractor.extract(video_annotation)
    
    logger.info("Video annotation extraction:")
    for i, scene in enumerate(scenes, 1):
        logger.info(f"  {i}. {scene}")
    
    return scenes


if __name__ == "__main__":
    print("Running scene extraction tests...\n")
    
    # Test basic scene extraction
    test_scene_extraction()
    print("\n" + "="*50 + "\n")
    
    # Test with video annotation format
    # test_scene_extraction_with_video_annotation()
    
    print("\nAll tests completed!")