'''
@package utils.py

A collection of useful functions
'''

import numpy as np
from scipy.spatial.distance import cdist

# -------------------------
# Vectorized kernels
# -------------------------
def make_se(gamma):
    # Squared exponential
    def _ker(x, y, g=gamma):
        _D = cdist(x, y, metric='sqeuclidean')
        return np.exp(-_D/g)
    return _ker

def make_exp(gamma):
    # Exponential
    def _ker(x, y, g=gamma):
        _D = cdist(x, y, metric='euclidean')
        return np.exp(-_D/g)
    return _ker

def make_m32(gamma):
    def _ker(x, y, g=gamma):
        _K = cdist(x, y, metric='euclidean') / g * np.sqrt(3)
        return (1.0 + _K) * np.exp(-_K)
    return _ker

def make_m52(gamma):
    def _ker(x, y, g=gamma):
        _K = cdist(x, y, metric='euclidean') / g * np.sqrt(5)
        return (1.0 + _K + _K*_K/3.0) * np.exp(-_K)
    return _ker

# -------------------------
# Analytical tangent vectors
# -------------------------
def tangent_1circle(x):
    _x = np.atleast_2d(x)
    _t = np.arctan2(_x[:,1], _x[:,0])
    _T = np.vstack([np.sin(_t), -np.cos(_t)]).T
    return _T

def tangent_2torus(x, R):
    _X = np.atleast_2d(x)
    _x, _y, _z = _X[:,0], _X[:,1], _X[:,2]
    _p = np.arctan2(_y, _x)
    _c, _s = np.cos(_p), np.sin(_p)
    _r = np.sqrt(_x**2 + _y**2)
    _T1 = np.vstack([
        -_s,
        _c,
        np.zeros_like(_s)]).T
    _T2 = np.vstack([
        -_z*_c,
        -_z*_s,
        _r-R]).T
    _T2 /= np.linalg.norm(_T2, axis=1)
    _T = np.swapaxes(np.array([_T1, _T2]), 0, 1)
    return _T.squeeze()

# -------------------------
# Error metric
# -------------------------
def RMSE(truth, pred):
    err = np.sqrt(np.mean((truth-pred)**2, axis=1))
    emx = np.max(err)
    return err, emx
