import torch
import torch.nn as nn


class BasicBlock(nn.Module):
    expansion = 1

    @staticmethod
    def conv3x3(in_planes, out_planes, stride=1):
        return nn.Conv2d(
            in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False
        )

    def __init__(self, in_planes, planes, stride=1):
        super().__init__()
        self.conv1 = self.conv3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = self.conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = self.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes, nf, hid=8):
        super().__init__()
        self.in_planes = nf

        self.conv1 = block.conv3x3(3, nf * 1)
        self.bn1 = nn.BatchNorm2d(nf * 1)
        self.relu = nn.ReLU()
        self.layer1 = self._make_layer(block, nf * 1, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, nf * 2, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, nf * 4, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, nf * 8, num_blocks[3], stride=2)
        self.avg_pool = nn.AvgPool2d(4)
        self.classifier = nn.Linear(
            nf * hid * block.expansion,
            num_classes,
        )

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.avg_pool(out)
        out = torch.flatten(out, 1)  # flatten tensor starting from dimension 1
        out = self.classifier(out)
        return out


def reduced_ResNet18(out_dim):
    """ResNet18 model"""
    return ResNet(block=BasicBlock, num_blocks=[2, 2, 2, 2], num_classes=out_dim, nf=20)


def ResNet18(out_dim):
    """ResNet18 model"""
    return ResNet(block=BasicBlock, num_blocks=[2, 2, 2, 2], num_classes=out_dim, nf=64)


def reduce_ResNet18_mini(out_dim, nf=20):
    return ResNet(BasicBlock, [2, 2, 2, 2], out_dim, nf, hid=32)


def ResNet18_mini(out_dim, nf=64):
    return ResNet(BasicBlock, [2, 2, 2, 2], out_dim, nf, hid=32)
