import torch
import json
import os
import numpy as np

def SVD_expected_value(input, var = 0.95, p = 7, regularization_lambda=1e-6):
    # U, S, V = torch.svd(input)
    identity_matrix = torch.eye(input.size(0), device=input.device) * regularization_lambda
    input_regularized = input + identity_matrix
    
    try:
        # Perform SVD
        U, S, V = torch.svd(input)
    except RuntimeError as e:
        print(f"SVD failed: {e}, falling back to PCA...")
        # Fallback to PCA if SVD fails
        U, S, V = torch.pca_lowrank(input_regularized, q=input.size(1))
    total_energy = torch.sum(S**2)
    cumulative_energy = torch.cumsum(S**2, dim=0)
    energy_threshold = var * total_energy
    num_singular_values = torch.sum(cumulative_energy <= energy_threshold) #Energy threshold
    num_singular_values = num_singular_values + p #Oversampling
    if num_singular_values >= U.shape[1]:
        num_singular_values = U.shape[1]
        
    if num_singular_values+1 > 0 and num_singular_values+1 < U.shape[1]:
        retained_energy = cumulative_energy[num_singular_values+1] 
        U_truncated = U[:, :num_singular_values+1]
    elif num_singular_values+1 >= U.shape[1]:
        retained_energy = total_energy
        U_truncated = U[:, :num_singular_values]
    else:
        retained_energy = 0 
    retained_energy_per = retained_energy/total_energy
    return U_truncated/torch.sqrt(retained_energy_per)

def log_svd_shapes(iteration, U_dict, output_file): #Save the shape of compression tensors at each iteration
    if os.path.exists(output_file):
        if os.stat(output_file).st_size == 0:
            all_results = {}
        else:
            with open(output_file, 'r') as f:
                all_results = json.load(f)
    else:
        all_results = {}

    if str(iteration) not in all_results:
        all_results[str(iteration)] = {}

    for layer_name, matrix_data in U_dict.items():
        matrix_shape = matrix_data[0].shape

        all_results[str(iteration)][layer_name] = {
            "shape": matrix_shape
        }

    with open(output_file, 'w') as f:
        json.dump(all_results, f, indent=4)