#!/usr/bin/env bash

. utils/parse_options.sh

if [ $# -gt 1 ]; then
    echo "Usage: $0 --mindepth 0 --maxdepth 1 [expdir]" 1>&2
    echo ""
    echo "Show the system environments and the evaluation results in Markdown format."
    exit 1
fi

[ -f ./path.sh ] && . ./path.sh
set -euo pipefail
if [ $# -eq 1 ]; then
    expdir=$1
else
    expdir=exp
fi

cat << EOF
<!-- Generated by $0 -->
# RESULTS
## Environments
- date: \`$(LC_ALL=C date)\`
EOF

echo

python3 << EOF
import sys, espnet, torch
pyversion = sys.version.replace('\n', ' ')

print(f"""- python version: \`{pyversion}\`
- espnet version: \`espnet {espnet.__version__}\`
- pytorch version: \`pytorch {torch.__version__}\`""")
EOF

cat << EOF
- Git hash: \`$(git rev-parse HEAD)\`
  - Commit date: \`$(git log -1 --format='%cd')\`
EOF

find "$expdir" -type f -name "avg_result.txt" | while IFS= read -r filepath; do
    wer_snt=0
    wer_corr=0
    wer_substitute=0
    wer_del=0
    wer_ins=0

    cer_snt=0
    cer_corr=0
    cer_substitute=0
    cer_del=0
    cer_ins=0

    while IFS= read -r line; do
        heading="$(echo "$line" | cut -d ":" -f 1)"
        value="$(echo "$line" | cut -d ":" -f 2 | tr -d ' ')"

        case "$heading" in
            "espnet_wer_delete") wer_del="$value" ;;
            "espnet_wer_insert") wer_ins="$value" ;;
            "espnet_wer_replace") wer_substitute="$value" ;;
            "espnet_wer_equal") wer_corr="$value" ;;
            "espnet_cer_delete") cer_del="$value" ;;
            "espnet_cer_insert") cer_ins="$value" ;;
            "espnet_cer_replace") cer_substitute="$value" ;;
            "espnet_cer_equal") cer_corr="$value" ;;
            "sentences") wer_snt="$value"; cer_snt="$value" ;;
        esac
    done < "$filepath"

    wer_wrd=$(awk -v corr="$wer_corr" -v repl="$wer_substitute" -v del="$wer_del" 'BEGIN {print corr + repl + del}')
    wer_err=$(awk -v repl="$wer_substitute" -v del="$wer_del" -v ins="$wer_ins" -v wrd="$wer_wrd" 'BEGIN {printf "%.3f", (repl + del + ins) / wrd}')

    cer_wrd=$(awk -v corr="$cer_corr" -v repl="$cer_substitute" -v del="$cer_del" 'BEGIN {print corr + repl + del}')
    cer_err=$(awk -v repl="$cer_substitute" -v del="$cer_del" -v ins="$cer_ins" -v wrd="$cer_wrd" 'BEGIN {printf "%.3f", (repl + del + ins) / wrd}')

    echo "## WER"
    echo "|Snt|Wrd|Corr|Sub|Del|Ins|Err|"
    echo "|---|---|---|---|---|---|---|"
    echo "|$wer_snt|$wer_wrd|$wer_corr|$wer_substitute|$wer_del|$wer_ins|$wer_err|"
    echo

    echo "## CER"
    echo "|Snt|Wrd|Corr|Sub|Del|Ins|Err|"
    echo "|---|---|---|---|---|---|---|"
    echo "|$cer_snt|$cer_wrd|$cer_corr|$cer_substitute|$cer_del|$cer_ins|$cer_err|"
    echo

    while IFS= read -r line; do
        heading="$(echo "$line" | cut -d ":" -f 1)"
        case "$heading" in
            "espnet_wer_delete"|"espnet_wer_insert"|"espnet_wer_replace"|"espnet_wer_equal"|"espnet_cer_delete"|"espnet_cer_insert"|"espnet_cer_replace"|"espnet_cer_equal"|"sentences")
                continue
                ;;
            *)
                echo "## $heading"
                echo "$line" | sed 's/\([0-9]\+\.[0-9]\{3\}\)[0-9]*/\1/'
                echo
                ;;
        esac
    done < "$filepath"
done
