# HyPlan: Non-Deterministic Planning for Hyperproperty Verification

This artifact contains **HyPlan**, a tool that soundly reduces model-checking of \forall\exists HyperLTL properties to fully-observable non-deterministic (FOND) planning instances.

## Overview

HyPlan reads a NuSMV system and a HyperLTL property and produces a PDDL planning instance that (soundly) approximates the model-checking problem:
If the planning problem admits a plan, the HyperLTL property holds on the given system. 
Compared to previous encodings using parity-games [2], PDDL allows us to represent the problem in a factored manner, leading to significant performance improvements.  

## Experiments 

This README contains instructions on how to build HyPlan and how to specify the input format.
**After you have built HyPlan**, you can have a look at the `experiments/README.md` file for instructions on how to reproduce the experiments from our paper [1].

## Structure 

- `full_version.pdf` is the full version of our paper including 
- `src/` contains the source code of HyPlan (written in F#).
- `experiments/` contains the systems used for evaluation and scripts to run them automatically.
- `test/` contains a simple test instance to check that everything was set-up correctly.
- `app/` is the target folder for the build. The final HyPlan executable will be placed here. 
- `/translator-fond` contains a Python script to translate non-deterministic (FOND) PDDL instances into SAS instances. We use this script for our experiments. 

## Build HyPlan

### Dependencies

To build and run HyPlan you require the following

- [_.NET 8 SDK_](https://dotnet.microsoft.com/en-us/download) (tested with version 8.0.100)
- [_spot_](https://spot.lrde.epita.fr/) (tested with version 2.11.6)


**Install Dependencies:**
Install the .NET 8 SDK (see [here](https://dotnet.microsoft.com/en-us/download) for details) and make sure it is installed correctly by running `dotnet --version`.
Download and build _spot_ (details can be found [here](https://spot.lrde.epita.fr/)). 
You can place the _spot_ executables in any location of your choosing. 
To use HyPlan you need to provide it with the **absolute** path to spot (see details below).

**Planner:**
HyPlan will generate a planning instance in the PDDL format (including non-deterministic **oneof** statements when necessary).
To solve this planning instance, you need an appropriate planner. In our experiments, we used [_Scorpion_](https://github.com/jendrikseipp/scorpion) to solve classical planning instances and [_MyND_](https://github.com/JackyCSer/MyNDPlanner) to solve FOND instances. 
However, you can, in theory, use any planner that supports the (non-deterministic) PDDL format.

### Build HyPlan

To build HyPlan run the following (when in the main directory of this artifact).

```shell
cd src/HyPlan
dotnet build -c "release" -o ../../app
cd ../..
```

Afterward, the HyPlan executable is located in the `app/` folder.
The HyPlan executable is not standalone and requires the other files located in the `app/` folder (as is usual for .NET applications).
Moreover, it will only run on systems that have the .NET Core Runtime installed. 
Test that everything works as expected by running 

```shell
app/HyPlan --version
```

### Connect spot to HyPlan

HyPlan requires the _ltl2tgba_ tool from the [_spot_](https://spot.lrde.epita.fr/) library.
HyPlan is designed such that it only needs the **absolute** path to _ltl2tgba_, so _spot_ can be installed and placed at whatever location fits best.
The absolute paths are specified in a `paths.json` configuration file. 
This file must be located in the same directory as the HyPlan executable (this convention makes it easy to find the config file, independent of the relative path HyPlan is called from). 
We already provide a template file `app/paths.json` that **needs to be modified**. 
After having built spot, paste the absolute path to the _ltl2tgba_ executables to the `paths.json` file. 
For example, if `/usr/bin/ltl2tgba` is the _ltl2tgba_ executable, the content of `app/paths.json` should be

```json
{
    "ltl2tgba":"/usr/bin/ltl2tgba"
}
```

### A First Example

To test that the paths have been installed correctly, we can verify our first instance by running

```shell
app/HyPlan --exp --reach ./tests/system.txt ./tests/formula.txt
```

This should generate a `dom.pddl` and `prob.pddl` file in your current directory. 
You can then use any (in this instance _classical_) planner to search for a plan for this PDDL instance. 

## Run HyPlan

After having built HyPlan and all its dependencies, you are ready to use HyPlan.
You can run HyPlan by running

```shell
app/HyPlan <options> <systemFile> <formulaFile>
```

where `<options>` are the command line options, `<systemFile>` is the (path to) the system, and `<formulaFile>` is the (path to) the HyperLTL formula.

### Input Format and Output Format
HyPlan supports several command-line options that togle which input HyPlan expects and which output it produces.

#### Input Format
Depending on which type of system you want to check (either explicit-state or NuSMV), use one of the following command-line options:

- `--exp` applies HyPlan to an explicit-state system. 
- `--nusmv` applies HyPlan to a symbolic NuSMV model.

If nothing is selected, HyPlan uses the `--nusmv` option by default.

For details on how the system and property are specified in either of the two input modes, see [HyPlan's Input Format](#hyplans-input-format).


#### Output Format

The main objective of HyPlan is to translate the HyperLTL verification problem into a planning problem.
To compare with [2], HyPlan also features the ability to compile to a parity game. 

- `--planning` compiles the verification problem to a PDDL planning instance.
- `--pg` compiles the verification problem to a parity game [2].

By default, HyPlan uses the `--planning` option.

When using the `--planning` option, HyPlan will create two files `./dom.pddl` and `./prob.pddl` that contain the PDDL planning problem. 
If needed, this planning instance is non-deterministic, i.e., uses `oneof` expressions and has the `:non-deterministic` requirement. 

When using the `--pg` option, produces a parity game `./game.pg` in the PGSolver format [3].

#### Additional Options

Additional (and optional) command-line options include

- `--debug` instructs HyPlan to print additional information and more detailed error messages.
- `--version` displays the HyPlan version.


## HyPlan's Input Format 

### Specifying HyperLTL Properties

The specification checked by HyPlan is the temporal logic HyperLTL.
A HyperLTL formula consists of an LTL-like body, preceded by a quantifier prefix. 
Formulas have the form `<qfPrefix> <ltlBody>`.

Here `<ltlBody>` can be one of the following:
- `1`: specifies the boolean constant true
- `0`: specifies the boolean constant false
- An atomic proposition atom. Depending on which type of system (explicit-state, NuSMV model,...) we consider, atomic propositions have different structures. We give details in the following sections.
- `(<ltlBody>)`: Parenthesis
- `<ltlBody> & <ltlBody>`: Conjunction
- `<ltlBody> | <ltlBody>`: Disjunction
- `<ltlBody> -> <ltlBody>`: Implication
- `<ltlBody> <-> <ltlBody>`: Equivalence
- `<ltlBody> U <ltlBody>`: Until
- `<ltlBody> W <ltlBody>`: Weak Until
- `<ltlBody> R <ltlBody>`: Release
- `F <ltlBody>`: Eventually
- `G <ltlBody>`: Globally
- `X <ltlBody>`: Next
- `! <ltlBody>`: Boolean Negation

The operators follow the usual operator precedences. To avoid ambiguity, we recommend always using parenthesis. 

The quantifier prefix `<qfPrefix>` can be one of the following:
- The empty string
- Universal quantification `forall <VAR>. <qfPrefix>`
- Existential quantification `exists <VAR>. <qfPrefix>`

Here `<VAR>` is a trace variable, which is any non-empty sequence consisting only of letters and digits (starting with a letter). 

For examples of HyperLTL properties (with the atomic propositions used in either case), see the following sections. 

### Specifying Explicit-state Transition Systems

When using option `--exp`, HyPlan expects an _explicit-state transition system_.
An explicit-state system has the form 

```
AP: "<AP>" ... "<AP>"
Init: <stateID> ... <stateID> 
--BODY-- 
<stateDefinition>
...
<stateDefinition>
--END--
```

Here, `<AP>` is an atomic proposition that can be any **escaped** string. 
`<stateID>` is a natural number specifying a state. 
The header specifies which states are initial (there must be at least one initial state) and which APs are used in the system.

A `<stateDefinition>` has the form 
```
State: <stateID> <apEval>
<stateID> ... <stateID>
```

It specifies which state we are defining and the evaluation of the atomic propositions in that state. 
The `<apEval>` has the form `{<apIndex> ... <apIndex>}` where `<apIndex>` is a natural number that identifies one of the provided APs. 
The second line lists all successors of that state.
Every state must have at least one successor state.

Consider the following example:

```
AP: "x" "y"
Init: 0 1
--BODY-- 
State: 0 {}
0 2 3
State: 1 {1}
0 1 2
State: 2 {0}
0 2 3
State: 3 {0 1}
2 3
--END--
```

This specification declares states `0` and  `1` as initial states and `"x"` and `"y"` as APs.
For each state, we give the evaluation of the atomic propositions by listing the indices of all APs which hold in the given state.
For example, in state `1`, AP `"x"` (index 0) does not hold but `"y"` (index 1) does.
Each state lists all successors of that node. 
For example, the successor states of state `0` are states `0`, `2`, and `3`.


HyperLTL properties on explicit-state systems are specified by using atomic propositions of the form `"<AP>"_<Var>` where `<AP>` is the AP as given in the system, and `<VAR>` is a trace variable from the quantifier prefix. 
This atomic proposition holds if, in the trace bound to `<VAR>`, the AP `<AP>` holds. 

An example property on the above example system would be:

```
forall A. exists B. X (G ("x"_A <-> "y"_B))
```

### Specifying Symbolic Transition Systems

When using option `--nusmv` (the default option), HyPlan expects a _NuSMV system_ with finite variable domains (so the system denotes a finite-state transition system).
HyPlan supports only a fragment of the NuSMV specification language.
In particular, we assume that the system consists of a _single module_. 

A single-module NuSMV model (as supported by HyPlan) has the following structure:

```
MODULE <name>
<variableDeclarationBlock>
<bodyBlock>
```

Here `<name>` is an arbitrary name (consisting only of letters) of that module.

#### Variable Declaration Block

The `<variableDeclarationBlock>` declares all variables and assigns a type. 
The variable declaration block has the form 

```
VAR 
<varName> : <type>;
<varName> : <type>;
...
<varName> : <type>;
```

containing a sequence of type assignments. 
Here `<varName>` is a variable name.
A valid variable name is any sequence of characters that starts with a letter or `_`, followed by an arbitrary sequence of letters, digits, or the special symbols `_`, `$`, `#`, `-`, `[`, `]`, `.` and is no reserved keyword.

`<type>` gives the type of that variable. 
Support types are:
- `boolean`: The boolean type with allowed values `TRUE` and `FALSE`.
- `{n_1, ..., n_M}` where `n_1`, ..., `n_m` are integers: This is a set type that can hold any natural number contained in the set.
- `l..h` where `l <= h` are integer: This range type can hold any integer between `l` and `h`. Internally, we treat it as a shorthand for `{l, l+1, ..., h-1, h}`.
- `array l..h of <type>` where `l <= h` are integer and `<type>` is an arbitrary (recursively defined) type: This array type models an array with valid indices for all integers between `l` and `h`. Internally, we eliminate arrays and instead introduce a separate variable for each position. For example, the type declaration `x : array 0..2 of boolean;` will be reduced to the type declarations `x[0] : boolean; x[1] : boolean; x[2] : boolean;`.


#### Assignments and Definitions

The `<bodyBlock>` defines the actual behavior of the system. In this block, we can pose initial conditions on all variables, describe how the variables should be updated in each step, and introduce additional (defined) shorthands. 
The `<bodyBlock>` is a sequence of assignment blocks and definition blocks, i.e., it has the form

```
<assignmentBlock>
...
<assignmentBlock>
<definitionBlock>
...
<definitionBlock>
<assignmentBlock>
...
<assignmentBlock>
...
```

where `<assignmentBlock>` and `<definitionBlock>` are defined in the following. 

##### Assignment Block

An assignment block (`<assignmentBlock>`) has the form 

```
ASSIGN 
<initOrNext>
<initOrNext>
...
<initOrNext>
```

where `<initOrNext>` poses an _initial_ or _next_ condition.
An `<initOrNext>` either has the form
```
init(<varName>) := <expression>;
```
or 
```
next(<varName>) := <expression>;
```
where `<varName>` is the name of a variable (the variable must be defined in the type declaration section) and `<expression>` is a NuSMV expression (formally defined below).
In the former case, we define all possible initial values for `<varName>`. 
The expression evaluates to a set of values (in the special case, where the expression is deterministic, it evaluates to a singleton set giving the precise value) and all such values are initial states of the system. 
The expression can refer to other variables, i.e., we allow initial conditions such as `init(x) := y;`, but there can be no _cyclic dependency_ when evaluating the initial expressions. 

In the latter case, a next expression defines all successor values for the variable.
In each step, we evaluate the expression in the current state and assign the resulting value to `<varName>` in the next state.
As before, the expression evaluates to a set of values (in the deterministic case a singleton set) and we consider all possible values in the next step.

In either of the two cases, the expression can refer to program variables (those defined in the type declaration section) and also _defined_ variables (defined below).
There must not exist cyclic dependencies between the defined variables when evaluating the initial or next expression. 


##### Definition Block

A definition block (`<definitionBlock>`) has the form 

```
DEFINE
<definition>
<definition>
...
<definition>
```

where `<definition>` has the form
```
<varName> := <expression>;
```
Here `<varName>` is a variable name that is _not_ listed in the `<variableDeclarationBlock>`.
Intuitively, `<varName> := <expression>;` defines the variable `<varName>` as a shorthand for the expression `<expression>`.
As in the assignment block, we forbid cylic dependecies during the evaluation of definitions.

#### Expressions

An expression is evaluated in a state `s`, i.e., a concrete mapping that maps each declared variable (defined in `<variableDeclarationBlock>`) to a value.
When evaluating an expression in state `s`, we compute a *set* of values (possibly a singleton set in case the expression is determinstic).
An `<expression>` can have the following forms:

- `TRUE`: the boolean true constant; evaluates to singelton set `{TRUE}`
- `FALSE`: the boolean false constant; evaluates to singelton set `{FALSE}`
- `<n>` where `<n>` is any integer: an integer constant; evaluates to singelton set `{<n>}`
- `<varName>`: a variable that is either declared in the `VAR` block or defined in a `DEFINE` block. In case `<varName>` is declared, it evaluates to `{s(<varName>)}`, i.e., the singleton set containing value of `<varName>` in the current state. In case `<varName>` is defined in a DEFINE block, we (recursively) evaluate the defined expression in state `s`.
- `toInt(<expression>)`: converts a boolean value to an integer; we first (recursively) evaluate the subexpression and map all `TRUE`s to `0`s and all `FALSE`s to `1`s.
- `toBool(<expression>)`: converts a integer value to a boolean; we first (recursively) evaluate the subexpression and map all `0`s to `FALSE`s and all other values to `TRUE`s.
- `case <expression>:<expression>; ... <expression>:<expression>; esac`: a case expression; During the evaluation, we search for the first `<expression>:<expression>` pair where the first expression (the guard) evaluates to true and then compute the value of the second expression in that pair.
- `{ <expression>, ..., <expression>}`: a set expression; We (recursively) evaluate all subexpressions and take the union of all sets
- `<expression> <opp> <expression>`: Binary operation. Here `<opp>` can be: `&` (and), `|` (or), `->` (implies), `=` (equals), `!=` (not equals), `<=` (less or equal), `>=` (greater or equal), `<` (less), `>` (greater), `+` (addition), `-` (subtraction). We (recursively) evaluate both operants and apply each binary operation to all possible value combinations. 
- `<opp> <expression>`: Unary operation. Here `<opp>` can be: `!` (boolean negation). We (recursively) evaluate the operant and apply each unary operation to all possible values. 


#### Example 

Consider the following example:

```
MODULE main
VAR
    mutation: boolean;
    action: 0..2;
    beverage: 0..2;
    water: 0..3;


ASSIGN
    init(mutation) := {TRUE, FALSE};
    next(mutation) := {TRUE, FALSE};

    init(action) := 0;
    next(action) := {0,1,2};

    init(beverage) := 0;
    next(beverage) :=
        case
            (action=1 & !(water=0)): 1;
            TRUE: beverage;
        esac;

    init(water) := 2;
    next(water) :=
        case
            (action=1 & !(water=0)): water - 1; -- make beverage
            (mutation & water=0): 1;
            (mutation & water=1): 2;
            (mutation & water=2): 3;
            (mutation & water=3): 3;
            (!mutation & water=0): 2;
            (!mutation & water=1): 3;
            (!mutation & water=2): 3;
            (!mutation & water=3): 3;
            TRUE: water;
        esac;

DEFINE
    NO_water := (action=1) & (water=0);
    NO_output := (action!=1)  | ((action=1)&(beverage=0));
```

#### HyperLTL in NuSMV Systems 

HyperLTL properties on symbolic NuSMV systems are specified by using atomic propositions of two forms.
Either they have the form
```
{<expression>}_<VAR>
```
where `<expression>` is an expression of type boolean and `<VAR>` is a trace variable.
This atomic proposition holds whenever `<expression>` evaluated in the current step of the trace bound to `<VAR>` evaluates to `{TRUE}`. 


Or of the form 
```
{<expression1>}_<VAR1> = {<expression2>}_<VAR2>
```
where `<expression1>` and `<expression2>` are expressions of the same type, and `<VAR1>` and `<VAR2>` are trace variables.
This atomic proposition states that the result of evaluating `<expression1>` on trace `<VAR1>` is the same as evaluating expression `<expression2>` on trace `<VAR2>`.

An example property on the above example system would be:

```
forall A. exists B. 
(
    ({action}_A = {action}_B)  U  ({NO_water}_B)
)
```

Here `NO_water` is a boolean variable in the system (in this case a defined one) and thus also a boolean expression.
`action` is a non-boolean expression that is compared across traces `A` and `B`.


## References  

[1] Non-Deterministic Planning for Hyperproperty Verification. Anonymous author(s). Under Review at ICAPS 2024.

[2] Prophecy Variables for Hyperproperty Verification. Raven Beutner and Bernd Finkbeiner. CSF 2021

[3] Solving Parity Games in Practice. Oliver Friedmanna and Martin Lange. ATVA 2009
