module AutomataUtil 

open System.Collections.Generic

open FsOmegaLib.DPA
open FsOmegaLib.SAT

let shortestPathsBetweenAllPairs (nodes : seq<'T>) (forwardEdges: 'T -> seq<'E * 'T>) (includeZeroLengthPaths : bool) =
    let next = new Dictionary<_,_>(Seq.length nodes * Seq.length nodes)

    for n in nodes do 
        for (e, n') in forwardEdges n do 
            if Seq.contains n' nodes then 
                // In case there are multiple edges we just take the first (has no impact as the length is the same)
                if next.ContainsKey (n, n') |> not then 
                    next.Add((n, n'), ([e],[n; n']))

        if includeZeroLengthPaths then 
            next.Remove((n, n)) |> ignore
            next.Add((n, n), ([], [n]))

    for k in nodes do 
        for i in nodes do 
            for j in nodes do 
                if next.ContainsKey (i, k) && next.ContainsKey (k, j) then 
                    if next.ContainsKey (i, j) |> not || (next.[i, j] |> fst |> List.length > (next.[i, k] |> fst |> List.length) + (next.[k, j] |> fst |> List.length) + 1) then 
                        next.Remove((i,j)) |> ignore
                        next.Add((i, j), (fst next.[i, k] @ fst next.[k, j], snd next.[i, k] @ (next.[k, j] |> snd |> List.tail) ) )

    next


// Returns all states from which SOME words are accepted and all states from which ALL words are accepted 
let findNonEmptyAndUniversalStates (dpa : DPA<'T, 'L>) = 

    // All states that have some accepting run
    let findNonEmptyStates (dpa : DPA<'T, 'L>) = 
        // All even colors in the DPA
        let evenColors = 
            dpa.States
            |> Set.map (fun x -> dpa.Color.[x])
            |> Set.filter (fun x -> x % 2 = 0)
            |> Set.toList

        // All edges that are SAT
        let satEdges = 
            dpa.Edges
            |> Map.map (fun _ sucs -> 
                sucs 
                |> List.filter (fun (g, _) -> DNF.isSat g)
                )

        let statesWithAcceptingSelfloop = 
            evenColors
            |> List.map (fun col -> 

                let statesOfSmallerColor = 
                    dpa.States
                    |> Set.filter (fun s -> dpa.Color.[s] <= col)

                let reachPairs = 
                    shortestPathsBetweenAllPairs statesOfSmallerColor (fun s -> satEdges.[s]) false

                // All states of col that hav an path to themself using only smaller colors
                let winningStates = 
                    dpa.States
                    |> Set.filter (fun s -> dpa.Color.[s] = col)
                    |> Set.filter (fun s -> reachPairs.ContainsKey (s, s))
                
                winningStates
                )
            |> Set.unionMany
        

        // Compute all states that can reach a state in 
        let reachPairs = 
            shortestPathsBetweenAllPairs dpa.States (fun s -> satEdges.[s]) false

        dpa.States
        |> Set.filter (fun s -> 
            statesWithAcceptingSelfloop
            |> Set.exists (fun s' -> reachPairs.ContainsKey (s, s'))
        )

    // All states for which ALL runs are accepting 
    let findUniversalStates (dpa : DPA<'T, 'L>) = 
        // Increment all colors
        let dpa' = {dpa with Color = dpa.Color |> Map.map (fun _ c -> c + 1)}

        // All states from which there exists some rejecting run
        let nonWinningStates = findNonEmptyStates dpa'

        dpa.States 
        |> Set.filter (fun s -> Set.contains s nonWinningStates |> not)

    findNonEmptyStates dpa, findUniversalStates dpa
